#include <math.h>
#include "utils/compat.h"
#include "or_stat.h"

void zero_time(or_st_time& t) {
    t.seconds	= 0;
    t.micros	= 0;
}

void get_wallclock_time(or_st_time& t) {
    struct timespec ts;
    getclock(TIMEOFDAY, &ts);
    t.seconds	= ts.tv_sec;
    t.micros	= ts.tv_nsec / 1000;
}

void add_time(or_st_time& t, or_st_time const& x) {
    int micros = t.micros + x.micros;
    if (micros >= 1000000) {
	micros -= 1000000;
	t.seconds++;
    }
    t.micros = micros;
    t.seconds += x.seconds;
}

void sub_time(or_st_time& t, or_st_time const& x) {
    int micros = t.micros - x.micros;
    if (micros < 0) {
	micros += 1000000;
	t.seconds--;
    }
    t.micros = micros;
    t.seconds -= x.seconds;
}

or_st_time float_to_time(float t) {
    or_st_time result;
    result.seconds = (Ubits32) floor(t);
    result.micros = (Ubits32) ((float)(t - result.seconds)*1000*1000);
    return result;
}


// convert time structure into a double
double time_to_seconds(or_st_time const& t) {
    return ((double) t.seconds) + ((double) t.micros) / 1e6;
}

// modifies	"d"
// effects	post(d) = pre(d) - x
static void sub_disk_stat(or_st_disk& d, or_st_disk const& x) {
    d.count -= x.count;
    d.blocks -= x.blocks;
    sub_time(d.elapsed, x.elapsed);
}

void sub_stats(OR_stat& s, OR_stat const& x) {
    sub_time(s.clock, x.clock);
    s.trans	-= x.trans;
    s.mods	-= x.mods;
    s.absorbed	-= x.absorbed;
    s.installs	-= x.installs;
    // Leave log size alone

    s.logflush_time -= x.logflush_time;

    sub_time(s.disk_time, x.disk_time);
    sub_time(s.disk_used, x.disk_used);
    sub_time(s.total_fetch_time,    x.total_fetch_time);
    sub_time(s.total_recv_time,     x.total_recv_time);
    sub_time(s.total_validate_time, x.total_validate_time);
    sub_time(s.total_send_time,     x.total_send_time);
    sub_time(s.total_trans_time,    x.total_trans_time);

    sub_disk_stat(s.disk_total,    x.disk_total);
    sub_disk_stat(s.disk_mreads,   x.disk_mreads);
    sub_disk_stat(s.disk_mwrites,  x.disk_mwrites);
    sub_disk_stat(s.disk_freads,   x.disk_freads);
    sub_disk_stat(s.disk_ireads,   x.disk_ireads);
    sub_disk_stat(s.disk_creads,   x.disk_creads);
    sub_disk_stat(s.disk_fwrites,  x.disk_fwrites);
    sub_disk_stat(s.disk_cwrites,  x.disk_cwrites);

    s.fetches		-= x.fetches;
    s.misses		-= x.misses;
    s.frag_writes	-= x.frag_writes;
    s.frag_sizes	-= x.frag_sizes;
    s.clean_count	-= x.clean_count;
    s.clean_live	-= x.clean_live;
}

// report stats on disk operations of a particular type.
// "t" is the total elapsed time
static void report(FILE* f, char const* label, or_st_disk const& s, double t) {
    if (s.count == 0) return;

    double thruput = 0.0;
    double elapsed = time_to_seconds(s.elapsed);
    if (elapsed > 0) {
	double bytes_per_second = (((double) s.blocks) * 512) / elapsed;
	thruput = bytes_per_second * 1e-6;
    }

    double fraction = 0.0;
    if (t > 0) {
	fraction = elapsed / t;
    }

    double avg_size = 0.0;
    avg_size = ((double) s.blocks) / (((double) s.count) * 2);
    char str[30];
    int len = strlen(label);
    sprintf(str, "  %%-%ds : ", len);
    fprintf(f, str, label);
    fprintf(f, "ops=%6d, size=%5.1f KB, speed=%6.3f MB/s, elapsed = "
	    "%9.5f for=%5.1f%%\n",
	    s.count, avg_size, thruput, elapsed, fraction*100.0);
}

void report_stats(FILE* f, OR_stat const& s) {
    double clock = time_to_seconds(s.clock);
    fprintf(f, "  elapsed time       = %8.3f seconds\n", clock);
    fprintf(f, "  transactions       = %8d\n", s.trans);
    fprintf(f, "  modifications      = %8d\n", s.mods);
    fprintf(f, "  absorbed           = %8d\n", s.absorbed);
    fprintf(f, "  installs           = %8d\n", s.installs);
    fprintf(f, "  logsize            = %8d\n", s.logsize / 1024);
    fprintf(f, "  max_logrecord      = %8d\n", s.max_logrecord);

    fprintf(f, "  logflush time      = %8.3f seconds\n", s.logflush_time/(1000.0*1000.0));

    // Server cache hit ratio
    fprintf(f, "  server fetches     = %8d\n", s.fetches);
    if (s.fetches > 0) {
	double ratio = ((double) (s.fetches-s.misses)) / ((double) s.fetches);
	fprintf(f, "  server hit ratio   = %8.1f\n", ratio * 100.0);
    }

    // Average fragment size at server
    if (s.frag_writes > 0) {
	int b = s.frag_sizes / s.frag_writes;
	fprintf(f, "  fragment size      = %8d bytes\n", b);
    }

    // Average live data at cleaning
    if (s.clean_count > 0) {
	int b = s.clean_live / s.clean_count;
	fprintf(f, "  cleaner live data  = %8d bytes\n", b);
    }

    // Disk utilization
    double dt = time_to_seconds(s.disk_time);
    fprintf(f, "  disk elapsed time  = %8.3f seconds\n", dt);
    if (dt > 0) {
	double util = time_to_seconds(s.disk_total.elapsed) / dt;
	fprintf(f, "  disk utilization   = %8.1f%%\n", util * 100.0);
    }

    // Transaction statistics
    double rt = time_to_seconds(s.total_recv_time);
    double vt = time_to_seconds(s.total_validate_time);
    double st = time_to_seconds(s.total_send_time);
    double tt = time_to_seconds(s.total_trans_time);
    fprintf(f, "  Total receive  time  = %8.3f seconds\n", rt);
    fprintf(f, "  Total validate time  = %8.3f seconds\n", vt);
    fprintf(f, "  Total send     time  = %8.3f seconds\n", st);
    fprintf(f, "  Total trans    time  = %8.3f seconds\n", tt);

    double ft = time_to_seconds(s.total_fetch_time);
    fprintf(f, "  Total fetch time     = %8.3f seconds\n", ft);

    // Other disk stats
    report(f, "total",		s.disk_total,	dt);
    report(f, "misc_reads",	s.disk_mreads,	dt);
    report(f, "misc_writes",	s.disk_mwrites,	dt);
    report(f, "fetch_reads",	s.disk_freads,	dt);
    report(f, "ireads",		s.disk_ireads,	dt);
    report(f, "cleaner_reads",	s.disk_creads,	dt);
    report(f, "flusher_writes",	s.disk_fwrites,	dt);
    report(f, "cleaner_writes",	s.disk_cwrites,	dt);
    fprintf(f, "============================================\n");
}
