/*
\section{Orefs}
This file contains the definition of Orefs.
*/

#ifndef _OREF_H
#define _OREF_H

#include "common/oref_basic.h"
#include "common/shortp.h"
#include "utils/bool_vector.h"
#include "utils/hashfuncs.h"
#include "utils/network.h"
#include "config/vdefs/PAGE_SIZE_BITS.h"
#include "config/vdefs/BASE_PAGE_SIZE_BITS.h"

// Onum_bits       = No. of bits in the name space for onums
// Page_num_bits   = No. of bits in oref for Page_num in a segment.
// Page_id_bits    = No. of bits in oref for the complete page_id.
// Segment_id_bits = No. of bits used to represent a Segment id

// The next three constants are "somewhat" independent for deciding
// what is the page size, how many max objects per page and segsize
// (There are constraints between them, of course!)
// NOTE: All other constants are simply derived from these.

// Page size = 1 << Page_size_bits
#define Page_size_bits PAGE_SIZE_BITS 
#define Onum_bits      (6 + PAGE_SIZE_BITS - BASE_PAGE_SIZE_BITS)
// 2^Page_num_bits pages in a segment
#define Page_num_bits   (5 - (PAGE_SIZE_BITS - BASE_PAGE_SIZE_BITS))

#define Segment_id_bits (sizeof(Oref)*byte_bits - Page_num_bits - Onum_bits)
#define Page_id_bits    (Segment_id_bits + Page_num_bits)

#define Max_pages_in_seg  (1 << Page_num_bits)
#define Max_objs_in_page (1 << Onum_bits)
#define Onum_bitmask ((1 << Onum_bits)-1)
#define Max_onum  Onum_bitmask

inline int Oref_cmp(Oref a, Oref b)   { return (int)a - (int)b;}
inline bool Oref_equal(Oref a, Oref b) { return a == b;}
inline bool Oref_lt(Oref a, Oref b)    { return a < b;}
inline bool Oref_gt(Oref a, Oref b)    { return a > b;}
inline bool Oref_le(Oref a, Oref b)    { return a <= b; }
inline bool Oref_ge(Oref a, Oref b)    { return a >= b;}
inline int  Oref_hash(Oref a)          { return hash_int(a);}
inline Uint Oref_page(Oref x)          { return x >> Onum_bits; }
inline Uint Oref_segment(Oref x)  { return x >> (Page_num_bits + Onum_bits); }
inline Uint Oref_index(Oref x)         { return x & Onum_bitmask;}
inline Uint Oref_sindex(Oref x)  {
    // Oref_sindex returns the index of the object in the segment
    return x & ((1 << Onum_bits + Page_num_bits) - 1);
}
inline Uint Oref_pagenum(Oref x) { return Oref_sindex(x) >> Onum_bits; }
// effects: Returns the page number in the segment where x exists

inline Uint Oref_onum(Oref x) { return Oref_sindex(x) & Onum_bitmask; }
// effects: Returns the object number in the page where x exists

inline Oref Oref_create(Uint page_id, Uint onum) {
    // effects: Given the segid and onum of object in page, returns its oref
    return (((Oref) page_id) << Onum_bits) | onum;
}

inline Oref Oref_screate(Uint segid, Uint index) {
    // effects: Given the segid and index of object in seg, returns its oref
    return (((Oref) segid) << (Onum_bits + Page_num_bits)) | index;
}


inline Oref Oref_seg_page_create(Uint segid, Uint pagenum, Uint onum) {
    // XXX better name needed!
    // effects: Given segid, pagenum and onum of an object, returns its oref
    Oref result = (segid << (Page_num_bits + Onum_bits)) |
                  (pagenum << Onum_bits) | onum;
    return result;
}


inline bool Oref_encode(Oref o, Network *net) {
    return net->send_ubits32(Ubits32(o), true);
}

inline bool Oref_decode(Oref *o, Network *net) {
    return net->recv_ubits32(o);
}

extern void Sort_orefs(Oref* list, int n);
/*
 * modifies	"list[0..n-1]"
 * effects	Sorts the list contents in increasing order.
 */

extern void Oref_print(Oref oref, FILE *fp=NULL);
// effects: Prints a human readable version of oref on fp


extern void Oref_unparse(Oref oref, char *str);
// requires: Sufficient space has been allocated in str
// effects: Prints a human readable version of oref into str


/* Various functions to convert between pagenums, pageids, segids etc */

inline Uint Pageid_from_segid(Uint segid, Uint pnum) {
    // effects: Returns the pageid of page number "pagenum" in segment "segid"
    th_assert(pnum < Max_pages_in_seg, "Page number too high in segment");
    return (segid << Page_num_bits) | pnum;
}

inline Uint Segid_from_pageid(Uint pageid) {
    // effects: Returns the segment index of pageid
    return pageid >> Page_num_bits;
}

inline Uint Pagenum_from_pageid(Uint pageid) {
    // effects: Returns the page number of pageid in its segment
    return pageid & ((1 << Page_num_bits) - 1);
}

inline void Pagenum_onum_from_index(Uint index, Uint& page_num, Uint& onum) {
    // effects: Given an object "index" in a segment, modifies "page_num",
    //          "onum" to contain the location of the object in the segment
    //          i.e. object number onum on page number page_num in segment
    page_num = index >> Onum_bits;
    onum = index & Onum_bitmask;
}


// A bitmap that has a bit per page in a segment
#define BITMAP_CLASS Seg_bmap
#define BITMAP_SIZE  Max_pages_in_seg
#include "utils/bitmap.h"

#endif /* _OREF_H */
