#include <stdio.h>
#include <string.h>
#include "utils/basic.h"
#include "utils/bool_vector.h"
#include "utils/network.h"
#include "utils/sort.h"
#include "or_obj.h"
#include "page.h"

#include "or/gc/gc.h"
#include "or/gc/class_map.h"

static Page empty_page(0); // Empty page kept around

struct Offset_info { // Struct used for sorting offsets
    Ubits16 offset;
    Offset onum;
};

// We need to define comparison operators on "Mod_object".
inline static int cmp(Offset_info off1, Offset_info off2) {
    return (off1.offset - off2.offset);
}

inline int operator < (Offset_info o1, Offset_info o2){return (cmp(o1,o2)< 0);}
inline int operator <=(Offset_info o1, Offset_info o2){return (cmp(o1,o2)<=0);}
inline int operator ==(Offset_info o1, Offset_info o2){return (cmp(o1,o2)==0);}
inline int operator !=(Offset_info o1, Offset_info o2){return (cmp(o1,o2)!=0);}
inline int operator >=(Offset_info o1, Offset_info o2){return (cmp(o1,o2)>=0);}
inline int operator > (Offset_info o1, Offset_info o2){return (cmp(o1,o2)> 0);}

Uint Page::empty_page_bytes() {
    return empty_page.bytes_available();
}

void Page::init() {
    header.orx = Null_or;
    header.osize = 0;
    header.min_offset = Page_size/Slot_size; 
}

int Page::new_onum() const {
    Uint onum = 0;
    for (; onum < header.osize; onum++) {
	if (header.offsets[onum] == Null_offset) return onum;
    }
    if (onum <= Max_onum) return onum;
    return -1;
}

Slot* Page::allocate(unsigned int onum, unsigned int slots) {
    unsigned int avail = bytes_available();
    slots += OR_obj_headers;
    // If the onum is being reused, there is no extra bytes need in the offset
    // table; otherwise, we need to add zero entries to it till onum's entnry
    unsigned int num_onums = header.osize;
    unsigned int offset_overhead = onum < num_onums ? 0:
	(onum - num_onums + 1) * Overhead_bytes; 
    if ( avail < slots * Slot_size + offset_overhead)
	return NULL;

    th_assert(onum >= num_onums || header.offsets[onum] == Null_offset,
	      "The onum is not free and allocation is being done on it");
    if (offset_overhead > Overhead_bytes) // NULL the extra entries tiil onum
	bzero((char *) &header.offsets[num_onums],
	      (onum - num_onums) * sizeof(Offset));
    header.offsets[onum] = header.min_offset - slots;
    th_assert(header.offsets[onum] < Page_slots, "Offset too high");
    header.min_offset = header.offsets[onum];
    if (onum >= num_onums) header.osize = onum + 1;
    Slot *allocptr = &data[header.offsets[onum]];
    return allocptr;
}

int Page::retain(Bool_vector* live_objs, Uint start_bool, 
		 int segnum, int pagenum) {

    // We sort the array of offsets according to the increasing offsets  (to
    // determine the size of the objects) and then simply shifts objects that
    // are not needed any more.
    // The sorting can be optimized by bucket sort and checking if sorting
    // really needs to be done

    int num_onums = header.osize;
    Offset_info *offsets = sort_offsets();
    int current_shift = 0;
    int prev_obj_offset = offsets[num_onums - 1].offset;
    int dead_count = 0;

    // If last obj on page becomes non-live
    for (int i = num_onums - 1; i >= 0; i--) {
	if (offsets[i].offset == Null_offset) 
	    // no more offsets to be checked
	    break;
	int test_elem = start_bool + offsets[i].onum;
	if (live_objs->test(test_elem)) {
	    if (current_shift) {
		printf("Moving oref=%d\n",
		       Oref_seg_page_create(segnum, pagenum, offsets[i].onum));
		// Object has to be shifted down.
		// memmove handles overlapping correctly
		int source_offset = offsets[i].offset;
		int obj_size      = prev_obj_offset - offsets[i].offset;
		int dest_offset   = source_offset + current_shift;
		memmove(&data[dest_offset], &data[source_offset],
			obj_size * Slot_size);
		// Important to fix the header table!
		header.offsets[offsets[i].onum] = dest_offset;
	    }
	} else {
	    // Object is not live. We need to increase the shift factor.
	    printf("Collecting oref=%d\n", //DEBUG
		   Oref_seg_page_create(segnum, pagenum, offsets[i].onum));
	    dead_count++;
	    int obj_size  = prev_obj_offset - offsets[i].offset;
	    current_shift += obj_size;
	    // Fix the header table.
	    header.offsets[offsets[i].onum] = Null_offset;
	}
	prev_obj_offset = offsets[i].offset;
    }
    delete [] offsets;
    printf("Purged %d of %d objects on this page.\n", //DEBUG
	   dead_count, num_onums); //DEBUG
    return dead_count;
}

Page_state Page::get_current_state() {
    // Simply use min_offset.
    Page_state ps;
    ps.min_offset = header.min_offset;
    return (ps);
}

void Page::reset_state(Page_state saved_state) {
    Ubits16 old_min_offset = saved_state.min_offset;

    // Reset all offsets that are "above" (less than) old_min_offset;
    int onum = 0;
    for (; onum < header.osize; onum++) {
	if (header.offsets[onum] < old_min_offset)
	    header.offsets[onum] = Null_offset;
    }
    header.min_offset = old_min_offset;
}

void Page::print(bool whole_print, FILE *fp) {
    if (!fp) fp = stderr;
    fprintf(fp, "Page: OR = %d, min_offset = %d, Num_objects = %d\n",
	    header.orx, header.min_offset,
	    header.osize);
    fprintf(fp, "Offsets: \n");
    for (int i = 0; i < header.osize; i++) {
	fprintf(fp, "<%d, %d> ", i, header.offsets[i]);
    }
    fprintf(fp, "\n ------------------ Objects ------------------ \n");
    Page::Slow_iter iter(this);
    Slot* slots;
    Uint onum, num_slots;
    while (iter.get(onum, slots, num_slots)) {
	fprintf(fp, "Onum = %3u, Total slots = %3u, Value = ", onum, num_slots);
	OR_obj *obj = (OR_obj *)slots;
	if (whole_print)
	    obj->print(num_slots - OR_obj_headers, fp);
	else 
	    fprintf(fp, "\n");
    }
    fprintf(fp, "\n");
}

bool Page::encode(Network *net) const {
    bool result = net->send_buffer(data, Page_size);
    return result;
}

bool Page::decode(Network *net) {
    bool result = net->recv_buffer(data, Page_size);
    return result;
}

Page::Slow_iter::Slow_iter(Page *iter_page) {
    offsets = iter_page->sort_offsets();
    index = 0;
    page = iter_page;
    size = page->header.osize;
}

bool Page::Slow_iter::get(Uint& onum, Slot*& obj, Uint& num_slots) {
    if (index < size) {
	onum = offsets[index].onum;
	num_slots = offsets[index+1].offset - offsets[index].offset;
	obj = &page->data[offsets[index].offset];
	index++;
	return TRUE;
    }
    return FALSE;
}

Page::Slow_iter::~Slow_iter() {
    delete [] offsets;
}

int  Page::search(Uint onum, Slot*& obj) {

    if (onum >= header.osize) return -1;
    obj = lookup(onum);
    if (!obj) return -1;

    // Now to get the size of the object from the offsets array
    // Find the offset that is next highest after this one

    Uint obj_offset = header.offsets[onum];
    Uint next_offset = Page_size/Slot_size;
    for (Uint i = 0; i < header.osize; i++) {
	Uint offset = header.offsets[i];
	if (offset > obj_offset && offset < next_offset)
	    next_offset = offset;
    }
    return next_offset - obj_offset;
}

// Internal methods

Offset_info* Page::sort_offsets() {
    // XXX Better/faster schemes are possible and will be implemented later
    int num_onums = header.osize;
    Offset_info* offsets = new Offset_info[num_onums+1];
    for (int i = 0; i < num_onums; i++) {
	offsets[i].offset = header.offsets[i];
	offsets[i].onum = i;
	th_assert(offsets[i].offset < Page_slots, "Offset too high");
    }
    quick_sort(offsets, num_onums);
    offsets[num_onums].offset = Page_size/Slot_size;
    return offsets;
}

#define QUICK_SORT_ELEMENT Offset_info
#include "utils/impl_sort.t"
