#ifndef _PAGE_H
#define _PAGE_H

#include <stdio.h>
#include "utils/basic.h"
#include "utils/bits.h"
#include "utils/array.h"
#include "utils/intarray.h"
#include "slot.h"
#include "or_num.h"
#include "oref.h"
#include "or_obj.h"

class Network;
class Bool_vector;
class Offset_info;

#if PAGE_SIZE_BITS > BASE_PAGE_SIZE_BITS
typedef Ubits16 Offset; // Offsets used to locate objects
#else
typedef Ubits8 Offset; // Offsets used to locate objects
#endif


// Bytes in a page
const Uint Page_size = 1 << Page_size_bits;
// Slots in a page
const Uint Page_slots = Page_size/Slot_size;
// Invalid offset in the offset array
const Uint Null_offset     = 0;
// Number of bits used for the min_offset
const Uint Min_offset_bits = Page_size_bits - Slot_bits;
// Mask used to extract min_offset
const Uint Min_offset_mask = (1 << Min_offset_bits) - 1;
// Max allowed min offset value
const Uint Max_min_offset  = Min_offset_mask;
// Byte overhead per obj (not stored in the object itself)
const Uint Overhead_bytes = sizeof(Offset);


// The state of a page captured by get_current_state
class Page_state {
 public:
  Ubits16  min_offset;      // Save min_offset
};

// Overview: A page is a fixed-sized unit of fetching and caching at the FE.
//           Each page contains a table that maps object index to offset in
//           the page.

// NOTE: To see how the name space has been broken up into pages, segments,
//       etc, read "common/oref.h"

class Persistent_cache;

union Page {
  public:

    void init();
    // effects:  Initializes a page.

    Page() {
    // effects: Creates an initialized page with Null_or
      init();
    }

    Page(OR_num orx) {
    // effects:	Creates an initialized page with no objects for OR number or
      init();
      header.orx = orx;
    }

    void set_or(OR_num orx) { header.orx = orx; }
    // modifies: this
    // effects:  Sets the OR number of this page to be "or".

    OR_num get_or() const { return header.orx;}
    // effects:  Returns the OR number of this page.

    Ubits64 get_vnum() const { return *((Ubits64 *) ((char*) data) + 8);}
    // effects: This method is used only for some performance tests
    // for the orphan detection code. It does not make any sense otherwise

    int num_objects() const;
    // effects:	Return the number of object names used on this page
    //          This number is >= the number of objects on the page

    int bytes_available() const;
    // effects: Returns the number of bytes available on this page.
    //          NOTE: The caller should add Overhead_bytes to the size of
    //          an object to determine whether a set of objects will fit
    //          on this page. 

    Slot* lookup(Uint onum);
    // effects: Looks up the object number "onum" in this page and returns a
    //          pointer to it. Returns NULL if the object does not exist

    Slot* fast_lookup(Uint onum);
    // requires: onum is an index of a valid existing object
    // effects: Looks up the object number "onum" in this page and returns a
    //          pointer to it.

    int new_onum() const;
    // effects: Returns a new (unused) onum available on this page
    //          If no onum is available, returns -1
    
    void clear_page();
    // effects: Sets the number of objects used in this page to be 0

    bool empty() const;
    // effects: Returns true iff there are no objects on that page

    Slot* allocate(Uint onum, Uint slots);
    // requires: onum is a new onum on the page (i.e. unused)
    // effects: Allocates space for an object with "slots" number of slots
    //          (this does not include the header slots)
    //          and assigns it onum. Returns a pointer to the allocated space
    //          (the pointer is guaranteed to be word aligned). Returns NULL
    //          if sufficient space is not available

    int retain(Bool_vector* live_objs, Uint start_bool, 
	       int segnum, int pagenum);
    // effects: Retains live objects specified by "live_objs" and "start_bool".
    //          Onum i is live iff live_objs[start_bool+i] is true.
    //          Onums (and object space) for which the booleans in
    //          live_objs are FALSE are marked for reuse. Objects that are
    //          retained may be moved around.
    //          Returns a count of dead objects.

    Page_state get_current_state();
    // effects: Returns Page_state object that can be used by reset_state
    //          to return self to the current state.

    void reset_state(Page_state saved_state);
    // requires: saved_state should be the result of the last call of
    //           calling get_current_state.
    // effects: Returns self to the state represented by saved_state.

    bool encode(Network *net) const;
    // effects: Sends the contents (as it is) over the network net
    //          Returns TRUE iff succeeds

    bool decode(Network *net);
    // effects: Receives the page from net and initializes this
    //          Returns TRUE iff succeeds

    int  search(Uint onum, Slot*& obj);
    // modifies: obj
    // effects: Searchs for the object number onum and modifes obj to point to
    // it. Returns the number of total slots in the object (i.e. including
    // header slots). Returns -1 if object does not exist.
    // WARNING: This function may go over the whole page to determine the
    // size of the object and may be inefficient. In general, this function
    // is used for printing/debugging objects and you should NEVER have
    // to use it.

    void print(bool whole_print, FILE *fp = NULL);
    // whole_print specifies if the whole object has to be printed

    class Iter {
	// An iterator for yielding the objects on the page
	
	// Once created, an iterator must be used and destroyed before any
	// objects are added/removed from the page. The effect is undefined if
	// an iterator method is called after such a change. 

      public:
	Iter(Page *p);
	// effects: Creates an initialized object using page
	//          that will start yielding objects from onum 0

        ~Iter();
	// effects: Deallocates the iterator.
      
        bool get(Slot*& obj);
	// modifies: obj	
        // effects: Sets obj to point to the next object (with the next higher
	///         onum) in the page and returns TRUE. If this object was
	//          deallocated, obj will be set to 0. Returns FALSE if there
	//          are no more objects (and returns an undefined value in obj).

      private:
	Page *page;    // Page from which objects will be iterated over.
	int index;     // Index of next object to be iterated over.
	int size;      // Size of the offsets array.
    };



    class Slow_iter {
	// An iterator for yielding the objects on the page
	
	// Once created, an iterator must be used and destroyed before any
	// objects are added/removed from the page. The effect is undefined if
	// an iterator method is called after such a change. This iterator
        // provides more information but it is slower than Iter.

      public:
	Slow_iter(Page *page);
	// effects: Creates an initialized object using page

	~Slow_iter();
	// effects: Deallocates the space

	bool get(Uint& onum, Slot*& obj, Uint& num_slots);
	// modifies: onum, obj, num_slots
	// effects: Sets "onum" to the next non-deallocated object. Sets obj
	//          to point to that object, num_slots to be the size in slots
	//          (including the header slots)
	//          and returns TRUE. Returns FALSE if there are no more
	//          objects (and does not modify onum, obj, num_slots

      private:
	Page *page;    // Page from which objects will be iterated over.
	Offset_info *offsets;// Offsets sorted in increasing order
	int index;           // Next object to be iterated over
	int size;            // Number of onums to be yielded
    };

    static Uint empty_page_bytes();
    // effects: Number of bytes available on an empty page

    static Uint total_obj_bytes(Uint num_slots);
    // effects: Returns the MAX number of bytes needed to put an object with
    //          field slots "num_slots" on a page. The actual number of bytes
    //          used on a page may be less (by 1 or 2 bytes)

    Page *next() { return next_; }
    // effects: Returns the next page in the list
 
    void set_next(Page *n) { next_ = n; }
    // effects: sets the next page in the list and places unspecified values
    // in all the other fields in the page.

  private:
    friend Iter;
    friend Slow_iter;
    friend Persistent_cache; // Persistent_cache needs low level 
                             // access for good performance.
    struct Page_header {
	Ubits32  orx;              // OR number of this page
	Ubits16  min_offset;      // Min offset of an object
	Offset   osize;           // Size of the offset array
	Offset   offsets[1];      // Offset array
	// The offsets are from the start of the page and are in terms of
	// slots. A Null_offset means an unused name.
    } header;
    Slot data[Page_size/Slot_size];    // The actual page contents
    Page *next_;                       // Used to store pages in lists.


    Offset_info* sort_offsets();
    // effects: Sorts the offsets in the offset table and places the
    //          <onum,offset> in increasing offset order and returns the result
    //          Caller must delete the result
    //          The size of the array is the size of the offset table + 1.
    //          (It adds an offset entry whose offset is Page_size/Slot_size)

    // Notes:
    // The offset table is laid out such that the ith entry corresponds to the
    // ith onum. However, the objects on the page are not laid out in onum
    // order. The objects are laid out "compactly", i.e. the next object
    // starts in the next the previous object ends
    // The objects are allocated from the other end of the array
};

inline Uint Page::total_obj_bytes(Uint num_slots) {
    return (num_slots + OR_obj_headers) * Slot_size + Overhead_bytes;
}

inline int Page::num_objects() const { return header.osize; }

inline Slot* Page::lookup(Uint onum) {
    if (onum >= header.osize) return NULL;
    unsigned int offset = header.offsets[onum];
    if (offset == Null_offset) return NULL;
    return &data[offset];
}

inline Slot* Page::fast_lookup(Uint onum) {
    th_assert(onum < header.osize, "Bad object number");
    unsigned int offset = header.offsets[onum];
    th_assert(offset != Null_offset, "Null object requested");
    return &data[offset];
}

inline bool Page::empty() const {
    return header.osize == 0;
}

inline Page::Iter::Iter(Page *p) { 
  page = p; 
  index = 0; 
  size = page->header.osize; 
}

inline Page::Iter::~Iter() {}
      
inline bool  Page::Iter::get(Slot*& obj) {
  if (index < size) {
    obj = &page->data[page->header.offsets[index]];
    index++;
    return TRUE;
  }
  return FALSE;
}

inline int Page::bytes_available() const {
  int bytes_avail = (header.min_offset * Slot_size)
    - sizeof(Ubits32) - sizeof(Ubits16) - sizeof(Ubits8)
    - header.osize * sizeof(Offset);
  return bytes_avail;
}

inline void Page::clear_page() {
    header.osize = 0;
}

#endif /* _PAGE_H */
