
#include "fe/main/fe_config.h"
#include "persistent_cache.h"
#include "utils/th_assert.h"
#include "utils/hashfuncs.h"
#include "compacted_map.h"

#include <stream.h>

Compacted_map::Compacted_map(unsigned size) {
  size = size/6;  // average 6 mappings (2 entries) per bucket

  // Round size to next power of 2
  unsigned mask = 1;
  unsigned bit_count = 0;
  while (mask < size) {
    mask = mask << 1;
    bit_count++;
  }
  size = mask;
  shift_bits = sizeof(unsigned)*8 - bit_count;
  table = new Compacted_map_entry[size];
  num_entries = size;
  num_mappings = 0;
}


Compacted_map::~Compacted_map() { delete [] table; }


int Compacted_map::decrement(OR_num orx, Uint page_id, Uint delta) {
  unsigned bucket_index = hash_unsigned(page_id) >> shift_bits;
  Compacted_map_entry *cur;
  Compacted_map_entry *prev = 0;
  int count=0;
  int i = 0;

  // Search for a mapping for <or, page_id> ...
  for (cur = table + bucket_index; cur != 0; cur = cur->next) {
    for (i = 0; i < 3; i++) {
      if (cur->mappings[i].orx() == orx && cur->mappings[i].page_id() == page_id) {
	count = cur->mappings[i].count();
        break;
      }
    }

    if (count) break;

    prev = cur;
  }
  
  // No entry found.
  if (count == 0) return 0;

  // Compute new number of cached objects.
  count -= delta;
  if (count > 0) {
    cur->mappings[i].set_count(count);
    return count;
  }
   
  // No more objects cached.
  cur->mappings[i].mark_free();
  num_mappings--;

  // XXXX should queue message to OR informing it that no more
  // objects from this page are cached if there is no intact version
  // of the page.
  
  // Deallocate entry if possible.
  if (prev && cur->mappings[0].is_free() && cur->mappings[1].is_free() 
      && cur->mappings[2].is_free()) {
    prev->next = cur->next;
    delete cur;
    num_entries--;
  }
  return count;
}



int Compacted_map::increment(OR_num orx, Uint page_id, Uint delta) {
  th_assert(delta > 0, "Invalid argument\n");

  unsigned bucket_index = hash_unsigned(page_id) >> shift_bits;
  Compacted_map_entry *cur;
  Compacted_map_entry *free_entry = 0;
  int free_index = 0;

  // Search for a mapping for <or, page_id> ...
  for (cur = table + bucket_index; cur != 0; cur = cur->next) {
    for (int i=0; i < 3; i++) {
      if (cur->mappings[i].orx() == orx && cur->mappings[i].page_id() == page_id) {
	// ... found a mapping; update it.
	int count = cur->mappings[i].count() + delta; 
	cur->mappings[i].set_count(count);
	return count;
      }

      // Remember free mappings along the way to reuse them.
      if (free_entry == 0 && cur->mappings[i].is_free()) {
	free_entry = cur;
	free_index = i;
      } 
    }
  }
  
  // No mapping found. If there was a free mapping in the
  // bucket reuse it. Otherwise, allocate a new entry.
  if (free_entry == 0) {
    free_entry = new Compacted_map_entry;
    cur = table + bucket_index;
    free_entry->next = cur->next;
    cur->next = free_entry;
    num_entries++;
  }

  free_entry->mappings[free_index].set_or(orx);
  free_entry->mappings[free_index].set_page_id(page_id);
  free_entry->mappings[free_index].set_count(delta);
  num_mappings++;

  return delta;
}


void Compacted_map::print_stats(ostream &o) {
  // Effects: Prints map statistics.
  o << "\nCompacted map: \n"
    << " Total space (bytes): " << num_entries * sizeof(Compacted_map_entry)
    << "\n Number of mappings: " << num_mappings << "\n";
}
  





