#ifndef _Optimal_replacer_h
#define _Optimal_replacer_h 1

#include "utils/array.h"
#include "utils/openhashmap.h"
#include "utils/hashfuncs.h"
#include "common/or_num.h"
#include "common/page.h"
#include "common/xref.h"
#include "trace_replacer.h"


class Stack_refs;

// 
// Declarations for a cache of trace entries organized as a hash table 
// that maps page ids to lists of last_access values for several trace
// entries for a page. Each list is ordered by increasing value.
//

// Array of accesses
declareArray(AccessesArray, Uint)

struct Trace_cache_entry {
  AccessesArray accesses;
  int accesses_cursor;

  Trace_cache_entry() { accesses_cursor = 0; }  
};

#define Uint_equal(i1, i2) (i1 == i2)
declareOpenHashMap(Trace_cache, Uint, Trace_cache_entry*, hash_int, Uint_equal)



//
// List that caches the next last_access value for a page.
//
struct Trace_list_entry {
  Trace_entry t;
  Trace_list_entry *next;
};


// Minimum number of entries into the future used to make replacement
// decisions.
const int Trace_window = 1000000;

class Optimal_replacer : public Trace_replacer {
  // 
  // Implements optimal page replacement in the cooperative cache
  // using a trace file (FE->trace_file) describing all the accesses.
  // It maintains a list of Trace_list_entry with entries for all cached
  // pages which is used to perform optimal page replacement, i.e.
  // the page that is accessed farthest into the future is replaced.
  // Currently only works for read-only traversals and traversals that
  // only access objects from OR 1.
  //


public:
  Optimal_replacer(Stack_refs const *sr);
  // Effects: Creates an optimal page replacer.
  // Requires: FE->trace_file is open for reading and refers to 
  //   a valid trace file and "sr" keeps track of pointers from the stack or
  //   registers into the page cache.

  ~Optimal_replacer();
  // Effects: Deletes the Optimal_replacer recovering all the 
  //   allocated storage.
 
  void bring_up_to_date(Uint access_counter);
  // Effects: Brings cached list up-to-date.
  
  Page *replace_page();
  // Effects: Computes the optimal page to replace (constrained 
  //   to not being referenced from the stack) and returns it.

  void add_page(OR_num orx, Uint page_id);
  // Effects: Add entry for page "p" to the list of cached pages.

private:
  Stack_refs const *sr;       // Pointer to storage arena being managed.
  Trace_list_entry *cached_list; // List with trace entries for cached pages.
 
  // Hash tables that cache portion of trace of current interest.
  Trace_cache *trace_cache1;
  Uint last_access_cached1;
  Trace_cache *trace_cache2;
  Uint last_access_cached2;

  void read_next_chunk(Trace_cache *dest, Uint &last_access);
  // Effects: Reads next chunk in trace file into dest.

  Trace_entry search_trace_caches(Uint access_counter, Trace_entry &stale);
  // Effects: Searches for an up-to-date version of stale in the trace caches
  //  and returns it.

};

#endif //_Optimal_replacer_h







