#ifndef _Page_map_h
#define _Page_map_h 1

#include "common/or_num.h"
#include "common/page.h"

class Persistent_cache;

class Page_map {
  //
  // Overview: A map from <or,page_id> to the index of the page in the PC.
  //
public:
  Page_map(unsigned size, Persistent_cache *pc);
  // Effects: Creates a map. size should be the predicted number
  //  of mappings in the map.

  ~Page_map();
  // Effects: Destroys the map and all its associated storage.

  Page *lookup(OR_num orx, Uint page_id) const;
  // Effects: Looks up page <or, page_id> in the map. If found
  // returns a pointer to the page in the PC. Otherwise,
  // returns 0.

  void store(Uint page_id, Uint index);
  // Requires: Page is not in the map.
  // Effects: Inserts a mapping from <or, page_id> to index in the map.

  void remove(OR_num orx, Uint page_id);
  // Requires: Page <or, page_id> is in the map.
  // Effects: Removes mapping for page <or, page_id> from map.

private:

  //
  // Private types:
  //
  class Page_map_mapping {
    // Overview: Describes a page that is cached intact in the PC.
  public:
    Page_map_mapping() { page_id = 0; }
    // Effects: Create a free entry.

    bool is_free() { return page_id == 0; }
    // Effects: Return true iff entry is free.

    void mark_free() { page_id = 0; }
    // Effects: Marks mapping as free.

    Page *is_page(OR_num orx, Uint pid, Page *base) {
      if (page_id == pid) {
        Page *res = base+page_index;
        if (orx == res->get_or()) return res;
      }
      return 0;
    }
    
    Uint page_id;
    Uint page_index;
  };


  struct Page_map_entry {
    // Overview: Groups several Page_map_mapping and a shared next into
    //   a 32 byte cache line.
    Page_map_entry() {next = 0;}
    
    Page_map_mapping mappings[3];
    Page_map_entry *next;
  };


  Page_map_entry *table; // Array with num_buckets entries.
  unsigned num_buckets;  // Number of buckets in table (It is always a power of 2).
  unsigned shift_bits;   // Number of bits in unsigned minus the bits in num_buckets.
  Page *base;
};


//
// Methods inlined for speed.
//


inline Page *Page_map::lookup(OR_num orx, Uint page_id) const {
  unsigned bucket_index = hash_unsigned(page_id) >> shift_bits;
  for (Page_map_entry *tmp = table + bucket_index; tmp != 0; tmp = tmp->next) {
    for (int i=0; i < 3; i++) {
      Page *p = tmp->mappings[i].is_page(orx, page_id, base);
      if (p) return p;
    }
  }
  return 0;
}


#endif //_Page_map_h




