#include <iostream.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <setjmp.h>

#include "utils/fail.h"
#include "utils/compat.h"
#include "fe/main/fe_config.h"
#include "storage_arena.h"

Storage_arena::Storage_arena(unsigned int size) {
  num_pages = size;
  extra = 0;

  // Linux mmap returns aligned on 4K boundaries
  // Allocate size+1 Pages to guard against misaligned boundary
#ifdef __linux__
  size++;
  extra = Page_size;
#endif

  start_ = (Page *) mmap(NULL,          // No specified address
			 (size) * sizeof(Page),    // This much space
			 PROT_READ | PROT_WRITE, // R&W access
			 MAP_ANONYMOUS |         // Not from a file (zeroed pages)
			 MAP_VARIABLE |          // No paricular place
			 MAP_PRIVATE,            // Changes are private
			 -1,                     // No file
			 0);                     // No offset in file
  if (start_ == (Page *)-1) {
    sysfail("mmap call failed");
  }

  actual_start_ = start_;

  // Make sure start_ is aligned on 8K boundary
#ifdef __linux__
  if ((long)start_ % Page_size)
    start_ =  (Page*)((char*)start_-((long)start_ % Page_size)+Page_size);
#endif

  next_page = 0;
  free_list = 0;
  free_list_size = 0;
}

Storage_arena::~Storage_arena(void) {

  if (munmap((caddr_t)actual_start_, num_pages*sizeof(Page) + extra) == -1) {
    sysfail("munmap call failed");
  }
}


Storage_arena *Storage_arena::split(unsigned int num) {
  th_assert(num < num_pages && next_page == 0, "Invalid argument");

  Storage_arena *new_arena = new Storage_arena;
  new_arena->num_pages = num_pages - num;
  num_pages = num;
  new_arena->start_ = start_+num;
  // new arenas are always aligned
  new_arena->actual_start_ = new_arena->start_;
  new_arena->next_page = 0;
  new_arena->free_list = 0;
  new_arena->free_list_size = 0;

  // Where does the extra go?
  if (extra != 0) {
    if (actual_start_ == start_) { // arena is aligned, so all in new_arena
      new_arena->extra = extra;
      extra = 0;
    } else {
      if (extra == Page_size) { // full page, so split
	extra = (char *) start_ - (char *) actual_start_;
	new_arena->extra = Page_size - extra;
      } else { // partial page, so stays with original
	new_arena->extra = 0;
      }
    }
  }
  return new_arena;
}


Page *Storage_arena::alloc_pages(int num) {
  if (num <= 0) return NULL;
  if (next_page + num > num_pages) return NULL;
  Page *res = &(start_[next_page]);
  next_page += num;
  return (res);
}



