#ifndef _Storage_arena_h
#define _Storage_arena_h

#include "common/page.h"

class Storage_arena {
  //
  // Overview: Implements the basic level of storage allocation that
  //   grabs virtual-memory space and hands it out as pages.

 public:
  //
  // Constructors:
  //

  Storage_arena(unsigned int size);
  // Requires: size > 0;
  // Effects: Allocates space for "size" pages

  ~Storage_arena(void);
  // Effects: Destroys the Storage_arena and returns all the storage
  //   memory associated with it.

  Storage_arena *split(unsigned int num_pages);
  // Requires: num_pages < size of arena and no pages were allocated.
  // Effects: Splits "this" in two storage arenas (useful to allocate
  //   contiguous arenas). "this" is changed to contain only
  //   "num_pages" and a new storage arena is returned that contains the
  //   rest of the pages in "this" (the caller is responsible for deleting
  //   the returned object).

  //
  // Methods:
  //

  Page *start() const;
  // Effects: Returns a pointer to the first page in this.

  unsigned int size() const;
  // Effects: Returns the total amount of VM space (in pages) in this.

  Page *alloc_page();
  // Effects: Allocates and returns a pointer to an empty page. The page is
  //   initialized by Page::init. If no free pages returns NULL.

  Page *alloc_pages(int num);
  // Effects: If num <= 0, returns NULL.
  // Otherwise allocates "num" contiguous pages and returns a pointer
  //to the first. If unable to do so, returns 0. The pages have
  // NOT been initialized.

  void free_page(Page *p);
  // Effects: Frees page p.

  unsigned num_free() const;
  // Effects: Returns the number of free pages in the storage arena.


 private:
  // Representation

  Page *start_;             // start of the paged region
  unsigned int num_pages;   // number of pages mmaped
  unsigned int next_page;   // index of the next page available
  Page *free_list;
  unsigned int free_list_size;

  // Linux mmap returns 4K aligned pages, so have to keep track of actual
  // start of region and where there is any extra space in this arena
  Page *actual_start_;      // start of region returned by mmap
  unsigned int extra;       // number of extra bytes

  Storage_arena() {}

};


inline Page *Storage_arena::start() const {
  return start_;
}

inline unsigned int Storage_arena::size() const {
  return num_pages;
}

inline void Storage_arena::free_page(Page *p) {
  p->set_next(free_list);
  free_list = p;
  free_list_size++;
  th_assert(free_list_size != 0, "Overflow\n");
}

inline Page *Storage_arena::alloc_page() {
  Page *p = 0;
  if (free_list != 0) {
    p = free_list;
    free_list = p->next();
    th_assert(free_list_size != 0, "Underflow\n");
    free_list_size--;
    p->init();
  } else if (next_page < num_pages) {
    p = &(start_[next_page++]);
    p->init();
  }
  return p;
}

inline unsigned Storage_arena::num_free() const {
  return free_list_size + num_pages - next_page;
}


#endif // _Storage_arena_h

