#include <string.h>
#include <stream.h>

#include "compiler/C++wrappers/obj.h"

#include "compiler/C/obj.h"

#include "runtime/stats.h"

#include "resident_object_table.h"
#include "persistent_cache.h"
#include "volatile_heap.h"
#include "surrogate_table.h"

#include "boot/type_init.h"

#include "fe/main/fe_config.h"

#include "fe/boot/fe_wk_xref.h"

#include "config/vdefs/DV_RESIDENCY_CHECK.h"
#include "config/vdefs/NDEBUG.h"
#include "config/vdefs/USE_REF_COUNTS.h"

#include "common/or_obj.h"
#include "common/shortp.h"
#include "common/null_or_ref.h"

#include "utils/fail.h"
#include "utils/mdebug.h"

inline Core Resident_object_table::lookup_and_install(OR_num orx, Oref oref) {
  Core target = lookup(orx, oref);

  if (target) return target;
 
  // Target object was not in the swizzle table, get it from
  // the pc and install it in the swizzle table.
  FE_STATS(num_pc_lookups++)
  OR_obj *target_fields = FE->pc->fetch(orx, oref);
  target = install_object(orx, oref, target_fields);
  th_assert(target != 0, "PC did not fetch fields\n");
  
  // At this point the target should have a null ref. count.
  th_assert(target->ref_count() == 0, "Ref. count != 0\n");

  return target;
}


inline Obj Resident_object_table::swizzle(Oref oref, Fields o, struct Class_s *t) {
  FE_STATS(num_swizzles++)
  th_assert(oref != 0, "Invalid oref");

  OR_num orx = o->orx();
  Core target = lookup_and_install(orx, oref);

#if USE_REF_COUNTS
  if (!o->is_written() || o->use_eager_ref_counting()) {
    // If o was not modified in the current transaction or if o 
    // was modified but it is marked for eager ref count updates, we
    // increment the count for target.
    inc_rc(target);
  } else {
    // If o was modified by the current transaction marks
    // o as "swizzled after copied" and does not increment count
    // for target.
    o->stamp_pointer_update();
  }
#endif

  // Return a pointer to the type "t" dispatch header of target
  Obj target_obj = target->to_obj();

  int offset = 0;
  struct Class_s *oclass = target_obj->get_class();
  if (oclass !=  (struct Class_s *)t)
    offset = Header_offset(oclass, t);

  return target_obj+offset;
}



void Resident_object_table::install_new_object(OR_num orx, Oref oref, Core c) { 
  // Link c to the head of its bucket's list.  
  int bindex = orefs_bucket(oref);
  c->set_next(SHORTP_AS_FULL(Core_c, swiz_tab[bindex]));
  swiz_tab[bindex] = FULL_AS_SHORTP(c);

  // Clean free entries in bucket's list.
  Core aux = c;
  while (aux != 0) {
    Core to_delete = aux->next();
    if (to_delete && to_delete->is_free()) {
      // If following entry is also free remove it from bucket
      aux->set_next(to_delete->next());
      free_list.insert(to_delete);
      continue;
    } 
    aux = to_delete;
  }

  // Set c's oref field
  c->oref() = oref;

  // Move c's fields to the persistent cache
  unsigned num_slots = c->num_slots();
  FE_STATS(avg_inst_object += num_slots)
  FE_STATS(num_installed++)

  Fields new_fields = FE->pc->allocate(orx, oref);

  num_slots += Fields_c::header_slots(); // account for header
  memcpy(new_fields, c->get_fields(), num_slots*Slot_size);
 

  FE->vh->free_object(c);

  // Make c point to the new fields location in the PC
  c->set_fields(new_fields);
}


Obj Resident_object_table::install_root(OR_num orx, Oref oref) {
  Core target = lookup_and_install(orx, oref);
  return target->to_obj();
}


Core Resident_object_table::install_surrogate_target(OR_num surr_or,
						     Oref surr_oref,
                                                     Xref target_xref) { 
  // Check if target is in swizzle table and if not install it 
  Core target = lookup_and_install(target_xref.orx, target_xref.oref);

  // Record relation between target object and OR surrogate.
  FE->surr_table->insert(target_xref.orx, target_xref.oref, surr_or, surr_oref);

  return target;
}


Core Resident_object_table::install_object(OR_num orx, Oref oref, OR_obj *o) {
  th_assert(lookup(orx, oref) == 0, "Object already in swizzle table\n");

  Xref target_xref;
  if (o->is_surrogate(target_xref)) { 
    // o is an OR surrogate and points to an object with xref target_xref.
    return install_surrogate_target(orx, oref, target_xref);
  }


#if 0 //XXXXXXXXXXXXXXXXXXX This should get back
  // o is a regular object. Lookup class in swizzle table.
  Oref class_oref = o->class_oref();
  Core class_core = lookup(or, class_oref);
  if (!class_core) {
    // FE is not caching class fetch it and install it
    th_fail("Dynamic loading/linking of classes is not implemented yet\n");
  }

  Class_c cl = class_core->to_obj(); 
#endif

  Oref class_oref = o->class_oref();
  Class_c cl = Class_from_oref(class_oref);
  
  // Try to reuse free entries in the bucket chain
  unsigned bindex = orefs_bucket(oref);
  Core tmp = SHORTP_AS_FULL(Core_c, swiz_tab[bindex]);
  for (;tmp != 0; tmp = tmp->next()) {
    if (tmp->is_free()) break;
  }
  
  Core target = tmp;
  unsigned hsize = cl.hdr_size();

  // Remove remaining free entries in bucket's list
  // and insert them in the free list.
  while (tmp != 0) {
    Core to_delete = tmp->next();
    if (to_delete && to_delete->is_free()) {
      // If following entry is also free remove it from bucket
      tmp->set_next(to_delete->next());
      free_list.insert(to_delete);
      continue;
    } 
    tmp = to_delete;
  }

  // Check if an appropriate entry was found...
  if (!target || target->entry_size() != hsize) {
    // ...if not allocate new entry.
    target = allocate(hsize);
    
    // Link target to the bucket's list
    target->set_next(SHORTP_AS_FULL(Core_c, swiz_tab[bindex]));
    swiz_tab[bindex] = FULL_AS_SHORTP(target);
  }

  // Initialize target's core and fields
  Fields target_fields = (Fields ) o;
  target->oref() = oref;
  target->set_fields(target_fields);
  target->set_ref_count(0);
  target_fields->init(index(target));

  // Initialize dispatch header
  cl.init_disp_hdr(target);
  
  FE_STATS(avg_inst_object += target->num_slots())
  FE_STATS(num_installed++)

  return target;
}


//
// The followoing C functions are used for compiled code to 
// access cache functionality. They are declared in 
// runtime/C/externs.h
//
struct Obj_s *Cache_swizzle(Fields_p v, Shortp _oref, struct Class_s *t) {
  // Handle empty maybes.
  if (_oref == NULL_OR_REF) return (struct Obj_s *)NULL_OR_REF;
  return (struct Obj_s *)FE->rot->swizzle(SHORTP_AS_DATA(Oref,_oref), 
				      (Fields) v, t);
}


inline Fields Resident_object_table::make_resident(Core c) {
  th_assert(c->is_discarded(), "Invalid argument\n");

  // Get object fields from the pc.
  Fields f = (Fields) FE->pc->fetch(c->or_discarded(), c->oref());
  th_assert(f != 0, "PC did not fetch fields\n");

  int c_index = index(c);
  c->set_fields(f);
  f->init(c_index);

#if DV_RESIDENCY_CHECK
  struct Class_s *cc = (struct Class_s *)c->get_class();
  c->set_dv(cc->dv); // XXX restore normal DV (must restore full DH)
#endif
  
  return f;
}


Fields_p Make_resident(struct Core_s *_c) {
  return (Fields_p) FE->rot->make_resident((Core) _c);
}


void Resident_object_table::print_swizzle_table_stats(ostream &o) {
  const unsigned hist_size = 10;
  unsigned tot_persistent=0; // Total number of objects in st.
  unsigned tot_free=0;       // Total number of free entries in st buckets.
  unsigned tot_entries=0;    // Total number of entries in st buckets.
  unsigned tot_null_rc=0;    // Total number of entries with null reference count.
  unsigned tot_discarded=0;  // Total number of entries with discarded fields.

  unsigned hist_chain[hist_size];    // Chain length histogram.
  unsigned hist_free[hist_size];     // Number of free entries per chain.
  unsigned hist_non_free[hist_size]; // Number of non-free entries in chain.

  for (unsigned i=0; i < hist_size; i++) {
    hist_chain[i] = 0;
    hist_free[i] = 0;
    hist_non_free[i] = 0;
  }
  
  for (unsigned i=0; i < swiz_buckets; i++) {
    unsigned bucket_free = 0;
    unsigned bucket_total = 0;

    Core tmp = SHORTP_AS_FULL(Core_c, swiz_tab[i]);
    for (; tmp != 0; tmp = tmp->next()) {
      if (tmp->is_free()) bucket_free++;
      else {
	if (tmp->is_discarded()) tot_discarded++;
	if (tmp->ref_count() == 0) tot_null_rc++;
      }
      bucket_total++;
    }

    tot_free += bucket_free;
    unsigned bucket_non_free = bucket_total - bucket_free;
    tot_persistent += bucket_non_free;
    tot_entries += bucket_total;

    if (bucket_free > hist_size)  bucket_free = hist_size-1;
    if (bucket_total > hist_size)  bucket_total = hist_size-1;
    if (bucket_non_free > hist_size) bucket_non_free  = hist_size-1;

    hist_chain[bucket_total]++;
    hist_free[bucket_free]++;
    hist_non_free[bucket_non_free]++;
  }

  o << " Total number of entries: " << tot_entries
    << "\n Total number of persistent objects: " << tot_persistent
    << "\n Total number of free entries: " << tot_free
    << "\n Total number of buckets: " << swiz_buckets 
    << "\n Total number of entries with null ref. count: " << tot_null_rc
    << "\n Total number of discarded entries: " << tot_discarded << endl;

  o << " Histogram of chain length: ";
  for (unsigned int i=0; i < hist_size; i++)
    o << " " << hist_chain[i];
  o << endl;

  o << " Histogram of non-free entries per chain: ";
  for (unsigned int i=0; i < hist_size; i++)
    o << " " << hist_non_free[i];
  o << endl;

  o << " Histogram free entries per chain: ";
  for (unsigned int i=0; i < hist_size; i++)
    o << " " << hist_free[i];
  o << endl;


  // Estimate clumping.
  unsigned sum = 0;
  for (unsigned int i=0; i < hist_size; i++)
    sum += hist_non_free[i]*i*i;
  o << " Clumping : " << (float)sum/tot_persistent - (float)tot_persistent/swiz_buckets 
    << endl;
}
