#include <strstream.h>

#include "compiler/C/extern.h"

#include "compiler/C++wrappers/core.h"
#include "compiler/C++wrappers/fields.h"

#include "persistent_cache.h"
#include "volatile_heap.h"
#include "storage_arena.h"
#include "resident_object_table.h"

#include "fe/main/fe_config.h"

#include "utils/fail.h"
#include "utils/mdebug.h"
#include "utils/wordsize.h"

// A null handle is never the handle of a valid allocated object, so we set
// Deleted_handle to its value
const Ubits32 Deleted_handle = Null_handle;

// The mask used for length in flist_entry. Must conform with fields.h
// To ensure that it does not overlap with the field's handle (index of
// its core in the rot).
const Uint Length_mask = 0xff; 

DEFINE_shortp(Freep);
implementArray(Vpages, Vpage*)

class Flist_entry {
    // Overview: This is a free list entry that is used for keeping track of
    // available objects in the volatile heap. Its layout must conform with the
    // layout of a Fields_c. That is hl must be the first word in the entry,
    // (it cannot have virtual methods), and the length mask must contain bits
    // that are not used by the handle in a Fields_c. 

  public:
    Handle handle() { 
      Fields f = (Fields) &hl;
      return f->get_handle();
    }
    // effects: Returns the handle of the free entry

    void set_handle(Uint a_handle) { 
	// effects: Sets the handle of the free entry to be a_handle
        Fields f = (Fields) &hl;
	f->set_handle(a_handle);
    }

    Uint length() {
	// effects: Returns the length of the free entry
	if ((hl & Length_mask) != Length_mask) {
	    return hl & Length_mask;
	} else {
	    return elength;
	}
    }

    void set_length(Uint a_length) {
	// effects: Sets the length of the free entry to be a_length
	if (a_length >= Length_mask) {
	    hl |= Length_mask;
	    elength = a_length;
	} else {
	    hl = (hl & ~Length_mask) | a_length;
	}
    }



  private:
    Slot hl; // Handle and length
 public:
    Freep_x next; // Next element in the list
 private:
    Slot elength; // Used only for the large_flist. Total num of slots
                  // Must be last instance variable in object.
};

const Uint Min_slots = 2; // Minimum number of slots of an object
const int Space_threshold = 100*1024; // If more than this space is
// fragmented in the volatile heap, compaction is done

// don't coalesce for now, but in long run will want to (with buddy system)
// keep pages separate (for later returning to the PC) and manageu7

inline Uint Volatile_heap::required_slots(Uint nslots) {
    th_assert(nslots <= Page_slots, "Object size requested too high");
    if (nslots < Min_slots) return Min_slots;
    return nslots;
}

Volatile_heap::Volatile_heap() {
    for (Uint i = 0; i < Flist_size; i++) flist[i] = NULL;
    large_flist =  NULL;
    free_space = 0;
}

Volatile_heap::~Volatile_heap() {
    // no work to do in the destructor: the persistent cache is responsible for
    // cleaning up for the memory XXX
}

Fields Volatile_heap::allocate(Uint nslots) {

    nslots = required_slots(nslots); // Get the actual slots to be allocated
    Slot* entry = allocate_entry(nslots);
    free_space -= nslots;
    if (entry) return (Fields) entry;

    // If compaction needs to be done, perform compaction and allocate slots
    if (free_space > Space_threshold) {
	compact();
	entry = allocate_entry(nslots);
	if (entry) return (Fields) entry;
    }

    // Try getting a page from the PC now
    entry = get_new_space(nslots);
    th_assert(entry, "Failed to allocate space in volatile heap");
    return (Fields)entry;
}

void Volatile_heap::free_object(Core o) {
    // For now, free tends to fragment memory: it just adds a new free list
    // block to the relevant free list, with no attempt to merge the block
    // into neighbouring blocks.

    Uint nslots = o->num_slots() + Fields_c::header_slots();
    nslots = required_slots(nslots);

    Freep entry = (Freep) o->get_fields();
    // Mark object length and that deleted when deleting
    th_assert(entry->handle() != Deleted_handle, "Object already free");
    insert_free_entry(entry, nslots);
}

void Volatile_heap::compact()  {

    // planned algorithm #0:
    // just merge adjacent free blocks ...
    // anything else requires testing using the ROT to reset the pointed at
    // object
    // in all cases rebuild the free list from scratch after compaction.

    // start by emptying out all the free lists
    for (Uint i = 0; i < Flist_size; i++)
	flist[i] = NULL;
    large_flist = NULL;

    for (int i = 0; i < alloc_pages.size(); i++) {
	Slot *ptr    = &(alloc_pages[i]->data[0]);
	Slot *endptr = &(alloc_pages[i]->data[Page_slots]);
	while (ptr < endptr) {
	    Freep entry = (Freep) ptr;
	    if (entry->handle() != Deleted_handle) {
		// Normal object: just skip it
		Core c = FE->rot->fetch(entry->handle());
		Uint nslots = c->num_slots();
		ptr += required_slots(nslots);
		continue;
	    }
	    // Garbage object: Find the total contiguous free area
	    ptr += entry->length();
	    // Keep scanning till a non-free handle is found or page ends
	    // We will keep the whole block in the freelists
	    while (ptr < endptr) {
		Freep subseq = (Freep) ptr;
		if (subseq->handle() != Deleted_handle)
		    break;
		ptr += subseq->length();
		assert(subseq->length() != 0);
		// setting the length for entry can trash the first word of
		// subseq (if the length of entry is large enough to require
		// using an extended length), so don't set the length of
		// entry until AFTER finished reading any values from subseq.
		entry->set_length(entry->length() + subseq->length());
	    }
	    // Add to the free list after determining the fully merged
	    // block's size.
	    insert_free_entry(entry, entry->length());
	}
    }

    // algorithm #1:
    // just optimize free space locally within pages, including free block
    // merges and moving all objects to the start of a page

    // algorithm #2 (more costly):
    // use a (cheap) approximate solution to bin packing, to best pack the
    // objects ... even just greedily stuffing objects into the first fit
    // bin
    // rebuild the free list from scratch after compaction.
}

Slot* Volatile_heap::split_object(Freep prev_ptr, Freep ptr, int length,
				 int nslots) {
    Uint left_slots = length - nslots;
    Freep new_entry = 0;
    if (left_slots) {
	// Some space has been left over. Just take the bottom part of the
	// object and place it in the front of the appropriate list
	th_assert(left_slots >= Min_slots, "Leftover space too small");
	new_entry   = (Freep)(((Slot *)ptr) + nslots); // Leftover space
    }

    // Fix prev_ptr->next to point to the right entry. That is, remove
    // ptr from the list it belonged to. Note: This takes care of an exact
    // nmatch also, i.e. left_slots == 0  

    Freep next_ptr = ptr->next.pointer();
    Uint orig_index = length - Min_slots;
    if (!prev_ptr && orig_index < Flist_size)
	flist[orig_index] = next_ptr;
    else if (!prev_ptr)
	large_flist = next_ptr;
    else
	prev_ptr->next = next_ptr;

    // Important: We first need to fix the existing ptr entry and then
    // insert new_entry into the freelist
    if (left_slots) insert_free_entry(new_entry, left_slots);

    return (Slot *) ptr;
}

Slot* Volatile_heap::allocate_entry(Uint nslots) {

    // Strategy: If nslots < Flist_size + Min_slots, we try to do a perfect fit.
    // If there are no free objects there, we try the large_flist and then
    // free lists starting from Flist_size down to the nslots. If nothing
    // is found, NULL is returned
    // If nslots >= Flist_size + Min_slots, "first-fit" is tried in the
    // large_flist. If it fails, returns NULL

    if (nslots < Flist_size + Min_slots) {
	int index = nslots - Min_slots;
	if (flist[index] != NULL) {
	    Freep entry = flist[index];
	    flist[index] = entry->next.pointer();
	    return (Slot *) entry;
	}
    }

    // Search the large_flist for free objects now
    Freep ptr = large_flist;
    Freep prev_ptr = NULL;
    for (; ptr != NULL; ptr = ptr->next.pointer()) {
	// Must ensure that there are at least Min_slots left after splitting
	// this space
	Uint length = ptr->length();
	if (length - Min_slots >= nslots || length == nslots) {
	    Slot* result = split_object(prev_ptr, ptr, length, nslots);
	    return result;
	}
	prev_ptr = ptr;
    }

    // Space not found in the large_flist also. Search in the various
    // list (from the high size end) till nslots + Min_slots size.

    for (Uint index = Flist_size - 1; index >= nslots; index--) {
	int entry_slots = index + Min_slots; // objects in list of this size
	if (flist[index]) {
	    Slot* result = split_object(NULL, flist[index], entry_slots, nslots);
	    return result;
	}
    }

    return NULL;
}

Slot* Volatile_heap::get_new_space(int nslots) {
    // Try to get space from the PC now

    Vpage *page = (Vpage*) FE->pc->borrow_page();
    if (!page) return NULL; // no room
    alloc_pages.append(page);
    free_space += Page_slots;

    // Allocate the object in the beginning and put the remaining in the
    // appropriate list

    Uint left_slots = Page_slots - nslots;
    th_assert(left_slots >= Min_slots, "Object size too big");
    Freep new_space = (Freep) (((Slot*) page) + nslots);
    insert_free_entry(new_space, left_slots);
    return (Slot*) page;
}

void Volatile_heap::insert_free_entry(Freep freeobj, Uint num_slots) {
    // requires: freeobj is num_slots slots long
    //           num_slots >= Min_slots
    // effects: Adds freeobj to the appropriate free list

    th_assert(num_slots >= Min_slots, "Too few slots for free object");
    freeobj->set_handle(Deleted_handle);
    freeobj->set_length(num_slots);

    // Put this object in the appropriate list now
    if (num_slots < Flist_size + Min_slots) {
	int index = num_slots - Min_slots;
	freeobj->next     = flist[index];
	flist[index]      = freeobj;
    } else {
	freeobj->next     = large_flist;
	large_flist       = freeobj;
    }
}


void *Alloc_class(unsigned size) {
  unsigned num_slots = (size + sizeof(Slot) - 1)/sizeof(Slot) + 1;
  /* Allocate an extra slot in case the result is not long-aligned */
  return (void*) NEXT_LONG((unsigned long)FE->vh->allocate(num_slots)); 
}

