#ifndef _Volatile_heap_h
#define _Volatile_heap_h

#include "compiler/C++wrappers/obj.h"
#include "compiler/C++wrappers/shortp.h"

#include "common/page.h"

#include "utils/bits.h"
#include "utils/array.h"

class Vpages;
class ostream;
class Flist_entry;
class Persistent_cache;

const Uint Flist_size = Page_slots/4; // Size of the freelist array

// The following defines a volatile page as Page_slots number of slots
typedef struct {
    Slot data[Page_slots];
} Vpage;

typedef Flist_entry *Freep;
declareArray(Vpages, Vpage*)

class Volatile_heap {
    //
    // Overview: Implements the volatile heap, which manages pages containing
    //   volatile objects.
    //

  public:
    //
    // Constructors:
    Volatile_heap();
    // Effects: Creates an initialized volatile heap

    ~Volatile_heap();
    // Effects: Destroys the volatile heap.

    Fields allocate(Uint nslots);
    // Effects: Allocates "nslots" slots for the fields of an object on the
    //          volatile heap ("nslots" must include space for fields header).
    //          Call fails if there isn't enough memory.

    void free_object(Core o);
    // Requires: "o" is the core of a non-discarded volatile object
    // Effects: Returns the space occupied by the fields of "o" to the
    //          heap.  Should be called by ROT when freeing volatile objects.

    void compact(void);
    // Effects: Compacts the space used by the volatile objects and returns 
    //           pages that become free to the persistent heap.

    // Debug functions
    void dump_slots(ostream o);
    void dump_chunks(ostream o);
    void dump_free_list(ostream o);

  protected:
    // Representation
    Freep flist[Flist_size]; // lists for different sizes
    // of free objects till a particular size. ith entry contains list for
    // objects of size i + Min_slots slots
    Freep large_flist; // Freelist for objects larger than
                         // Freelist_size + Min_slots
    int free_space; // Number of slots available in the free lists
    Vpages alloc_pages; // Pages allocated for volatile objects

    Slot* allocate_entry(Uint nslots);
    // requires: nslots >= Min_slots
    // effects: Allocates an object of size nslots from the appropriate
    //          list and returns it. If it is not able to allocate space,
    //          returns NULL

    Uint required_slots(Uint nslots);
    // Effects: Returns the actual number of slots needed to store nslots 
    //          (if nslots is too small, the number of allocated slots can be
    //          larger). Call fails if nslots is larger than the maximum
    //          object size that will fit in a page.

    Slot* split_object(Freep prev_ptr, Freep ptr, int length, int nslots);
    // modifies: list_start
    // requires: prev_ptr is NULL and ptr is list_start or
    //           prev_ptr->next == ptr. There is sufficient space available
    //           in ptr for allocating an object of size nslots
    //           length is the size of the block pointed by ptr
    // effects:  Splits the space in ptr to allocate space of size nslots
    //           slots from ptr to ptr + ptr->length. If any space is left, it
    //           is put in the appropriate free list
    //           Returns a pointer to the allocated object

    Slot* get_new_space(int nslots);
    // effects: Allocates an object of size nslots number of slots on
    //          a new page grabbed from the persistent cache

    void insert_free_entry(Freep freeobj, Uint num_slots);
    // requires: freeobj is num_slots slots long
    //           num_slots >= Min_slots
    // effects: Adds freeobj to the appropriate free list


};

#endif //  _Volatile_heap_h
