#ifndef Cplus_CORE_H
#define Cplus_CORE_H

#include "compiler/C/obj.h"
#include "compiler/C/dv.h"
#include "compiler/C/extern.h"

#include "shortp.h"
#include "fields.h"

#include "common/or_num.h"
#include "common/oref.h"
#include "common/obj_bitfield.h"
#include "common/xref.h"

#include "utils/basic.h"
#include "utils/array.h"

class Core_c {

  /* Overview: Wrapper class for struct Core_s defined in fe/compiler/C */

  public:
    Core_c() {};
    /* Default constructor should do nothing because we do not want to touch
       all memory in ROT when it is created. */
    
    inline Obj to_obj() const {return (Obj) this;}
    /* Returns Obj by adding proper offset. */
    /* Obsolete: Was useful when multiple dispatch vectors were supported */

    void set_fields(Fields f) {c.fields = FULL_AS_SHORTP(f);}
    /* Set the fields to f */

    Fields get_fields() const {return (Fields) SHORTP_AS_FULL_V(c.fields);}
    /* Convert Core to F */

    void set_dv(struct DV_hdr_s *m) {c.dv = FULL_AS_SHORTP(m);}
    /* Set the dv field to contain the DV m */

    struct Obj_DV * get_dv() const {return SHORTP_AS_FULL(struct Obj_DV, c.dv);}
    /* Returns the dv field.*/

    Shortp &methods_shortp() {return c.dv;}
    /* Returns the dv field as a Shortp rather than a pointer. */

    void zero_dv(void) {c.dv = FULL_AS_SHORTP(0);}
    /* Zero dv marking entry as non-free (used to consume rot[0]) */

    Oref &oref() {return c.oref_;}
    /* Returns the object's oref */
    
    struct Class_s * get_class() const {
    /*  Returns a class object that describes this. */
      return get_dv()->hdr.self_class;
    }

    Slot *slots() const {return get_fields()->slots();}
    /* Returns a pointer to the first slot in the object. The slot after
       the fields header */
        

    int num_slots() const {
    /* Returns number of slots in the fields portion of an object	
       excluding the fields header. */
      int nslots = get_dv()->hdr.num_slots;
      if (nslots >= 0) return nslots;
      /* Assumes that variable sized objects store the size in the
	 first slot of the fields and that they have a -1 num_slots
	 in the dv. Note that the size is stored in bytes, to support
	 strings efficiently. */
      return (int) (*slots() + 2*Slot_size - 1)/Slot_size;
    }

    Obj_bitfield bitfields() const {
    /* Returns bitfields indicating the pointer/data nature of slots.
       This code must match Obj_c::bitfields. It is replicated for 
       efficiency (slots()) */
      Obj_bitfield bf =  get_dv()->hdr.bf; 
      if (bf != OBJ_BF_VARBF) return bf;
      /* Assumes that objects with variable bitfields store the bitfield
	 in the second slot of the fields and that they have a OBJ_BF_VARBF
	 bitfield in the dv. For 64-bit slots, it is in the seond "half" of
	 the slot. */
      return *((Obj_bitfield *) ((char*)slots() + sizeof(int)));
    }

    inline int has_var_bitfields() const {
    /* If the object does not have variable bitfields returns 0
       Otherwise returns the byte number in the object fields where
       the object bitfields can be found. */
      return (get_dv()->hdr.bf == OBJ_BF_VARBF) ? sizeof(int) : 0; 
    }

    int hdr_size() const {return 1;} 
    /* Return the size of the core's dispatch header */
    /* Obsolete: Was used when multiple dispatch vectors were supported */

    bool is_discarded() const {return SHORTP_IS_DATA(c.fields);}
    /* Returns true iff object was discarded */

    void mark_discarded(OR_num orx) {c.fields = DATA_TO_SHORTP(orx);}
    /* Stores or in fields and marks object as discarded */
    
    OR_num or_discarded(void) const {return SHORTP_AS_DATA(OR_num, c.fields);}
    /* Requires entry is discarded object, returns the or number */

    bool has_xref(OR_num o, Oref oref_) const {
    /* Returns: true iff the object corresponding to this has xref=<o,oref>*/
      th_assert(o != 0, "Assumes or num 0 is invalid\n");
      th_assert(oref_ != 0, "Assumes oref 0 is invalid\n");
      if (c.oref_ != oref_) return false;
      return ((is_discarded()) ? or_discarded() : get_fields()->orx()) == o;
    }

    Xref xref(void) const {
    /* Return the xref of this */
	Xref tmp;
	tmp.oref = c.oref_;
	tmp.orx = (is_discarded()) ? or_discarded() : get_fields()->orx();
	return tmp;
    }

    bool is_persistent() const {
    /* Returns true iff object is persistent */
      th_assert(!is_free(), "Invalid this");
      return c.oref_ != 0;
    }
       
    bool is_free() const {return SHORTP_IS_DATA(c.dv);}
    /* Returns true iff the ROT entry containing this core is free */
    
    void mark_free() {
    /* Stores dh_words in fields and marks entry as free */
      c.dv = DATA_TO_SHORTP(hdr_size());
      c.oref_ = 0;   /* Assumes oref 0 is invalid */
    }

    unsigned int entry_size(void) const
    /* Requires core is free, returns the dh size it used to hold */
    { return SHORTP_AS_DATA(unsigned int, c.dv); }
    
    unsigned int ref_count() const { return c.rc8_next24 >> 24; }
    /* Returns reference count */
    
    void set_ref_count(unsigned int rc) {
    /* Sets the reference count to "rc" */
      c.rc8_next24 &= 0x00ffffff;
      c.rc8_next24 |= (rc << 24);
    }

    Core next() const { 
    /* Returns next pointer. */
      unsigned handle = c.rc8_next24 & 0x00ffffff;
      /* Null handle corresponds to null pointer. */
      return (handle != 0) ? (Core) ROT_FETCH(handle) : 0; 
    }
        
    void set_next(Core n) { 
    /* Sets next pointer to "n" */
      c.rc8_next24 &= 0xff000000;
      if (n != 0) c.rc8_next24 |= ROT_INDEX((Core_p)n);
    }
	
    void init() { 
      c.dv = DATA_TO_SHORTP(1);
      c.oref_ = 0;   /* Assumes oref 0 is invalid */
      c.fields = DATA_TO_SHORTP(0);
      c.rc8_next24 = 0;
    }

 
  private:
    struct Core_s c;
};

declareArray(Cores, Core)

#endif /* Cplus_CORE_H */
