#ifndef Cplus_FIELDS_H
#define Cplus_FIELDS_H

#include "compiler/C/fields.h"
#include "compiler/C/extern.h"

#include "typedefs.h"

#include "common/slot.h"
#include "common/or_num.h"
#include "common/page.h"

/* Overview: Wrapper class for struct Field_s defined in fe/compiler/C 

   The "handle" in a Field_s structure is the index of the ROT entry
   corresponding to this object. It provides an efficient way of
   accessing the ROT entry of an object starting from its Fields_s
   structure, which is important for compaction/eviction.
   
   The stamp contains the following bits: "read", "write", "pointer
   update" and "swizzled after copied". read is set if the object was
   read or written in the current transaction. write is set if the
   object was written in the current transaction. pointer update is
   set if the object was written during the current transaction and
   one of its pointers was modified. "swizzled after copied" is set if
   the object was written during the current transaction and one of
   its pointers was swizzled after its copy was made. 
   
   "usage" reflects the recent usage of the object. It is set by the
   method dispatch code at each invocation.  Having the lru stamp in
   the Fields_s structure is also important because it avoids having
   to look at the ROT entry when computing frame usage. 
   Max_usage is the maximum possible usage value: */

const unsigned Max_usage = 15;
const unsigned Max_usage_bits = 4;
const unsigned Max_decayed_usage = (Max_usage + 1) >> 1;
/* Maximum value of usage after it is decayed in a scan.
   Must be kept in sunc with the code that decays usage 
   which is currently in Persistent_cache::compute_page_usage. */

class Fields_c {
  public:
    void init(unsigned int handle) { f.hus = handle << 8; }
    /* Initializes fields setting its handle and marking the object as 
       installed. */

    void set_handle(unsigned int handle) {
    /* Sets the handle. */
        f.hus &= 0x000000ff; f.hus |= handle << 8; 
    }

    unsigned int get_handle(void) const { return f.hus >> 8;}
    /* Returns the handle */

    Core to_core() const { return (Core) ROT_FETCH(get_handle()); }
    /* Return core that points to this fields (requires fields is valid) */
    
    inline Slot * slots() const {return (Slot *)((Fields)this + 1);}
    /* Returns pointer to the first slot (pointer or data). */

    inline void stamp_read() { f.hus |= READ_MASK; }
    /* Stamp object as read. */

    inline void unstamp_read() {f.hus &= ~READ_MASK;}
    /* Stamp object as not read. */

    inline bool is_read() const {return IS_READ(&f);}
    /* Returns whether object is read. */

    inline void stamp_written() {f.hus |= WRITTEN_MASK;}
    /* Stamp object as written. */

    inline void unstamp_written() {f.hus &= ~WRITTEN_MASK;}
    /* Stamp object as not written. */

    inline bool is_written() const {return IS_WRITTEN(&f);}
    /* Returns whether object is written. */

    inline void stamp_pointer_update() {f.hus |= POINTER_UPDATE_MASK;}
    /* Stamp object's pointer update bit */

    inline void unstamp_pointer_update() {f.hus &= ~POINTER_UPDATE_MASK;}
    /* Stamp object as not containing a pointer update */

    inline bool has_pointer_updates() const {return f.hus & POINTER_UPDATE_MASK;}
    /* Returns whether object has had any pointer updates */

    inline void stamp_eager_ref_counting() {f.hus |= EAGER_REF_COUNTING_MASK;}
    /* Stamp object to indicate that reference counts should be updated eagerly
       whenever one of its pointers is modified. */

    inline void unstamp_eager_ref_counting() {f.hus &= ~EAGER_REF_COUNTING_MASK;}
    /* Unstamp object to indicate that reference counts for the objects it 
       refers to should be updated lazily.*/

    inline bool use_eager_ref_counting() const {
      /* Returns whether reference counts should be updated eagerly when 
	 a pointer in the object is modified. */
      return f.hus & EAGER_REF_COUNTING_MASK;
    }

    inline void unstamp_all() {
      /* Clears read, write, pointer updates and eager ref counting bits. */
      f.hus &= ~(READ_MASK | WRITTEN_MASK | POINTER_UPDATE_MASK 
		 | EAGER_REF_COUNTING_MASK);
    }
  
    inline unsigned int get_usage() const {return f.hus & USAGE_MASK; } 
    /* Returns usage information */

    inline void set_usage(unsigned int u) {
      /* Sets usage to be u */
      f.hus = (f.hus & ~USAGE_MASK) | u;
    } 
   
    Page *page() const { return ((Page *) ((long)this & ~(Page_size-1)));}
    /* Return the objects's page. */

    OR_num orx() const { return page()->get_or();}
    /* Return object's OR (Page_size should be a power of 2). */

    Ubits64 vnum() const { return page()->get_vnum();}
    /* Return object's OR (Page_size should be a power of 2).
       Used only for orphan detection experiments */

    inline void mark_invalid() { f.hus = 0;}
    /* Mark the object as invalid */

    inline bool is_invalid() const {  return f.hus == 0; }
    /* Return true iff the object is invalid */

    static unsigned header_slots(void) {return 1;}
    /* Return the number of slots in a Fields header */

  private:
    Fields_s f;
    /* Slot fields[]; */           /* rest of object fields */
};

#endif /* Cplus_FIELDS_H */
