#ifndef _C_RESIDENCY_H
#define _C_RESIDENCY_H

#include "obj.h"
#include "extern.h"

/*
    Object states
    -------------

    Abstractly, an object may be in one of four residency states, each of
    which is more specific than the previous ones.

        UNKNOWN: In this state, the object pointer is not even known to
           be swizzled, and nothing is known about whether the object is
           resident in the cache.
        DISCOVERED/SWIZZLED: The object pointer points to a valid ROT
           entry, but the fields of the object are not known to be resident
        RESIDENT: The fields of the object are resident, but their actual
           location is not known and must be fetched from the ROT entry.
        FIXED: An object is fixed by a (Fields *) variable, if the fields of
           the object are located at the location the variable points to.
    
    During execution, certain invariants are maintained. An object
    pointer, once discovered, remains discovered forever.  An object
    that is RESIDENT or FIXED remains RESIDENT or FIXED while there is
    a pointer in the stack or in a register to the object's fields.

    The variable "self" is guaranteed to be RESIDENT on entry to a method.
    All other arguments are guaranteed to be DISCOVERED.

        statement RESIDENT_SHORTP(T &result, struct OC##_f_s *v, field name F,
                                  class OC, type FT, typeobj FTO)
           Discover the field with name "F" and type "FT" in the Fields "v" of 
           a class "OC". Place the value of the discovered pointer in "result".
           The object pointer returned becomes swizzled but the object
           it refers to does not necessarily become resident. 

        T_f RESIDENT_CORE(T c, type T);
            Make the core "c" RESIDENT. Return a fields pointer that
            currently fixes "c". "c" should be a variable for best
            performance.

        T_f FIX_FIELDS(T c, type T);
            "c" must be a resident core. Returns a fields pointer
            that currently fixes "c".
        
        statement RESIDENT_OBJ(T o);
            Make the object "o" RESIDENT.  */

#define RESIDENT_SHORTP(R,V,FN,OC,FT,FTO)                                     \
    _BEGIN_                                                                   \
        struct OC##_f_s* MT1 = V;                                             \
        Shortp MT2 = MT1->FN;                                                 \
        if (SHORTP_IS_DATA(MT2)) {                                            \
            R = (FT)Cache_swizzle((Fields_p)MT1,MT2,FTO);                     \
            MT1->FN = FULL_AS_SHORTP(R);                                      \
        } else {                                                              \
            R = (FT)SHORTP_AS_FULL_V(MT2);                                    \
        }                                                                     \
    _END_                                                                     


#define FIX_FIELDS(C,T) (struct T##_f_s *)GET_FIELDS(C)

#define CORE_IS_RESIDENT(C) (!(((Core_p)(C))->fields & 0x1))

#define RESIDENT_CORE(C,T)                                                    \
     ((CORE_IS_RESIDENT(C)) ? FIX_FIELDS(C,T) :                               \
         (struct T##_f_s *)Make_resident((Core_p)(C)))
                                        
#define RESIDENT_OBJ(O)                                                       \
     ((CORE_IS_RESIDENT(OBJ_CORE(O))) ? 0 : Make_resident(OBJ_CORE(O)))

#endif /* _C_RESIDENCY_H */
