#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stream.h>

#include "config/vdefs/SHM.h"
#include "config/vdefs/FE_DEFAULT_CACHE_SIZE.h"
#include "config/vdefs/COLLECT_EXECUTION_TRACE.h"
#include "config/vdefs/OPTIMAL_PAGE_REPLACEMENT.h"
#include "config/vdefs/LRU_PAGE_REPLACEMENT.h"
#include "utils/fail.h"
#include "utils/communication.h"
#include "utils/hostname.h"
#include "utils/other_unix.h"
#include "utils/th_assert.h"
#include "common/client_fe.h"
#include "common/locator.h"
#include "common/page.h"

#include "boot/type_init.h"
#include "fe/main/fe_config.h"

#include "cache1/resident_object_table.h"
#include "cache1/or_recv_data_msg.h"
#include "cache1/persistent_cache.h"
#include "cache1/volatile_heap.h"
#include "cache1/surrogate_table.h"

#include "runtime/stats.h"
#include "runtime/tm.h"
#include "runtime/or_info.h"
#include "runtime/or_recv_message.h"
#include "runtime/handle.h"

static char usage[] = 
"Usage: %s [options below]\n"
"-s<cachesize>    cachesize in Kbytes\n" 
"-b<avg_obj_size> average objectsize in bytes\n"
"-e               Use eviction instead of compaction to manage cache\n" 
"-r<num>          Number of scans in last scanned\n"
"-f<num>          Number of pages scanned per evicted page\n"
"-d<num>          Discards Page_size/num bytes per page\n" 
"-n<num>          Number of scan pointers\n"
"-p<FE_port>      number of port where FE will listen for clients\n"
"-o<or_num>       initial or number to connect to [1]\n"
"-a               run in background\n"
"-c<Theta file>   compile early-Theta file\n"
"-C<Theta file>   compile Theta file\n"
"-v<Theta file>   create veneer interface\n"
"-m<number>       size of fetch group in pages\n"
"-D<debug_level>  Print debugging info\n"
"-i               Allow SIGIO to be delivered to FE\n"
"-l<language>	  language to use for veneer interface\n"
"                 'C-binary', 'C++' and 'Erlang'"
;

FE_config::FE_config (int argc, char* argv[]) {

  // First initialize with default values.
  // client interface 

  // Note: All the entities allocated here are being deleted in the
  //       destructor

  client_port = 0;
  slave = TRUE;
  shm = FALSE;
  shm_bufsize = 256;
  
  // or interface 
  initial_or_num = 1;
  locator = new Locator();
  comm = new Communication();
  fe_stats = new FE_stats;

  // cache 
  cachesize = FE_DEFAULT_CACHE_SIZE;

  max_prefetch = 1;      // No adaptive prefetching by default.
  use_eviction = false;
  background_replacement = true;
  min_free_pages = Max_pages_in_seg;
  scans_remembered = 20;
  low_threshold = 0.3;
  high_threshold = 0.8;
  scan_factor = 3;
  num_scan_ptrs = 3;
  discard_factor = 3;
  avg_obj = 30;
  per_undo_log_size = 128*1024; // in slots
  or_map = new OR_info_map;
  surr_table = new Surrogate_table;
  use_full_scan = false;
  max_rot_waste = 1;

  trace_file = 0;

  // misc
  runInBackground = FALSE;
  debug_level = 0;
  allow_sigio = FALSE;

  // Message handlers
  OR_recv_invalidation_msg *inv_msg = new OR_recv_invalidation_msg;
  OR_recv_root_msg *root_msg = new OR_recv_root_msg;
  OR_recv_commit_reply_msg *commit_reply_msg = new OR_recv_commit_reply_msg;
  OR_recv_alloc_reply_msg *alloc_reply_msg = new OR_recv_alloc_reply_msg;
  OR_recv_debug_reply_msg *debug_reply_msg = new OR_recv_debug_reply_msg;
  OR_recv_data_msg* data_reply_msg = new OR_recv_data_msg;

  comm->register_handler(inv_msg);
  comm->register_handler(root_msg);
  comm->register_handler(commit_reply_msg);
  comm->register_handler(data_reply_msg);
  comm->register_handler(alloc_reply_msg);
  comm->register_handler(debug_reply_msg);

  // compiler 
  srcfile = 0;
  theta_srcfile = 0;
  veneer_srcfile = 0;
  veneer_language = 0;

  process_options(argc, argv);

  cerr << "FE configuration: ";
  cerr << "\n Cache size (KB): " << cachesize/1024 
       << "\n Fixed fetch group size (0 means adaptive): " << max_prefetch
       << "\n Use Eviction: " << (int)use_eviction 
       << "\n Number of scans remembered: " << scans_remembered 
       << "\n Adaptive prefetching threshold for empty cache: " 
       << low_threshold 
       << "\n Adaptive prefetching threshold for full cache: "
       << high_threshold
       << "\n Scan factor: " << scan_factor
       << "\n Discard factor: " << discard_factor 
       << "\n Number of scan pointers: " << num_scan_ptrs
       << "\n Use full scan: " << ((use_full_scan) ? "true" : "false") 
       << "\n Predicted average object size: " << avg_obj 
       << "\n Page size: " << Page_size 
       << "\n Optimal page replacement: " 
       << ((OPTIMAL_PAGE_REPLACEMENT) ? "true" : "false") << "\n\n";
}

FE_config::~FE_config () {
    delete locator;
    delete comm;
    delete tm;
    delete or_map;
    delete surr_table;
    delete rot;
    delete vh;
    delete pc;

    if (trace_file) fclose(trace_file);
}

void FE_config::process_options(int argc, char* argv[]) {

  /* Have to reset "optind", which may have been set by using "getopt" for
     the client options.  This will in turn cause problems if something
     tries processing FE options before this code is called.  This is a
     good example of how global state can get you into trouble.  */

  optind = 1;

  while (1) {
      int opt;
      if ((opt = getopt(argc, argv, "s:b:o:jiap:c:C:v:l:m:SD:er:f:d:t:T:n:FE:Ru:w:")) != EOF) {
      switch (opt) {
	  int length;
	  char *fname;
      case 's':
	cachesize = atoi(optarg);
	if (cachesize > 0)
	  cachesize = (cachesize * 1024);
	else {
	  fail("Couldn't interpret %s as a cache size", optarg);
	}
	break;
      case 'b':
	avg_obj = atoi(optarg);
	if (avg_obj <=  0)
	  fail("Couldn't interpret %s as an object size", optarg);
	break;
      case 'o':
	initial_or_num = atoi(optarg);
	if (initial_or_num <= 0) 
	    fail("Bad initial or number ", optarg);
	break;
      case 'p':
	client_port = atoi(optarg);
	slave = FALSE;
	if (client_port <= 0) {
	  fail("Couldn't interpret %s as a port number", optarg);
	}
	break;
      case 'a':
	runInBackground = TRUE;
	break;
      case 'i':
	allow_sigio = TRUE;
	break;
      case 'c':
	length = strlen(optarg);
	fname = new char[length + 1];
	strcpy(fname, optarg);
        srcfile = fname;
        break;
      case 'C':
	length = strlen(optarg);
	fname = new char[length + 1];
	strcpy(fname, optarg);
        theta_srcfile = fname;
        break;
      case 'E':
	trace_file = fopen(optarg, ((OPTIMAL_PAGE_REPLACEMENT 
				     || LRU_PAGE_REPLACEMENT) ? "r" : "w"));
	break;
      case 'v':
	length = strlen(optarg);
	fname = new char[length + 1];
	strcpy(fname, optarg);
        veneer_srcfile = fname;
        break;
      case 'l':
        veneer_language = new char[strlen(optarg)];
	strcpy(veneer_language, optarg);
        break;
      case 'm':
      {
        int number = atoi(optarg);
        if (number >= 0) {
          max_prefetch = number;
          break;
        } 
        else {
          fail("Couldn't interpret %s as a non-negative integer", optarg);
        }
      }
      case 'S':
#if !SHM
	fail("Not configured to use shared memory");
#else
	shm = TRUE;
#endif
	break;
      case 'D':
	debug_level = atoi(optarg);
        break;
      case 'e':
	use_eviction = TRUE;
        discard_factor = 1;
	break;
      case 'r':
	scans_remembered = atoi(optarg);
        break;
      case 'f':
	scan_factor = atoi(optarg);
        break;
      case 'F':
	use_full_scan = true;
        break;
      case 'n':
	num_scan_ptrs = atoi(optarg);
        break;
      case 'd':
	discard_factor = atoi(optarg);
        break;
      case 't':
	low_threshold = atof(optarg);
        break;
      case 'T':
	high_threshold = atof(optarg);
        break;
      case 'R':
	background_replacement = false;
        break;
      case 'u':
        per_undo_log_size = atoi(optarg);
        if (per_undo_log_size <  0)
          fail("Couldn't interpret %s as an undo log size", optarg);
        per_undo_log_size = per_undo_log_size*1024/sizeof(Slot);
        break;
      case 'w':
        max_rot_waste = atoi(optarg);
        if (max_rot_waste <  0)
          fail("Couldn't interpret %s as a percentage", optarg);
        break;

      default:
	fail(usage, argv[0]);
      }
    }
    else
	break;
  }

  // Check for dependencies between options
  if ((COLLECT_EXECUTION_TRACE || OPTIMAL_PAGE_REPLACEMENT 
       || LRU_PAGE_REPLACEMENT) && !trace_file)
    fail("Need to use -E option to specify trace file");
  
  if (OPTIMAL_PAGE_REPLACEMENT || LRU_PAGE_REPLACEMENT) {
    // Optimal page replacement requires eviction and no adaptive prefetching.
    use_eviction = true;
    max_prefetch = 1;
  }

  // If doing adaptive prefetching always keep a free segment in the
  // cache. Otherwise, 
  min_free_pages = (max_prefetch == 0) ? Max_pages_in_seg : max_prefetch;
}

void FE_config::init() {
    
    // exceptions
    // Initialize_exceptions();

    tm = new TM();

    // cache
    Storage_arena *sa_pc = new Storage_arena(2*cachesize/sizeof(Page));
    Storage_arena *sa_rot = sa_pc->split(cachesize/sizeof(Page));

    // Assumes 90% of the pc is occupied by persistent objects.  
    if ((rot = new Resident_object_table(cachesize*9/10, avg_obj, sa_rot)) == 0)
      fail("ROT creation failed");
    if ((pc = new Persistent_cache(sa_pc)) == 0)
      fail("PC creation failed.");
    if ((vh = new Volatile_heap()) == 0) 
      fail("VH creation failed.");

    // Initialize types.
    cout << "Initing builtins...\n";
    Init_builtin_types();
    cout << "Inited builtins\n";


    // connect to the initial OR
    Address or_address = OR_address(initial_or_num);
    Ubits32 or_hostid; int or_port;
    if (!locator->lookup(or_address, &or_hostid, &or_port))
	fail("OR Address not found in locator");

    bool success;
    Network *net = new Network(or_address, or_hostid, or_port, success);
    th_assert(net && success, "Could not connect to the OR");
    net->set_nodelay();
    // insert network in comm
    comm->include(net);
    // get address from OR
    if (!(Address::Error_address).encode(net)) fail("failed requesting addr");
    net->flush();
    if (!address.decode(net)) fail("failed receiving address");

    // Add the OR to the or_map
    or_map->add_or(initial_or_num);
    address.print();
    fprintf(stderr, "\n");
    if (!Get_or_stats(initial_or_num, fe_stats->prev)) 
      th_fail("Faled to get the stats message from the OR\n");

}
