#ifndef _COMMIT_INFO_H
#define _COMMIT_INFO_H

// This object keeps track of information used ONLY during commit processing.
// Only the TM code should have knowledge about this object

#include "compiler/C++wrappers/core.h"

#include "common/xref.h"
#include "common/or_num.h"
#include "common/or_set.h"

#include "utils/basic.h"
#include "utils/map.h"
#include "utils/valuekey.h"

class Transaction;
class OR_set;
class Trans_log;
class Modset;
class Surrogate_table;

// Information maintained per OR
struct OR_commit_data {
    Transaction *trans; // Transaction object to be sent to this OR
};

// Map: Cores to new xrefs (for newly persistent objects)
typedef PtrKey<Core_c> Core_key;
typedef Map<Core_key, Xref> New_object_table;
// Map from Or numbers to information maintained per OR
typedef Map<UIntKey, OR_commit_data>  OR_commit_set;


class Commit_info {
  public:
    Commit_info();
    // effects: Creates an initialized object.

    ~Commit_info();

    Transaction* get_transaction(OR_num orx, bool create);
    // effects: Returns the transaction object corresponding to the OR or
    //          If none exists and create is TRUE, creates one for it
    //          (else returns NULL)

    OR_set const *participants() {return parts;}
    // effects: Returns the participants for this commit

    bool is_read_only() { return read_only;}
    // effects: Returns TRUE if the trans being committed is read_only

    bool make_transaction(Trans_log *tlog);
    // modifies: this
    // effects:	 Scans the objects in tlog and generates complete transaction
    //           objects for servers from which objects have been accessed.
    //           Guarantees that the Ros/Mos of all the participants have been
    //           placed in increasing order of Orefs (i.e. sorted)
    //		 Note: This method may send messages to the servers and wait
    //           for their replies (for allocation requests)
    //		 Returns true iff the transaction modified/read any
    //           persistent objs.

    void process_before_commit_reply();
    // requires: make_transaction has been called on this
    //           This MUST be called before the destructor gets called
    // effects: Cleans up data structures that are not needed after commit reply

    void commit_reply_processing(bool committed);
    // requires: committed is TRUE iff transaction committed aborted (reply
    //           has been received)
    // effects:  If the transaction aborted, frees the xrefs and space that
    //           was allocated for new objects. Else the new xrefs/space is
    //           permanently allocated to those objects


    // An iterator for going over the newly persistent objects
    // requires: newly persistent objects are not addede/deleted while the
    //           iterator is being used
    class New_persistent_iter {
      public:
	New_persistent_iter(Commit_info* info);
	// effects: Creates an initialized iterator. The caller MUST
	//          call destroy when it gets done with the usage

	bool get(Core& c, Xref& x);
	// modifies "c", "x"
	// effects  Sets "c" to the next core, x to its xref and returns true.
	//          Returns false if there are no more objects (and does not
	//          modify "c", "x")
	
	void destroy();
	// require: this not beused after this call
	// effects: destroys any space allocated for this iterator.


      private:
	MapGenerator<Core_key, Xref> gen;
    };

  private:
    friend New_persistent_iter;

    bool read_only;           // If the current transaction is read only
    OR_commit_set    *or_set; // Information maintained per OR
    New_object_table *new_table; // Keeps track of a mapping from new object
                              // cores to their newly allocated xrefs
    OR_set *parts;            // Participants for this transaction
    int ros_mos_predict;      // Size hint for the total size of ROS and MOS
    Surrogate_table *new_surrs; // New surrogates created in this transaction

    Transaction *last_trans;  // Caches last transaction object.
    //
    // Internal Methods
    //

    Transaction* slow_get_transaction(OR_num orx, bool create);
    // requires: No entry for "or" is present in this.
    // effects:  Same as get_transaction

    void recursive_unswizzle(Core c, Fields copy, Xref xref, Modset *mset);
    // requires: last_trans contain the transaction object for xref.or 
    //           xref is the xref of c
    //           mset is the mos (for xref.or) if c is a persistent object
    //           mset is the nos (for xref.or) if c is a volatile object
    // effects:  Unswizzles c, adds it to mset and adds volatile
    //           objects reachable from c to the nos of xref.or
    //           Recursively unswizzles volatile objects reachable from c
    //           till all these objects have been added to the nos of xref.or

    Oref get_oref(OR_num dest_or, Oref dest_oref, OR_num orx);
    // requires: last_trans to point to the transaction object for "or"
    // effects: Returns an oref at or "or" for the object <dest_or, dest_oref>
    //          If required, an OR surrogate is created and added to the
    //          NOS of "or"'s transaction object

    Oref get_surr_oref(OR_num dest_or, Oref dest_oref, OR_num orx);
    // requires: dest_or != or + requires of get_oref
    // effects: Returns an oref at or "or" for the object <dest_or, dest_oref>
    //          If required, an OR surrogate is created and added to the
    //          NOS of "or"'s transaction object

    OR_num get_coord_participants(OR_set* parts);
    // requires: space has been allocated for parts
    // modifies: parts
    // effects: Determines the set of participants for this transaction
    //          and puts them in parts. Returns the chosen coordinator for
    //          this transaction 

};

inline Transaction* Commit_info::get_transaction(OR_num orx, bool create) {
    OR_commit_data info;
    bool found = or_set->find(orx, info);
    if (found) return info.trans;
    return slow_get_transaction(orx, create);
}

inline Commit_info::New_persistent_iter::New_persistent_iter(Commit_info *cinfo)
  : gen(*cinfo->new_table) {
}

inline bool Commit_info::New_persistent_iter::get(Core& c, Xref& xref) {
    Core_key key;
    if (gen.get(key, xref)) {
	c = key.val;
	return TRUE;
    }
    return FALSE;
}

inline void Commit_info::New_persistent_iter::destroy() {
}

#endif /* _COMMIT_INFO_H */
