#include "or_info.h"

#include "cache1/persistent_cache.h"

#include "fe/main/fe_config.h"

#include "common/page.h"
#include "common/xref.h"

#include "utils/intarray.h"
#include "utils/fail.h"
#include "utils/mdebug.h"

//
// Types used by the PC maps:
//
// Info the PC needs about objects that have been allocated in current trans.
class Alloc_info {
  public:
    Uint          used_pageid;	// Page in which objs have been allocated.
    Page_state    saved_state;	// The saved page state.
};

declareArray(Alloc_set, Alloc_info)
implementArray(Alloc_set, Alloc_info)

// Rep invariant:
//   If any page is cached for this OR then current_page and current_page_id
//   corresponds to a cached page.
//   

// Remove a page from being consider as a valid page if it has less
// space available than the following constant
// const int min_alloc_threshold = 2*Page_size/Max_objs_in_page;
const int min_alloc_threshold = 32;

inline bool OR_info::allocatable_page(Page* p) {
    return p->bytes_available() >= min_alloc_threshold;
}

void OR_info::add_alloc_rights(Uint page_id, Page* p) {
    UIntElem pkey(page_id);
    if (allocatable_page(p)) {
	if (!avail->contains(page_id)) avail->add1(page_id);
    } else {
	avail->remove(pkey);
    }
}

// Define maps and sets

#define SET_ELEM UIntElem
#include "utils/impl_set.t"

#define GENERATOR_ELEMENT UIntElem
#include "utils/impl_generator.t"

#define IMPL_KEY   UIntKey
#define IMPL_VALUE Seg_info *
#include "utils/impl_map.t"

// Instantiate the Hash table for mapping or numbers to OR_info
#define IMPL_KEY  UIntKey
#define IMPL_VALUE OR_info *
#include "utils/impl_map.t"

OR_info::OR_info (OR_num or_num) {
    last_invalidation = 0;
    last_inv_acked    = 0;
    ornum             = or_num;
    alloc_set         = new Alloc_set(1);
    current_page      = NULL;
    current_page_id   = 0;
    segment_map       = new Seg_map;
    avail             = new Page_set;
}

OR_info::~OR_info () {
    // Delete the entries in the segment_map
    delete alloc_set;
    MapGenerator<UIntKey, Seg_info *> gseg(*segment_map);
    UIntKey segkey;
    Seg_info *si;
    while (gseg.get(segkey, si)) delete si;
    delete segment_map;
    delete avail;
}

Seg_info * OR_info::lookup_seg_info(Uint segid) {    
     Seg_info *seg_info;
     if (segment_map->find(segid, seg_info)) return seg_info;
     // New segment.
     seg_info = new Seg_info;
     seg_info->cached_pages = Seg_bmap(FALSE);
     seg_info->alloc_rights = Seg_bmap(FALSE);
     seg_info->empty = Seg_bmap(FALSE);
     segment_map->store1(segid, seg_info);
     return seg_info;
}

Oref OR_info::allocate(Page *p, Uint pid, Uint num_slots) {
    int onum = p->new_onum();
    if (onum < 0 || p->allocate(onum, num_slots) == NULL) return NULL_OREF;

    Page_state saved_state   = p->get_current_state();
    current_page    = p;
    current_page_id = pid;
    FE->pc->lock_page(p);

    if (alloc_set->size() > 0 && alloc_set->high().used_pageid == pid) {
      // Page is already in alloc_set.
	Oref oref = Oref_create(pid, onum);
	return oref;
    }
 
    // A new page is being used for allocation, enter in alloc_set
    // Note: If a page is reused in allocation (same trans) in a non
    // sequential manner, there can be duplicate entries for that page
    Alloc_info painfo;
    painfo.used_pageid   = pid;
    painfo.saved_state = saved_state;
    alloc_set->append(painfo);

    add_alloc_rights(pid, p);
    Oref oref = Oref_create(pid, onum);
    return oref;
}    


Page * OR_info::get_uncached_page(Uint &pid) {
    // Try all segments and find the first unused page on a segment
    MapGenerator<UIntKey, Seg_info *> g(*segment_map);
    UIntKey segkey;
    Seg_info *seg_info;
    while (g.get(segkey,seg_info)) {
	// Find first uncached, allocatable, empty page in this segment.
	Seg_bmap pages = seg_info->alloc_rights & (~seg_info->cached_pages)
	    & seg_info->empty;

	if (pages.is_all_false()) {
	  // No bits set.
	  if (seg_info->cached_pages.is_all_false())
	    g.remove();
	  continue; 
	}
	Seg_bmap::Iter iter(&pages, TRUE);
	Uint pnum;
	while (iter.get(pnum)) {
	    // Found uncached page pnum in the segment that is empty
	    pid = Pageid_from_segid(segkey.val, pnum);
	    Page *page = FE->pc->alloc_and_lock_page(ornum, pid);
	    th_assert(page, "PC could not allocate page for or_info_map");
	    seg_info->cached_pages.set(pnum);
	    return page;
	}
    }

    return NULL;
}

Oref OR_info::get_new_oref(int num_slots) {
    // Look in currrent allocation page for the OR.
    // If no room there, find a cached page and try there.
    // If no room there, try next cached page.
    // If no room in any cached page, allocate and use an uncached page.

    // First try current page.
    Page *page  = current_page;
    Uint pid    = current_page_id;
    Oref oref;

    if ((page != NULL) && FE->pc->valid_page(ornum, pid, page)) {
	oref = allocate(page, pid, num_slots);
	if (oref != NULL_OREF)
	    return oref;
    }

    // Go to the next pages in this segment and try to allocate objects there
    Uint cur_seg = Segid_from_pageid(pid);
    Seg_info *cur_sinfo = NULL;
    if (segment_map->find(cur_seg, cur_sinfo)) {
	// Check pages that have alloc rights and are empty or cached
	Seg_bmap smap = cur_sinfo->alloc_rights &
	    (cur_sinfo->cached_pages | cur_sinfo->empty);
	Seg_bmap::Iter iter(&smap, TRUE);
	Uint cur_pnum = Pagenum_from_pageid(pid);
	Uint pnum;
	while (iter.get(pnum)) {
	    if (pnum <= cur_pnum) continue;

	    Uint page_id = Pageid_from_segid(cur_seg, pnum);
	    if (!cur_sinfo->cached_pages.test(pnum)) {
		// The page must be empty
		th_assert(cur_sinfo->empty.test(pnum),
			  "Uncached page for all must be empty");
		page = FE->pc->alloc_and_lock_page(ornum, page_id);
		cur_sinfo->cached_pages.set(pnum);
	    } else {
		page = FE->pc->lookup_page(ornum, page_id);
		th_assert(page && FE->pc->valid_page(ornum, page_id, page), 
			  "Page should be present in cache");
	    }
	    if ((oref= allocate(page, page_id, num_slots)) != NULL_OREF) {
		return oref;
	    }
	}
    }

    // Go through all the available pages and check if they are cached and
    // space is available

    SetGenerator<UIntElem> pgen(*avail);
    UIntElem pkey;
    while (pgen.get(pkey)) {
	pid = pkey.val;
	Uint segid = Segid_from_pageid(pid);
	Seg_info *seg_info = NULL;
	// If the page is no longer cached or there are no alloc  rights for
	// that page, remove it from the available list

	Seg_bmap smap(FALSE);
	Uint pnum = Pagenum_from_pageid(pid);
	if (segment_map->find(segid, seg_info)) {
	    smap = seg_info->cached_pages & seg_info->alloc_rights;
	    if (smap.test(pnum)) {
		// Page is present and has rights to it
		// Check if space can be allocated on it
		page = FE->pc->lookup_page(ornum, pid);
		th_assert(page != NULL, "Page marked as cached absent in PC.");
		if ((oref= allocate(page, pid, num_slots)) != NULL_OREF) {
		    return oref;
		}
		// If it is still a good page, keep it
		if (allocatable_page(page)) continue;
	    }
	}
	// The page is either uncached or not good enough; remove it from avail
	th_assert(!seg_info || !smap.test(pnum) || !allocatable_page(page),
		  "Trying to remove a good page");
	pgen.remove();
    }

//     MapGenerator<UIntKey, Seg_info *> g(*segment_map);
//     UIntKey segkey;
//     Seg_info *seg_info;
//     while (g.get(segkey, seg_info)) {
// 	Seg_bmap calloc_pages= seg_info->cached_pages & seg_info->alloc_rights;
// 	Seg_bmap::Iter iter(&calloc_pages, TRUE);

// 	Uint pnum;
// 	// Look at all the pages in this segment and allocate an object
// 	// on any page that is cached and we have allocation rights to
// 	while (iter.get(pnum)) {
// 	    pid = Pageid_from_segid(segkey.val, pnum);
// 	    page = FE->pc->lookup_page(ornum, pid);
// 	    th_assert(page != NULL, "Page marked as cached absent in PC.");
// 	    if ((oref= allocate(page, pid, num_slots)) != NULL_OREF) {
// 		return oref;
// 	    }
// 	}
//     }

    // No room in cached pages, allocate an uncached page.
    page = get_uncached_page(pid);
    if (!page) return NULL_OREF;
    oref = allocate(page, pid, num_slots);
    return oref;
}

inline void OR_info::commit_alloc_set() {
    Seg_info* seg_info = NULL;
    Uint last_seg = 0;

    // Keep marking the earlier empty pages as non-empty if objects
    // have been allocated on them
    for (int i = 0 ; i < alloc_set->size(); i++) {
	Alloc_info *inf = &alloc_set->slot(i);
	FE->pc->unlock_page(ornum, inf->used_pageid);
	Uint segid = Segid_from_pageid(inf->used_pageid);
	if (!seg_info || segid != last_seg) {
	    seg_info = segment_map->fetch(segid);
	    last_seg = segid;
	}
	Uint pnum = Pagenum_from_pageid(inf->used_pageid);
	// Mark the non_empty pages as non_empty
	seg_info->empty.reset(pnum);
    }
    // Discard the entries in the alloc_map
    alloc_set->clear();
}

inline void OR_info::abort_alloc_set() {
    Page_set pset;

    // Go through all the entries in the alloc_set in reverse order
    for (int i = alloc_set->size(); i >= 0 ; i--) {
	Alloc_info *inf = &alloc_set->slot(i);
	if (pset.contains(inf->used_pageid)) continue;
	// Reset the state of the page.
     	Page *page = FE->pc->lookup_page(ornum, inf->used_pageid);
	th_assert(page, "Page not present in persistent cache");
	page->reset_state(inf->saved_state);
	pset.add1(inf->used_pageid);
	FE->pc->unlock_page(ornum, inf->used_pageid);
    }
    alloc_set->clear();
}

OR_info_map::OR_info_map() {
    last_returned = new OR_info(0);
    store1(0, last_returned);
}

OR_info_map::~OR_info_map() {
    // Delete all the or_info objects
    MapGenerator<UIntKey, OR_info *> iter(*this);
    UIntKey orkey;
    OR_info *or_info;
    while (iter.get(orkey, or_info)) {
	delete or_info;
    }
}

void OR_info_map::add_or(OR_num orx) {
    OR_info *or_info = new OR_info(orx);
    th_assert(!contains(orx), "Entry already present for or");
    store1(orx, or_info);
}

void OR_info_map::commit_orefs() {
    // Throw away the alloc_set entries in the or_map;
    // Go through all the entries in the or_map;
    // We also unmark empty pages that have objects on them now

    MapGenerator<UIntKey, OR_info *> iter(*this);
    UIntKey orkey;
    OR_info *or_info;
    while (iter.get(orkey, or_info)) {
	or_info->commit_alloc_set();
    }
}

void OR_info_map::abort_orefs() {
    // Reset the state of all pages in all entries in the or_map;
    // Go through all the entries in the or_map;
    MapGenerator<UIntKey, OR_info *> iter(*this);
    UIntKey orkey;
    OR_info *or_info;

    // Note: This code REQUIRES that Page::reset_state is not called on a page
    // more than once. So we scan the array from the high end and construct a
    // map for the pages

    while (iter.get(orkey, or_info)) {
	or_info->abort_alloc_set();
    }
}
