#include "compiler/C++wrappers/core.h"
#include "compiler/C++wrappers/fields.h"

#include "commit_info.h"
#include "or_info.h"
#include "fe_send_message.h"
#include "or_recv_message.h"
#include "trans_log.h"
#include "tm.h"

#include "cache1/resident_object_table.h"

#include "fe/main/fe_config.h"

#include "utils/communication.h"
#include "utils/mdebug.h"

#include "common/or_set.h"
#include "common/xref.h"
#include "common/locator.h"
#include "common/transaction.h"

declareArray(Request_ids, Ubits32)
implementArray(Request_ids, Ubits32)

Timer cumm_sort_timer;

TM::TM() {
    tlog = new Trans_log;
    abort_transaction = FALSE;
    allow_immediate_abort = TRUE;
    num_commits = 0;
}

TM::~TM() {
    delete tlog;
}

bool TM::commit() {
    // If transaction has aborted then finish the transaction's work and return
    num_commits++;
    if (must_abort()) {
	abort(FALSE);
	return FALSE;
    }

    cumm_commit_timer.start();
    cumm_make_timer.start();
    // Create a commit_info object for data that is used only during commit
    Commit_info *cinfo = new Commit_info;

    // Get the complete transaction objects for all the participants
    cinfo->make_transaction(tlog);
    // NOTE: Even if we don't have to send to the OR, the following code
    //       is executed to clean up the data structures. It should
    //       not result in any messages/problems with any OR

    cumm_make_timer.stop();
    // Send the commit message to the participants.

    cumm_send_timer.start();
    OR_recv_commit_reply_msg reply_msg;
    Request_ids ids;  // To keep track of which ids to wait for
    // Receive the commit reply
    reply_msg.commit       = TRUE;
    reply_msg.num_received = 0;

    send_commit(cinfo, &reply_msg, &ids);
    cumm_send_timer.stop();

    cumm_process_timer.start();

    reply_msg.num_expected = ids.size();

    // Perform processing while the commit is going on
    process_before_commit_reply(cinfo);
    cumm_process_timer.stop();
    cumm_recv_timer.start();

    for (int i = 0; i < ids.size(); i++) {
	int num_handled = FE->comm->handle_messages_until_done(ids[i]);
        if (num_handled < 0)
	  th_fail("Communication problem while receiving "
		  "commit reply message");
    }

    bool committed = reply_msg.commit;
    cumm_recv_timer.stop();

    // Perform processing according to the commit/abort result
    complete_transaction(cinfo, committed);
    delete cinfo;

    cumm_commit_timer.stop();

    return committed;
}

void TM::abort(bool perform_longjmp) {
    complete_transaction(NULL, FALSE);
    if (perform_longjmp)
	fprintf(stderr, "Perform Longjmp not supported yet!!");
	// SIGNAL_ABORT(); XXX to be used later
}

//
//                    Internal Methods and procedures
//

void TM::process_before_commit_reply(Commit_info *cinfo) {
    cinfo->process_before_commit_reply();
    tlog->reset_read_bits();
    FE->rot->adj_rc_commit_request();
}

void TM::complete_transaction(Commit_info* cinfo, bool committed) {
    // Note: The relevant objects have been shrunk by the invalidation
    //        processing code.

    // If cinfo is NULL, then committed must be FALSE.
    // cinfo is NULL only when this method is being called from TM::abort

    th_assert(cinfo || !committed, "cinfo is NULL and transaction committed");

    // If no commit request was sent we need to reset the read bits.
    // The other bits are reset by the rot code.
    if (!cinfo) tlog->reset_read_bits();

    // If the transaction committed, install the objects in the ROT
    if (committed) {
      install_new_objects(cinfo);
    }

    // This should only be called after the commit reply is received. 
    // and after the new objects are installed in install_new_objects.
    // It makes final fixes to the reference counts in the ROT.
    FE->rot->adj_rc_commit_reply(committed);

    if (cinfo) cinfo->commit_reply_processing(committed);

    // Clear the transaction log for the next transaction
    tlog->clear();
    abort_transaction = FALSE;
}

void TM::install_new_objects(Commit_info *cinfo) {
    Commit_info::New_persistent_iter iter(cinfo);
    Core c;
    Xref xref;

    while (iter.get(c, xref)) {
	Fields f = c->get_fields();
	f->unstamp_read();
	FE->rot->install_new_object(xref.orx, xref.oref, c);
    }
    iter.destroy();
}

bool TM::send_commit(Commit_info *cinfo, OR_recv_commit_reply_msg *reply_msg,
		     Request_ids *ids) {

    OR_set const *parts = cinfo->participants();
    OR_set::Elements parts_iter(parts);
    OR_num orx;
    bool read_only = cinfo->is_read_only();
    
    while (parts_iter.get(orx)) {
	// Get the transaction object, set up the send message
	Transaction *trans = cinfo->get_transaction(orx, FALSE);

	if (FE->debug_level > 1) {
	    // Unparse the transaction object
	}

	bool is_coord = orx == trans->coordinator;
	FE_send_commit_msg msg(is_coord, trans);

	// Register a reply message for all participants if this transaction
	// is read_only. Else just expect one from the coordinator
	OR_recv_commit_reply_msg *register_msg;
	register_msg = (read_only || is_coord)? reply_msg : 0;
	Ubits32 req_id = 0;
	bool success =FE->comm->send(OR_address(orx), &msg, register_msg, req_id);
	// If we are expecting a message, keep the id in the array
	// else we reject the id;
	// For read_only, we should probably register one message only
	// and pass the same id for the remaining participants
	if (register_msg) {
	    ids->append(req_id);
	}

	if (!success) return FALSE;
    }
    return TRUE;
}

void TM::print_stats(FILE* fp) {
    extern Timer cumm_ros_send_timer, cumm_mos_send_timer, cumm_nos_send_timer;

    float cumm_make_time = cumm_make_timer.elapsed();
    float cumm_send_time = cumm_send_timer.elapsed();
    float cumm_recv_time = cumm_recv_timer.elapsed();
    float cumm_sort_time = cumm_sort_timer.elapsed();
    float cumm_process_time = cumm_process_timer.elapsed();
    float cumm_commit_time = cumm_commit_timer.elapsed();
    fprintf(fp, "Transaction Statistics\n");
    fprintf(fp, "Number of commits = %d\n", num_commits);
    fprintf(fp, "Make = %9.5f, Sort = %9.5f, Process = %9.5f\n", 
	    cumm_make_time, cumm_sort_time, cumm_process_time);
    fprintf(fp, "Send = %9.5f, Recv = %9.5f, Total = %9.5f\n",
	    cumm_send_time, cumm_recv_time, cumm_commit_time);

    float cumm_ros_send_time = cumm_ros_send_timer.elapsed();
    float cumm_mos_send_time = cumm_mos_send_timer.elapsed();
    float cumm_nos_send_time = cumm_nos_send_timer.elapsed();
    fprintf(fp, "C_Ros = %9.5f, C_Mos = %9.5f, C_Nos = %9.5f\n",
	    cumm_ros_send_time, cumm_mos_send_time, cumm_nos_send_time);
}


void TM::reset_statistics() {
    extern Timer cumm_ros_send_timer, cumm_mos_send_timer, cumm_nos_send_timer;

    cumm_make_timer.reset();
    cumm_send_timer.reset();
    cumm_recv_timer.reset();
    cumm_sort_timer.reset();
    cumm_process_timer.reset();
    cumm_commit_timer.reset();
    num_commits = 0;
    cumm_ros_send_timer.reset();
    cumm_mos_send_timer.reset();
    cumm_nos_send_timer.reset();
}
