#ifndef _TM_H
#define _TM_H

// This is the transaction manager at the FE. It is responsible for all the
// transaction related activities at the client

#include <stdio.h>

#include "or_info.h"
#include "invalidation.h"

#include "common/or_num.h"
#include "common/oref.h"

#include "utils/basic.h"
#include "utils/bits.h"
#include "utils/Timer.h"

class Trans_log;
class Commit_info;
class Request_ids;
class OR_recv_commit_reply_msg;


class TM {
  public:
    TM();
    // effects: Default constructor. Creates an initialized object

    ~TM();

    bool must_abort() const { return abort_transaction;}
    // effects: Returns TRUE if the current transaction must be aborted
    //          because of some invalidation message that was earlier received

    Trans_log* transaction_log() const { return tlog;}
    // effects: Returns a pointer to the transaction log object stored in TM

    bool commit();
    // effects: Commits the current transaction. Starts the next transaction
    //          and returns TRUE iff transaction committs.

    void abort(bool perform_longjmp);
    // effects: Aborts the current transaction. Resets the transaction state
    //          and starts the next transaction. If perform_longjmp is TRUE,
    //          this method makes a longjmp to the client interface code (if
    //          immediate abort has been enabled)

    bool invalidate_objects(OR_num orx, const Oref* inv_orefs, int set_size,
			    Ubits32 msg_start, Ubits32 msg_end,
			    Msg_Wait_Type wait_type);
    // effects: Invalidates the objects whose orefs are specified in
    //          "inv_orefs". "set_size" is the size of the
    //          array inv_orefs. msg_start and msg_end indicate the
    //          invalidation message numbers to which these
    //          inv_orefs correspond. Note that this method does not
    //          abort the transaction but sets it up so that it can
    //          be aborted later (by calling abort_transaction_if_needed)
    // returns: TRUE iff the transaction must be aborted

    void enable_immediate_abort() {allow_immediate_abort = TRUE;}
    // effects: Allows the TM to perform a longjmp if an
    //          invalidation message aborts the current transaction

    void disable_immediate_abort() { allow_immediate_abort = FALSE;}
    // effects: Prevents the TM from performing a longjmp even if an incoming
    //          invalidation message aborts the current transaction

    bool is_immediate_abort_enabled() {return allow_immediate_abort;}
    // effects: Returns TRUE iff immediate_abort has been enabled

    void print_stats(FILE* fp = stderr);
    // effects: Print the statistics of the TM on fp

    void reset_statistics(void);
    // effects: Resets TM statistics.

  private:
    // Representation
    Trans_log *tlog; // Information about read/modified objects

    // Variables used/set by invalidation code
    bool abort_transaction;     // TRUE if the current trans has to be aborted
    bool allow_immediate_abort;
    // When this variable is TRUE, the TM performs a longjmp if an
    // invalidation message aborts the current trans (else it does not)

    // Stats information
    int num_commits;
    Timer cumm_make_timer, cumm_send_timer, cumm_recv_timer, cumm_commit_timer,
          cumm_process_timer;

    // ----------------------------------------------------------------------
    //
    // Internal Methods:
    //
    // ----------------------------------------------------------------------

    void process_before_commit_reply(Commit_info *cinfo);
    // requires: Given transaction has been sent, but response hasn\'t come back.
    // effects   Perform computation that must be done before commit succeeds,
    //           but which may be put off until after request has been sent.

    void complete_transaction(Commit_info* cinfo, bool committed);
    // requires: This routine is called after the result of the transaction
    //           is known (committed is TRUE iff transaction committed)
    //           cinfo is a valid object that was used to create the current
    //           transaction info. process_before_commit_reply has been called
    //           OR cinfo is NULL and committed is FALSE. This is the case
    //           when the transaction has to be aborted by force

    void install_new_objects(Commit_info *cinfo);
    // requires: Transaction has committed successfully and cinfo refers to
    //           the commit object used for committing this transaction
    //           (called by complete_transaction)
    // effects:  Installs the new objects into the persistent heap

    // Methods used by the commit RPC code

    bool send_commit(Commit_info *cinfo, OR_recv_commit_reply_msg* reply_msg,
		     Request_ids *ids);
    // modifies: ids
    // effects:  Sends the commit message to the different participants (using
    //           cinfo) and installs "reply_msg" as the receive handler for the
    //           reply/ies to be received. Stores the ids for the registered
    //           handlers in "ids". If the trans is read-only, it registers
    //           reply_msg for all parts, else only for the coordinator.
    //           Returns TRUE iff succeeds

};

#endif /* _TM_H */
