#include "compiler/C++wrappers/core.h"
#include "compiler/C++wrappers/fields.h"

#include "trans_log.h"
#include "tm.h"

#include "cache1/resident_object_table.h"

#include "fe/main/fe_config.h"

#include "common/obj_bitfield.h"
#include "common/ros.h" 

#include "utils/intarray.h"
#include "utils/mdebug.h"

// Predicted sizes of volatile, persistent, modified object sets and copy sets
unsigned int const Predict_psize = 512;
unsigned int const Predict_vsize = 512;
unsigned int const Predict_nsize = 512;
unsigned int const Predict_msize = 512;


// Per-or logs.
struct OR_logs {
  Ros ros;             // Log of orefs of objects from or "onum" that were accessed 
                       // during the current transaction.
  Mods mods;           // Log of modifications of objects from or "onum"

  bool sorted;         // Whether sets have been sorted
  int last;            // Last element known to be yielded

  OR_logs() : ros(Predict_psize), mods(Predict_msize), sorted(false), last(0) {}
};

#define IMPL_KEY   UIntKey
#define IMPL_VALUE OR_logs *
#include "utils/impl_map.t"

implementArray(Mods, Mod_record)
implementArray(Data, Slot)

// Comparison operators for Mod_record for the sorting code.
inline bool operator <  (Mod_record const &m1, Mod_record const &m2) {
  return m1.oref < m2.oref;
}
inline bool operator <= (Mod_record const &m1, Mod_record const &m2)  {
  return m1.oref <= m2.oref;
}
inline bool operator >  (Mod_record const &m1, Mod_record const &m2) {
  return m1.oref > m2.oref;
}

#define QUICK_SORT_ELEMENT Mod_record
#include "utils/impl_sort.t"



Trans_log::Trans_log() : persistent_copies(FE->per_undo_log_size), last(0), 
  last_ornum(0), non_persistent(Predict_vsize), volatile_mods(Predict_msize), 
  volatile_copies(), new_volatile(Predict_nsize), read_only(true) {
#if TRANS_ORPHAN_CHECK
  watermark = 0;
  glb = 0;
#endif
}

Trans_log::~Trans_log() { clear(); }

void Trans_log::clear() {
  // Delete all OR logs
  UIntKey orx;
  OR_logs *logs;
  MapGenerator<UIntKey,  OR_logs *> gen(persistent);
  while (gen.get(orx, logs)) delete logs;
  
  
  persistent.clear();
  persistent_copies.clear();
  last = 0;
  last_ornum = 0;
  last_vol = 0;

  non_persistent.clear();
  volatile_mods.clear();
  new_volatile.clear();
  volatile_copies.clear();  

  read_only = true;
}


//
// Methods and functions to log accesses.
//
void Trans_log::add_new_or(OR_num orx) {
  last = new OR_logs;
  persistent.add(orx, last);
#if TRANS_ORPHAN_CHECK
  // If OR was not accessed before in this transaction
  // check for possible inconsistencies that might be observed
  check_or_status(or);
  Ubits64 vnum = f->vnum();
  if (watermark < vnum) watermark = vnum;
  watermark = 0;
  if (watermark > glb) {
    get_or_info();
  } 
#endif
}

inline void Trans_log::log_read(Core c) {
  Fields f = c->get_fields();
  th_assert(!f->is_read(), "Tried to mark a read object");
  f->stamp_read();

  if (c->is_persistent()) {
    OR_num orx = f->orx();
    th_assert(f->orx() > 0, "Bad OR number = %d\n");
    if (orx == last_ornum) {
      last->ros.add_obj(c->oref());
      last->sorted = false;
      return;
    }
    
    if (!persistent.find(orx, last)) add_new_or(orx);
  
    last_ornum = orx;
    last->ros.add_obj(c->oref());
    last->sorted = false;
  } else {
    // Volatile objects.
    non_persistent.append(c);
  }
}

inline void Trans_log::log_written(Core c) {
    Fields f = c->get_fields();
    th_assert(!f->is_written(), "Tried to mark a written object");
   
    f->stamp_written();
    read_only = false;

    
    // Make a copy of the current object state for modified
    // objects so that it can be reverted to if the transaction aborts
    Uint num_slots = c->num_slots() + Fields_c::header_slots();
    bool is_persistent = c->is_persistent();
    Data &copies = is_persistent ? persistent_copies: volatile_copies;
    Fields copy_fields = NULL;

    if (num_slots <= (unsigned)(FE->per_undo_log_size - copies.size()) || !is_persistent) {
	// Memcpy is being used for efficiency rather than copies->concat
	copies._enlarge_by(num_slots);
	copy_fields = (Fields) (copies.as_pointer() + copies.size() - num_slots);
	memcpy(copy_fields, f, num_slots * Slot_size);
    } else {
	// No space for copying a persistent object switch to using
        // eager reference count updates.
      Obj_bitfield bf = c->bitfields();
      if (bf != OBJ_BF_ALLDATA)
	f->stamp_eager_ref_counting();
    }

    // Create a log record for the access
    Mod_record m;
    m.oref = c->oref();
    m.copy = (copy_fields) ? (copies.size() - num_slots) :  -ROT_INDEX((Core_s*)c);
    if (is_persistent) {
      OR_num orx = f->orx();
      th_assert(f->orx() > 0, "Bad OR number = %d\n");
      if (orx != last_ornum) {
	if (!persistent.find(orx, last)) add_new_or(orx);
	last_ornum = orx;
      }
      last->mods.append(m);
      last->sorted = false;

      // Having mod objects in ros simplifies code to handle invalidations     
      if (!f->is_read()) last->ros.add_obj(m.oref);
    } else {
      volatile_mods.append(m);
      if (!f->is_read()) non_persistent.append(c);
    }
}

inline void Trans_log::log_new(Core c) {
    // Creator must call this
    new_volatile.append(c);
    // We must ensure that this object does not get unnecessarily copied in
    // the volatile set. That is why the read/write stamps are being marked

    Fields f = c->get_fields();
    th_assert(!f->is_read(), "Newly created object marked as read");
    th_assert(!f->is_written(), "Newly created object marked as written");
    f->stamp_read();
    f->stamp_written();
    read_only = false;
}


void Log_read(struct Core_s *o) {
    FE->tm->transaction_log()->log_read((Core)o);
}


void Log_write(struct Core_s *o) {
    FE->tm->transaction_log()->log_written((Core)o);
}


void Log_new(struct Core_s *o) {
    FE->tm->transaction_log()->log_new((Core)o);
}


//
// Iterators for accessing various sets
//

void Trans_log::sort_logs(OR_logs *logs) {
  if (logs->sorted) return;
  logs->sorted = true; 
  logs->last = 0;
 
  extern Timer cumm_sort_timer;
  cumm_sort_timer.start();
 
  // Sort ros and mods if not sorted yet.
  logs->ros.sort();
  quick_sort(logs->mods.as_pointer(), logs->mods.size());
  
  // Eliminate duplicates and entries that correspond to objects 
  // that have also been written from ros
  Oref *rset = logs->ros.rset.as_pointer();
  int ros_sz = logs->ros.rset.size();

  // If logs->mods is empty set mos to a dummy value.
  int mos_sz = logs->mods.size();
  Mod_record dummy; dummy.oref = 0; 
  Mod_record *mos = (mos_sz) ? logs->mods.as_pointer() : &dummy;

  int from = 0, to = 0, mos_ind = 0;
  Oref prev = 0;
  
  // skip portion of array without duplicates.
  while (from < ros_sz) {
    Oref cur = rset[from];
    if (prev == cur) {
      // A duplicate. Make to point to the current element
      // from goes ahead
      to = from++;
      break;
    } else if (cur == mos[mos_ind].oref) {
      // An entry for a modified object
      to = from++;
      if (mos_ind < mos_sz) mos_ind++;
      break;
    } else {
      prev = rset[from++];
      to++;
    }
  }
  
  // Invariant: from > to. 
  // Process rest of array.
  while (from < ros_sz) {
    Oref cur = rset[from];
    if (prev == cur) {
      // A duplicate 
      from++;
    } else if (cur == mos[mos_ind].oref) {
      // An entry for a modified object
      from++;
      if (mos_ind < mos_sz) mos_ind++;
    } else {
      // A non-duplicate, non-modified object
      prev = rset[from]; 
      rset[to++] = rset[from++];
    }
  }
 
  logs->ros.rset.remove(ros_sz-to);

  cumm_sort_timer.stop();
}


void Trans_log::sort_logs() { 
   UIntKey key;
   OR_logs *logs;
   MapGenerator<UIntKey, OR_logs* > gen(persistent);
 
   while (gen.get(key, logs)) sort_logs(logs);
}


Trans_log::Persistent_iter::Persistent_iter(Trans_log* tlog) : gen(tlog->persistent) {}
	
bool Trans_log::Persistent_iter::get(OR_num &orx, Ros *&ros, bool &has_mods) {
  UIntKey key;
  OR_logs *logs;
  
  if (!gen.get(key, logs)) return false;

  orx = key.val;
  ros = &logs->ros;
  has_mods = (logs->mods.size() > 0);
  
  return true;
}


Trans_log::Volatile_iter::Volatile_iter(Trans_log *trans_log, Iter_option opt) {
  tlog = trans_log;
    no_sets = 0;
    set_index = 0;
    index = 0;
    for (Uint i = 0; i < Trans_iter_sets; i++) sets[i] = NULL;

    // Set up the sets to be iterated over.
    if (opt & Iter_volatile)
	sets[no_sets++] = &tlog->non_persistent;
    if (opt & Iter_new_volatile)
	sets[no_sets++] = &tlog->new_volatile;
    th_assert(no_sets, "Iterator called without setting any options");

    core_set = sets[set_index]->as_pointer();
    size = sets[set_index]->size();
}

bool Trans_log::Volatile_iter::get_from_next_set(Core& c) {

    set_index++;
    if (set_index >= no_sets) {
	size = 0;
	return FALSE;
    }
    core_set = sets[set_index]->as_pointer();
    size = sets[set_index]->size();
    index = 0;
    return get (c); // Note: Recursive call on get
}



Trans_log::Mod_iter::Mod_iter(Trans_log *tlog, OR_num orx, bool only_new) {
  Mods *mods;
  index = 0;
  if (orx == Null_or) {
    // Iteration over modified volatile objects
    mods = &tlog->volatile_mods;
    copy_buffer = tlog->volatile_copies.as_pointer();
    if (only_new) index = tlog->last_vol;
    tlog->last_vol = tlog->volatile_mods.size();
  } else {
    OR_logs *tmp = tlog->persistent.fetch(orx);
    mods = &tmp->mods;
    copy_buffer = tlog->persistent_copies.as_pointer();
    if (only_new) index = tmp->last;
    tmp->last = tmp->mods.size();
  }

  mod_arr = mods->as_pointer();
  size = mods->size();
}


void Trans_log::reset_read_bits() {
  // Reset read bits in persistent objects.
  Trans_log::Persistent_iter piter(this);
  OR_num orx;
  bool has_mods = true;
  Ros *ros;
  
  while (piter.get(orx, ros, has_mods)) {
    Oref oref;
    Ros::Elements gen(ros);
    while (gen.get(oref)) {
      Core c = FE->rot->lookup(orx, oref);
      if (c && !c->is_discarded()) c->get_fields()->unstamp_read();
    }
  }

  Trans_log::Volatile_iter viter(this, Iter_volatile);
  Core c;
  while (viter.get(c)) {
    c->get_fields()->unstamp_read();
  }
}




#if TRANS_ORPHAN_CHECK
void Trans_log::check_or_status(OR_num or_check) {
    ors_used.set(or_check);
}

void Trans_log::get_or_info() {
    printf("Should never enter this method!!\n");
    // Update the glb by soliciting information from ORs that have
    // glb < watermark
}
#endif

