#ifndef _TRANS_LOG_H
#define _TRANS_LOG_H

// This is an abstraction that keeps track of objects that have been read and
// modified by the current transaction. It is also responsible for keeping the
// copies of the modified objects.

#include "compiler/C++wrappers/obj.h"
#include "compiler/C++wrappers/core.h"

#include "config/vdefs/TRANS_ORPHAN_CHECK.h"

#include "utils/array.h"
#include "utils/map.h"
#include "utils/valuekey.h"

// Maximum number of sets in the transaction iterator
unsigned int const Trans_iter_sets = 2;

typedef unsigned int Iter_option;

// Constants for the Trans_iter iterator
unsigned int const Iter_volatile = 0x1;
unsigned int const Iter_new_volatile = 0x2;

class Old_ref_log;
class OR_logs;
class Ros;


// Record describing a modification performed during a transaction.
struct Mod_record {
  Oref oref;    // If object is persistent, its oref; otherwise, oref is undefined
  int copy;     // If positive, offset into persistent_copies array
                // that locates this object's copy. Otherwise, 
                // the object has no copy (can only happen for persistent
                // objects) and "copy" is equal to -1 times the index of 
                // the object's core  in the ROT
};

declareArray(Mods, Mod_record)
declareArray(Data, Slot)


class Trans_log {
public:
  Trans_log();
  // effects Default constructor. Creates an initialized object

  ~Trans_log();
  // effects Default destructor
   
  void clear();
  // effects: Resets the transaction log for the next transaction

  //
  // Iterators for accessing the transaction logs. 
  //
  // Note: Once an iterator from one of the following classes is created 
  // it must be used and destroyed before any change is made to trans_log.
  // The effect is undefined if an iterator method is called after
  // such a change. 
  //
	
  class Persistent_iter {
    //  An iterator for yielding the persistent objects accessed by the
    //  current transaction.
  public:
    Persistent_iter(Trans_log* tlog);
    // effects Return an iterator over the read set in tlog
	
    bool get(OR_num &orx, Ros *&ros, bool &has_mods);
    // modifies "or", "ros", "has_mods"
    // effects  Returns false if there are no more ORs that were used 
    //          during the current transaction (and does not modify any 
    //          of the arguments). Otherwise, returns true and sets "or" 
    //          to the next used OR, assigns the set of orefs of objects 
    //          from "or" that were read but not written to "ros", and sets
    //          "has_mods" to true iff some object from "or" was modified.
    //          sort_logs (see bellow) can be called before the generator
    //          is created to ensure that all yielded "ros" are sorted by 
    //          increasing oref.
    // NOTE: any pointer returned in ros can only be used until the next time
    //       clear is called on tlog.

  private:
    MapGenerator<UIntKey, OR_logs* > gen; // generator for log of persistent objects
  };


  class Volatile_iter {
    //  An iterator for yielding the volatile objects read or written by the
    //  current transaction
	
  public:
    Volatile_iter(Trans_log* tlog, Iter_option opt);
    // effects Return an iterator over the read/write volatile set in tlog
    //         The caller can pass the following options (or a combination
    //         of them by Oring them). If a combination of sets is
    //         specified, the elements of 1 set will be yielded
    //         together. However, the order of the sets will be arbitrary
    //
    //         Iter_volatile     -- yields volatile objs (not newly created)
    //         Iter_new_volatile -- yields newly created volatile objs
	
    bool get(Core &c);
    // modifies "c"
    // effects  Sets "c" to the next object and returns true.
    //          Returns false if there are no more objects (and does not
    //          modify "c")

  private:
    Trans_log* tlog; // Transaction log for which elements are yielded
    Cores* sets[Trans_iter_sets];
    // Core sets from which elements have to be yielded. The sets to be
    // yielded are put from 0 onwards. If all sets are not being yielded,
    // the remaining entries are NULL. All these variables have been added
    // for efficient yielding of objects
    Core* core_set;     // Current set being interated over
    int no_sets;        // No. of sets in "sets" that are non-null
    int set_index;      // Current set being yielded
    int size;           // Size of core_set
    int index;          // Index of next element to be yielded.
    // Invariants:
    //    core_set == sets[set_index],
    //    no_sets <= Trans_iter_sets, set_index < Trans_iter_sets,
    //    size == core_set->size(),  index <= core_set->size()

    bool get_from_next_set(Core& c);
    // requires: This method is called only by Trans_Iter::get
    // effects:  Sets the iterator to proceed to the next set and then
    //           calls "get" to yield the next element from that set.
    //           The returned value and the value of "c" is set as
    //           specified in get.
  };


  class Mod_iter {
    //  An iterator for yielding the objects modified by the current 
    //  transaction and their original copies.
    //  NOTE: This iterator does NOT yield volatile objects
    //        newly created in this transaction
	
  public:
    Mod_iter(Trans_log* tlog, OR_num orx=Null_or, bool only_new=false);
    // requires Some modification to an object from "or" is in tlog or "or"=Null_or
    // effects Return an iterator over the set of modified objects from
    //         OR "or" (or modified volatile objects if  "or"=Null_or).
    //         sort_logs (see bellow) can be called before the generator
    //         is created to ensure that objects are yielded in increasing 
    //         oref order. If "only_new" is true, the generator will yield 
    //         all the entries that were added to the log since the last 
    //         time a Mod_iter was created but (usually) no more.

    bool get(Core& c, Fields &f);
    // modifies "c", "f"
    // effects  Sets "c" to the next object, f to the base copy of the
    //          object and returns true. Returns false if there are no
    //          more objects (and does not modify c, f)

    private:
	Mod_record *mod_arr;  // The set of modified objects and their copies
	int index;            // Index of next element to be yielded.
	int size;             // Size of core_set
        Slot *copy_buffer;    // Buffer containing copies  
    };


    void sort_logs();
    // effects: sorts the logs of accessed persistent objects by increasing xref.
    //          The logs remain sorted until new accesses are logged.
  
    void reset_read_bits();
    // effects: Resets the read bits of all volatile objects (except newly created) 
    //    and persistent objects that have been read but not modified.


    bool is_read_only() { return read_only; }
    // effects: Returns true iff no objects were modified since the current
    //    transaction started.

    //--------------------------------------------------------------
    //  Private section
    //--------------------------------------------------------------

    void log_read(Core c);
    // requires: c has not been marked as read
    // effects: Marks c as read and stores it in a transaction log

    void log_written(Core c);
    // requires: c has not been marked as written
    // effects: Marks c as written and stores it in the transaction log.
    //          It also makes a copy of the object (to speedup recovery
    //          on transaction aborts and reference count adjustments)

    void log_new(Core c);
    // requires: c refers to a newly created object in this transaction
    // effects: Adds c to the set of newly created objects in this transaction



  private:
    // Representation for Trans_log

    friend Persistent_iter; 
    friend Volatile_iter;
    friend Mod_iter;
    friend Old_ref_log;

    Map<UIntKey, OR_logs* > persistent;  // Xrefs of persistent objs read/written
                                         // by transaction (organized per or) and 
                                         // offsets into persistent copies (also
                                         // organized per or)
    
    Data  persistent_copies; // Buffer for copies of modified 
                             // persistent objects
    
    // Next two variables cache last OR log that was accessed and the id
    // of the corresponding OR. 
    OR_logs* last;
    OR_num   last_ornum;      
                         
    Cores non_persistent;  // Cores of volatile objs read/written by transaction
    Mods  volatile_mods;   // Cores of modified volatile objects and their copies.
    Data  volatile_copies; // Buffer for copies of modified volatile objects 
    int last_vol;          // Index of last yielded volatile object

    Cores new_volatile; // Cores of new volatile objects created by transaction
    bool read_only;     // Whether the current transaction is read-only

    // Helper methods
    void add_new_or(OR_num orx);
    static void sort_logs(OR_logs *logs);

#if TRANS_ORPHAN_CHECK
    OR_bmap ors_used;
    Ubits64 watermark;
    Ubits64 glb;

    void check_or_status(OR_num or_check);
    void get_or_info();
#endif

};


inline bool Trans_log::Volatile_iter::get(Core& c) {
  if (index < size) {
    // At this point index < sets[set_index]->size()
    c = core_set[index];
    index++;
    return TRUE;
  }
  return get_from_next_set(c);
}


inline bool Trans_log::Mod_iter::get(Core& c, Fields &cop) {
  if (index < size) {
    int copy_index = mod_arr[index].copy;
    if (copy_index >= 0) {
      cop = (Fields) (copy_buffer + copy_index); 
      c = cop->to_core();
    } else {
      // Can only happen for persistent objects for which no copy exists
      cop = 0;
      c = (Core_c*) ROT_FETCH(-copy_index);
    }

    index++;
    return TRUE;
  }
  return FALSE;
}


#endif /* _TRANS_LOG_H */



