//  ClassData.java -- the data from a Java class file

//  Copyright 1996, 1997 Arizona Board of Regents;
//  see COPYRIGHT file for details.



package toba;

import java.io.*;
import java.util.*;


public class ClassData {


// instance variables

    public String name;			// class name
    public String cname;		        // name as used in C
    public String fname;		        // name used for toba file
    public String supername;		// superclass name
    public int major, minor;		// version numbers
    public int access;			// access flags and other flags
    public Constant[] constants;	        // constant table
    public String[] interfaces;		// interfaces implemented
    public Field[] fields;		// fields
    public Field[] methods;	        // methods
    public Hashtable symtab;		// symbol table (fields & methods)
    public Attribute[] attributes;	// class attributes

    // not set by constructor, but see Supers.java:

    public ClassData superclass;         // superclass data 

    public Field[] static_mtable;        // static methods of this class alone
    public Field[] instance_mtable;      // all instance methods implemented in

    public int static_num_slots;
    public int instance_num_slots;
    public long static_bit_fields;
    public long instance_bit_fields;
                                        //   this class alone
    public Field[] virtual_mtable;       // all virtual methods of this class 
                                        //   and its superclasses

    public int nsupers;		        // number of superclasses incl self
    public int ninters;		        // number of interfaces implemented
    public int nothers;		        // number of "other" classes referenced
    public Vector supers;
    public Vector inters;
    public Vector others;

    public Vector instance_fields;
    public Vector static_fields;

    public boolean is_persistable; 

    // not set by constructor, but see InsGen.java:

    public Hashtable arrays = new Hashtable();


// flags from Java class files

public static final int ACC_PUBLIC		= 0x0001;
public static final int ACC_PRIVATE		= 0x0002;
public static final int ACC_PROTECTED	= 0x0004;
public static final int ACC_STATIC		= 0x0008;
public static final int ACC_FINAL		= 0x0010;
public static final int ACC_SYNCHRONIZED	= 0x0020;
public static final int ACC_VOLATILE	= 0x0040;
public static final int ACC_TRANSIENT	= 0x0080;
public static final int ACC_NATIVE		= 0x0100;
public static final int ACC_INTERFACE	= 0x0200;
public static final int ACC_ABSTRACT	= 0x0400;



//  new ClassData(d) -- load class data from DataInputStream d
//  new ClassData(d, name) -- load class data and validate name
//
//  Note:  the file is closed after loading.

ClassData(DataInputStream d, String classname)
    throws ClassFormatError, IOException, ClassNotFoundException
{
    this(d);
    if (classname != null && ! classname.equals(this.name))
    	throw new ClassNotFoundException(classname);
}

ClassData(DataInputStream d)
    throws ClassFormatError, IOException
{
    symtab = new Hashtable();

    // read header
    if (d.readInt() != 0xCafeBabe)
	throw new ClassFormatError("bad magic number");
    minor = d.readUnsignedShort();
    major = d.readUnsignedShort();

    // read constant table
    constants = Constant.load(d);

    int length = constants.length;
    Constant c[] = new Constant[length+4];
    System.arraycopy(constants, 0, c, 0, length);

    c[length] = new Constant();
    c[length].index = length;
    c[length].tag = Constant.CLASS;
    c[length].value = "java.lang.Object";

    c[length+1] = new Constant();
    c[length+1].index = length+1;
    c[length+1].tag = Constant.CLASS;
    c[length+1].value = "java.lang.String";

    c[length+2] = new Constant();
    c[length+2].index = length+2;
    c[length+2].tag = Constant.CLASS;
    c[length+2].value = "java.lang.Class";

    c[length+3] = new Constant();
    c[length+3].index = length+3;
    c[length+3].tag = Constant.CLASS;
    c[length+3].value = "java.lang.NullPointerException";

    constants = c;

    // read access flags, class, superclass
    access = d.readUnsignedShort();

    // ZZZ: AAA: Nov 30
    name = (String)constants[d.readUnsignedShort()].value;

    cname = Names.hashclass(name);
    fname = Names.classfile(name);
    int i = d.readUnsignedShort();

    if (i > 0) {
	supername = (String)constants[i].value;

	// ZZZ: TEMPORARY, to avoid class libraries
	/*
	if (Names.hashclass(supername).equals("java_lang_Object")) {
	  supername = null;
	  // constants[i] = null;
	}
	*/
    }

    // read interface list
    interfaces = new String[d.readUnsignedShort()];
    for (i = 0; i < interfaces.length; i++) {
	int j = d.readUnsignedShort();
	if (j > 0)
	    interfaces[i] = (String)constants[j].value;
	else
	    interfaces[i] = null;
    }

    // read fields
    fields = new Field[d.readUnsignedShort()];
    for (i = 0; i < fields.length; i++) {
	Field f = new Field(d, constants);
	f.next = (Field)symtab.get(f.name);
	symtab.put(f.name, f);
	// for naming purposes, assume no shadowing
	f.cname = "th_" + Names.hashvar(f.name);
	fields[i] = f;
    }


    // read methods
    methods = new Field[d.readUnsignedShort()];
    for (i = 0; i < methods.length; i++) {
	Field f = new Field(d, constants);
	f.next = (Field)symtab.get(f.name);
	symtab.put(f.name, f);
	f.cname = Names.hashmethod(f.name, this.name, f.signature);
	f.selfClass = cname;
	methods[i] = f;
    }

    // read class attributes

    attributes = new Attribute[d.readUnsignedShort()];
    for (i = 0; i < attributes.length; i++)
	attributes[i] = new Attribute(d, constants);

    // all done
    d.close();				// close file for convenience of caller

}



//  getmethod(name, climb) -- find "()V" method by name
//
//  if climb is true, and if superclasses have been loaded, check them too.

Field getmethod(String name, boolean climb)
{
    for (ClassData k = this; k != null; k = k.superclass) {
	for (Field m = (Field)k.symtab.get(name); m != null; m = m.next) {
	    if (m.signature.equals("()V")) {
		return m;
	    }
	}
	if (!climb)
	    break;
    }
    return null;
}


} // class ClassData
