//  Instr.java -- Java bytecode instructions

//  Copyright 1996, 1997 Arizona Board of Regents;
//  see COPYRIGHT file for details.



package toba;

import java.io.*;

class Instr {		// one instruction in the bytecode



    // instance variables

    int label;		// instruction label; 0 if not target
    int pc;		// program counter
    String before;	// stack state before execution of instruction
    String after;	// stack state after execution
    Instr succ;		// successor operation, if no branch
    Opcode opcode;	// operation code
    int length;		// instruction length
    int val;		// first or only operand value
    int[] more;		// additional values, if any
    Constant con;	// associated constant table entry
    boolean isTarget;	// is this instruction a jump target?
    boolean isReached;	// is this instruction ever reached?
    boolean isBoundary;	// is ins on range boundary (including by jump)?
    int xseg;		// exception range segment number



//  new Instr(bytecode, offset, ctab) -- build instruction instance
//
//  Note: relative jumps are turned into absolute PC values

Instr(byte[] code, int offset, Constant ctab[])
{
    int f, i, j, k, n, o, op;

    pc = offset;			// program counter
    before = after = "";		// recalculated later

    op = code[pc] & 0xFF;		// binary opcode
    opcode = Opcode.table[op];		// Opcode struct
    if (opcode == null)
	throw new VerifyError("illegal opcode " + op + " at pc=" + pc);

    if (opcode.kind == Opcode.WIDE) {	// special form: the "wide" bytecode
	length = 4;
	op = code[pc + 1] & 0xFF;
	opcode = Opcode.table[op];
	val = u2(code, pc + 2);
	if (opcode.kind == Opcode.IINC) {
	    length = 6;
	    more = new int[1];
	    more[0] = i2(code, pc + 4);
	}
	return;
    }

    length = opcode.length;		// tableswitch, lookupswitch fixed later
    f = opcode.flags;

    if (f != 0) {

	o = offset + 1;
	if ((f & Opcode.I8) != 0)	val = i1(code, o);
	else if ((f & Opcode.I16) != 0)	val = i2(code, o);
	else if ((f & Opcode.I32) != 0)	val = i4(code, o);
	else if ((f & Opcode.U8) != 0)	val = u1(code, o);
	else if ((f & Opcode.U16) != 0)	val = u2(code, o);
	if ((f & Opcode.PC) != 0)
	    val += offset;

	if ((f & Opcode.MORE) != 0) switch (opcode.kind) {
	    case Opcode.IINC:
	    case Opcode.MNEWA:
		more = new int[1];
		more[0] = i1(code, offset + length - 1);
		break;
	    case Opcode.TBLSW:
	    case Opcode.LKPSW:
		i = (offset + 4) & ~3;
		if (opcode.kind == Opcode.TBLSW) {
		    n = 3 + (i4(code, i + 8) - i4(code, i + 4) + 1); 
		    k = 1;
		} else {
		    n = 2 + 2 * i4(code, i + 4);
		    k = 2;
		}
		length = (i - offset) + 4 * n;
		more = new int[n];
		for (j = 0; j < n; j++)
		    more[j] = i4(code, i + 4 * j);
		more[0] += offset;
		for (j = 3; j < more.length; j+= k)
		    more[j] += offset;
		break;
	}

	if ((f & Opcode.CTAB) != 0) {
	    if (val < ctab.length)
	    	con = ctab[val];
	    else
	    	throw new VerifyError(
		    "bad ctab index, pc=" + offset + " val=" + val);
	}
    }

    return;
}



//  nextStack() -- compute stack state for successor instruction

String nextStack()
{
    StringBuffer s = new StringBuffer(this.before);
    char c;

    if (opcode.pop > s.length())	// sanity check
	throw new VerifyError("stack underflow: pc=" + pc +
	    " stack=" + before + " opcode=" + opcode.name);

    s.setLength(s.length() - opcode.pop);
    switch (opcode.kind) {
	case Opcode.LDC:
	    switch (con.tag) {
		case Constant.INTEGER:	s.append("i");	break;
		case Constant.LONG:	s.append("xl");	break;
		case Constant.FLOAT:	s.append("f");	break;
		case Constant.DOUBLE:	s.append("xd");	break;
		default:		s.append("a");	break;
	    }
	    break;
	case Opcode.GETF:
	case Opcode.GETS:
	    c = regtype(((Ref)con.value).signature);
	    if (c == 'l' || c == 'd')
		s.append('x');
	    s.append(c);
	    break;
	case Opcode.PUTF:
	case Opcode.PUTS:
	    c = regtype(((Ref)con.value).signature);
	    if (c == 'l' || c == 'd')
		s.setLength(s.length() - 1);
	    break;
	case Opcode.SWAP:
	    s.setLength(s.length() - 2);
	    s.append(before.charAt(before.length() - 1));
	    s.append(before.charAt(before.length() - 2));
	    break;
	case Opcode.DUP:
	    s.append(before.substring(before.length() - 1));
	    break;
	case Opcode.DUP2:
	    s.append(before.substring(before.length() - 2));
	    break;
	case Opcode.DUPX1:
	    s.insert(before.length()-2, before.substring(before.length()-1));
	    break;
	case Opcode.D2X1:
	    s.insert(before.length()-3, before.substring(before.length()-2));
	    break;
	case Opcode.DUPX2:
	    s.insert(before.length()-3, before.substring(before.length()-1));
	    break;
	case Opcode.D2X2:
	    s.insert(before.length()-4, before.substring(before.length()-2));
	    break;
	case Opcode.MNEWA:
	    s.setLength(s.length() - more[0]);
	    s.append(opcode.push);
	    break;
	case Opcode.IVIRT:
	case Opcode.INONV:
	case Opcode.ISTAT:
	case Opcode.IINTR:
	    String sig = ((Ref)con.value).signature;
	    String ret = sig.substring(sig.indexOf(')') + 1);
	    c = regtype(ret);
	    s.setLength(s.length() - argwords(sig));
	    if (c == 'l' || c == 'd')
		s.append('x').append(c);
	    else if (c != 'v')
		s.append(c);
	    break;
	default:
	    s.append(opcode.push);
	    break;
    }
    return s.toString();
}



//  argwords(sig) -- return number of argument words consumed by a call

static int argwords(String sig)
{
    int n = 0;

    for (int i = 1; i < sig.length(); i++) {
	switch (sig.charAt(i)) {
	    case ')':	return n;
	    default:	n += 1;  break;	// error
	    case 'B':	n += 1;  break;
	    case 'C':	n += 1;  break;
	    case 'D':	n += 2;  break;
	    case 'F':	n += 1;  break;
	    case 'I':	n += 1;  break;
	    case 'J':	n += 2;  break;
	    case 'S':	n += 1;  break;
	    case 'Z':	n += 1;  break;
	    case 'L':
		n += 1;
		while (sig.charAt(++i) != ';')
		    ;	
		break;
	    case '[':
		n += 1;
		while (sig.charAt(++i) == '[')
		    ;
		if (sig.charAt(i) == 'L')
		    while (sig.charAt(++i) != ';')
			;	
		break;
	}
    }
    return n;
}



//  regtype(sig) -- return register type character for field signature

static char regtype(String sig)
{
    switch (sig.charAt(0)) {
	case 'B':  return 'i';
	case 'C':  return 'i';
	case 'D':  return 'd';
	case 'F':  return 'f';
	case 'I':  return 'i';
	case 'J':  return 'l';
	case 'S':  return 'i';
	case 'V':  return 'v';
	case 'Z':  return 'i';
	default:   return 'a';
    }
}



//  u1(code, offset) -- load 1-byte unsigned integer from code.
//  u2(code, offset) -- load 2-byte unsigned integer from code.
//  i1(code, offset) -- load 1-byte signed integer from code.
//  i2(code, offset) -- load 2-byte signed integer from code.
//  i4(code, offset) -- load 4-byte signed integer from code.

static int u1(byte code[], int offset)
{
    return code[offset] & 0xFF;
}

static int u2(byte code[], int offset)
{
    return ((code[offset] & 0xFF) << 8) | (code[offset + 1] & 0xFF);
}

static int i1(byte code[], int offset)
{
    return code[offset];
}

static int i2(byte code[], int offset)
{
    return (code[offset] << 8) | (code[offset + 1] & 0xFF);
}

static int i4(byte code[], int offset)
{
    return (code[offset] << 24) | ((code[offset + 1] & 0xFF) << 16) |
	((code[offset + 2] & 0xFF) << 8) | (code[offset + 3] & 0xFF);
}




//  dump(d, list) -- dump list of instructions on a file as C comments

static void dump(PrintWriter d, Instr list[])
{
    d.println();
    d.println("/*");
    for (int i = 0; i < list.length; i++) {
	Instr ins = list[i];
	if (ins.isBoundary)
	    d.print("> ");
	else
	    d.print("  ");
	if (ins.isTarget)
	    d.print("L" + ins.label + ":  ");
	else if (ins.isReached)
	    d.print("     ");
	else
	    d.print("  xxx  ");
	if (ins.pc < 10)
	    d.print(" ");
	d.print(ins.pc + ".  " + ins.before);
	if (ins.before.length() < 10)
	    d.print("          ".substring(ins.before.length()));
	else
	    d.print("  ");
	d.print(ins.opcode.name);
	if ((ins.opcode.flags & ~Opcode.NFT) != 0)
	    d.println(" " + ins.val);
	else
	    d.println();
    }
    d.println("*/");
    d.println();
}



} // class Instr
