//  Method.java -- analysis of one Java method

//  Copyright 1996, 1997 Arizona Board of Regents;
//  see COPYRIGHT file for details.



package toba;

import java.io.*;
import java.util.*;



class Method {			// information about one Java method


    // instance variables

    ClassData cl;		// containing class
    Field fl;			// method field

    String astack;		// argument stack on entry
    String rstack;		// stack state at time of return

    int max_stack;		// maximum stack used
    int max_locals;		// maximum locals used
    byte code[];		// raw bytecode

    int oflags;			// boolean OR of flags of all reachable opcodes
    Instr instrs[];		// instruction list
    short pcmap[];		// pc to instruction mapping
    Handler handlers[];		// exception table


    //  Bit vectors record exactly which local variables are used, by JVM type.
    //  Only stores are recorded; if it's not stored, it can't be loaded.
    //  Don't forget the implicit stores of method parameters, though.

    static BitSet[] varused;		// local variables used, by type
    static int[] maxvar;		// highest number used, by type

    //  JVM types that appear on the stack.
    static BitSet stktypes;			// types used on stack



// manifest constants

static final int CHAR_INDX_SIZE = 128;  // size of array indexed by (C) chars

// ZZZ: changed for Thor
static final String JVM_TYPES="ailfd";  // Java Virtual Machine datatypes



//  new Method(cl, fl) -- build method struct from classdata and field
//
//  for abstract or native methods, many fields are left uninitialized.

Method(ClassData cl, Field fl)
    throws ClassFormatError
{
    int c, i, n;
    Instr ins;
    Handler h;

    this.cl = cl;			// record containing class
    this.fl = fl;			// record corresponding Field struct
    sigscan();				// always scan signature info

    // check for abstract or native method, and bail out if no code
    if ((fl.access & (ClassData.ACC_ABSTRACT | ClassData.ACC_NATIVE)) != 0)
    	return;				// nothing more to do

    // initialize
    stktypes = new BitSet();
    maxvar = new int[CHAR_INDX_SIZE];
    varused = new BitSet[CHAR_INDX_SIZE];
    for (i = 0; i < JVM_TYPES.length(); i++) {
	c = (int)JVM_TYPES.charAt(i);
	maxvar[c] = -1;
	varused[c] = new BitSet();
    }

    // mark variables that are passed as arguments
    for (i = 0; i < astack.length(); i++)  {
        c = astack.charAt(i);
	if (c == 'x') {
	    markvar((char)astack.charAt(i+1), i);
	    i++;
	} else
	    markvar((char)c, i);
    }

    // find and unpack the "Code" attribute
    byte[] cattr = Attribute.find(fl.attributes, "Code");
    if (cattr == null)
	throw new ClassFormatError(
	    "no Code attribute for " + cl.name + "." + fl.name);
    ByteArrayInputStream b = new ByteArrayInputStream(cattr);
    DataInputStream d = new DataInputStream(b);
    try {
	max_stack = d.readUnsignedShort();
	max_locals = d.readUnsignedShort();

	code = new byte[d.readInt()];
	d.readFully(code);

	// exception table
	handlers = new Handler[d.readUnsignedShort()];
	for (i = 0; i < handlers.length; i++) {
	    h = new Handler();
	    h.start = d.readUnsignedShort();
	    h.end = d.readUnsignedShort();
	    h.jump = d.readUnsignedShort();
	    h.type = d.readUnsignedShort();
	    handlers[i] = h;
	}
    } catch (IOException e) {
	throw new ClassFormatError(
	    "Code attribute of " + cl.name + "." + fl.name);
    }

    // construct instruction list
    Vector insvec = new Vector();
    pcmap = new short[code.length];

    for (int pc = 0; pc < code.length; pc += ins.length) {
	pcmap[pc] = (short)insvec.size();
	ins = new Instr(code, pc, cl.constants);
	insvec.addElement(ins);
    }
    instrs = new Instr[insvec.size()];
    insvec.copyInto(instrs);

    // set boundary bits for instructions at ends of exception ranges
    for (i = 0; i < handlers.length; i++) {
	h = handlers[i];
	instrs[pcmap[h.start]].isBoundary = true;	// mark first instr
	if (h.end < code.length)
	    instrs[pcmap[h.end]].isBoundary = true;	// mark last instr
    }
 
    // number exception range segments, marking each instr by segment number
    if (handlers.length > 0) {
	for (i = n = 0; i < instrs.length; i++) {
	    ins = instrs[i];
	    if (ins.isBoundary)
		n++;
	    ins.xseg = n;
	}
    }

    // mark directly reachable code, noting jump targets and stack depths
    mark(0, "", 0);			// begin at entry, with empty stack

    // mark exception handling code
    for (i = 0; i < handlers.length; i++)
	mark(handlers[i].jump, "a", -1);

    // assign labels to jump targets
    n = 0;
    for (i = 0; i < instrs.length; i++) {
	ins = instrs[i];
	if (ins.isTarget)
	    ins.label = n++;
    }
}



//  sigscan() -- scan method signature to set astack and rstack

private void sigscan()
{
    String s = fl.signature;			// signature being scanned
    StringBuffer a = new StringBuffer();	// arg stack being built
    int i = 1;					// current position (skip "(")

    if ((fl.access & ClassData.ACC_STATIC) == 0) // instance function?
	a.append('a');				// "self" parameter

loop:
    for (i = 1; ; i++) {
	switch (s.charAt(i)) {

	    case 'L':				// object
		i = s.indexOf(';', i);
		a.append('a');
		break;

	    case '[':				// array
		while (s.charAt(i) == '[')
		    i++;
		if (s.charAt(i) == 'L')
		    i = s.indexOf(';', i);
		a.append('a');
		break;

	    // primitive types
	    case 'B':  a.append('i');   break;
	    case 'C':  a.append('i');   break;
	    case 'D':  a.append("xd");  break;
	    case 'F':  a.append('f');   break;
	    case 'I':  a.append('i');   break;
	    case 'J':  a.append("xl");  break;
	    case 'S':  a.append('i');   break;
	    case 'Z':  a.append('i');   break;

	    case ')':  break loop;		// end of parameters
	}
    }

    astack = a.toString();

    switch (s.charAt(++i)) {
	case 'V':  rstack = "";    break;	// void
	case 'B':  rstack = "i";   break;
	case 'C':  rstack = "i";   break;
	case 'D':  rstack = "xd";  break;
	case 'F':  rstack = "f";   break;
	case 'I':  rstack = "i";   break;
	case 'J':  rstack = "xl";  break;
	case 'S':  rstack = "i";   break;
	case 'Z':  rstack = "i";   break;
	default:   rstack = "a";   break;
    }
}



//  mark(pc, stack, segnum) -- mark reachable code, maintaining stack state
//
//  If segnum differs from segment number of instruction at pc, the first
//  instruction (at pc) is also marked as a boundary instruction.
//
//  Also marks local variables referenced by STORE-class instructions

private void mark(int pc, String stack, int segnum)
{
    int i = pcmap[pc];
    Instr ins = instrs[i];
    ins.isTarget = true;
    if (ins.xseg != segnum)
	ins.isBoundary = true;

    while (true) {
	ins = instrs[i++];

	if (ins.isReached) {			// if already processed
	    if (!ins.before.equals(stack))	// sanity check
		throw new VerifyError( "stack mismatch: pc=" + pc +
		   " stack1=" + ins.before + " stack2=" + stack);
	    return;
	}

	ins.isReached = true;			// mark instruction as processed
	oflags |= ins.opcode.flags;		// accumulate flag bits

	ins.before = stack;
	if (stack.length() > 0) {
	    char c = stack.charAt(stack.length() - 1);
	    stktypes.set((int)c);
	    if (ins.opcode.kind == Opcode.STORE)
		markvar(c, ins.opcode.var + ins.val);
	}

	stack = ins.after = ins.nextStack();

	if ((ins.opcode.flags & Opcode.JSRI) != 0) {	// if JSR instruction
	    mark(ins.val, stack + "i", ins.xseg);
	    					// mark JSR routine with arg
	    mark(ins.pc + ins.length, stack, -1);
	    					// mark successor as jump tgt
	} else if ((ins.opcode.flags & Opcode.PC) != 0) // if other jump
	    mark(ins.val, stack, ins.xseg); 	// mark target instrs

	if ((ins.opcode.flags & Opcode.SWCH) != 0) {	// if switch instruction
	    mark(ins.more[0], stack, ins.xseg);		// mark default
	    for (int j = 3; j < ins.more.length; j++) {
		mark(ins.more[j], stack, ins.xseg);	// mark other labels
		if (ins.opcode.kind == Opcode.LKPSW)
		    j++;			// skip tag if lookupswitch
	    }
	}

	if ((ins.opcode.flags & Opcode.NFT) != 0)
	    break;				// opcode never falls through
	else
	    ins.succ = instrs[i];		// note sequential successor
    }
}



//  markvar(jtype, n) -- mark a local variable as being used.

private void markvar(char jtype, int n)
{
    int i = (int)jtype;
    if (maxvar[i] < n)
	maxvar[i] = n;
    varused[i].set(n);
}



//  target(pc) -- return label corresponding to jump target given in pc terms

private int target(int pc)
{
    return instrs[pcmap[pc]].label;
}



} // class Method
