//  Opcode.java -- definitions of Java bytecode operations

//  Copyright 1996, 1997 Arizona Board of Regents;
//  see COPYRIGHT file for details.



package toba;

import java.io.*;



class Opcode {



    // instance variables

    int length;		// instruction length (0 if variable)
    int kind;		// kind (classification) of operation
    int flags;		// flag bits
    int var;		// variant for distinguishing similar ops
    int pop;		// number of words popped
    String push;	// number and types of words pushed
    String name;	// mnemonic name
    String opr;		// operator for use in generated code



// global opcode table

static Opcode table[];	



// instruction flags

static final int I8   = 0x0001;		// instr has 8-bit signed arg
static final int I16  = 0x0002;		// instr has 16-bit signed arg
static final int I32  = 0x0004;		// instr has 16-bit signed arg
static final int U8   = 0x0010;		// instr has 8-bit unsigned arg
static final int U16  = 0x0020;		// instr has 16-bit unsigned arg
static final int MORE = 0x0080;		// instr has further additional args

static final int NFT  = 0x0100;		// no fallthrough to following instr
static final int PC   = 0x0200;		// argument is PC relative
static final int CTAB = 0x0800;		// argument indexes constant table

static final int UNS  = 0x1000;		// unsigned variant of a shift
static final int SWCH = 0x2000;		// switch instruction
static final int INST = 0x4000;		// instanceof (including cast & aastore)
static final int JSRI = 0x8000;		// JSR instruction flag



// newarray types

static final int T_BOOLEAN	=  4;
static final int T_CHAR		=  5;
static final int T_FLOAT	=  6;
static final int T_DOUBLE	=  7;
static final int T_BYTE		=  8;
static final int T_SHORT	=  9;
static final int T_INT		= 10;
static final int T_LONG		= 11;



// symbolic definitions of operations and classes of operations
// (these generally don't have to match Java bytecode codes, although many do)

static final int ACONST = 1;	// aconst_null (constant load)
static final int CONST = 2; 	// constant load
static final int LDC = 18;  	// ldc, ldc2, ldc2_w
static final int LOAD = 21; 	// load operation
static final int STORE = 54;	// store operation

static final int IINC = 132;	// iinc
static final int GETS = 178;	// getstatic
static final int PUTS = 179;	// putstatic
static final int GETF = 180;	// getfield
static final int PUTF = 181;	// putfield

static final int NEW = 187;	// new
static final int ACAST = 192;	// checkcast
static final int INSTC = 193;	// instanceof

static final int NEWA = 188;	// newarray
static final int ANEWA = 189;	// anewarray
static final int MNEWA = 197;	// multianewarray
static final int ALEN = 190;	// arraylength
static final int ALOAD = 46;	// array load operation
static final int ASTOR = 79;	// array store operation

static final int NOP = 87;	// no code: nop, pop, pop2
static final int DUP = 89;	// dup
static final int DUPX1 = 90;	// dup_x1
static final int DUPX2 = 91;	// dup_x2
static final int DUP2 = 92;	// dup2
static final int D2X1 = 93;	// dup2_x1
static final int D2X2 = 94;	// dup2_x2
static final int SWAP = 95;	// swap

static final int UNOP = 116;	// unary operator
static final int FTOI = 143;	// f2l, d2l, f2i, d2i conversion
static final int BINOP = 96;	// binary operator
static final int DIVOP = 108;	// division operator
static final int FREM = 114;	// floating (or double) remainder
static final int SHIFT = 121;	// shift operator

static final int CMP = 149;	// compare without branching
static final int IFZRO = 153;	// compare to zero and branch
static final int IFCMP = 159;	// compare and branch

static final int TBLSW = 170;	// tableswitch
static final int LKPSW = 171;	// lookupswitch
static final int GOTO = 167;	// goto, goto_w
static final int JSR = 168;	// jsr, jsr_w
static final int RET = 169;	// ret, ret_w

static final int IVIRT = 182;	// invokevirtual
static final int INONV = 183;	// invokenonvirtual
static final int ISTAT = 184;	// invokestatic
static final int IINTR = 185;	// invokeinterface
static final int RETV = 172;	// return value (ireturn, areturn, etc.)
static final int RETRN = 177;	// return, no value

static final int THROW = 191;	// athrow
static final int MENTR = 194;	// monitorenter
static final int MEXIT = 195;	// monitorexit

static final int WIDE = 196;	// wide (special instruction form)




//  opcode table initialization

static {

    table = new Opcode[256];

    //  op len kind  pop push   name               var flags

    op(  1, 1, ACONST,0, "a",   "aconst_null",      1, 0); 
    op(  2, 1, CONST, 0, "i",   "iconst_m1",       -1, 0);
    op(  3, 1, CONST, 0, "i",   "iconst_0",         0, 0);
    op(  4, 1, CONST, 0, "i",   "iconst_1",         1, 0);
    op(  5, 1, CONST, 0, "i",   "iconst_2",         2, 0);
    op(  6, 1, CONST, 0, "i",   "iconst_3",         3, 0);
    op(  7, 1, CONST, 0, "i",   "iconst_4",         4, 0);
    op(  8, 1, CONST, 0, "i",   "iconst_5",         5, 0);
    op(  9, 1, CONST, 0, "xl",  "lconst_0",         0, 0);
    op( 10, 1, CONST, 0, "xl",  "lconst_1",         1, 0);
    op( 11, 1, CONST, 0, "f",   "fconst_0",         0, 0);
    op( 12, 1, CONST, 0, "f",   "fconst_1",         1, 0);
    op( 13, 1, CONST, 0, "f",   "fconst_2",         2, 0);
    op( 14, 1, CONST, 0, "xd",  "dconst_0",         0, 0);
    op( 15, 1, CONST, 0, "xd",  "dconst_1",         1, 0);
    op( 16, 2, CONST, 0, "i",   "bipush",           0, I8);
    op( 17, 3, CONST, 0, "i",   "sipush",           0, I16);
    op( 18, 2, LDC,   0, "*",   "ldc",              0, CTAB + U8);
    op( 19, 3, LDC,   0, "*",   "ldc_w",            0, CTAB + U16);
    op( 20, 3, LDC,   0, "x*",  "ldc2_w",           0, CTAB + U16);

    op( 21, 2, LOAD,  0, "i",   "iload",            0, U8);
    op( 26, 1, LOAD,  0, "i",   "iload_0",          0, 0);
    op( 27, 1, LOAD,  0, "i",   "iload_1",          1, 0);
    op( 28, 1, LOAD,  0, "i",   "iload_2",          2, 0);
    op( 29, 1, LOAD,  0, "i",   "iload_3",          3, 0);
    op( 22, 2, LOAD,  0, "xl",  "lload",            0, U8);
    op( 30, 1, LOAD,  0, "xl",  "lload_0",          0, 0);
    op( 31, 1, LOAD,  0, "xl",  "lload_1",          1, 0);
    op( 32, 1, LOAD,  0, "xl",  "lload_2",          2, 0);
    op( 33, 1, LOAD,  0, "xl",  "lload_3",          3, 0);
    op( 23, 2, LOAD,  0, "f",   "fload",            0, U8);
    op( 34, 1, LOAD,  0, "f",   "fload_0",          0, 0);
    op( 35, 1, LOAD,  0, "f",   "fload_1",          1, 0);
    op( 36, 1, LOAD,  0, "f",   "fload_2",          2, 0);
    op( 37, 1, LOAD,  0, "f",   "fload_3",          3, 0);
    op( 24, 2, LOAD,  0, "xd",  "dload",            0, U8);
    op( 38, 1, LOAD,  0, "xd",  "dload_0",          0, 0);
    op( 39, 1, LOAD,  0, "xd",  "dload_1",          1, 0);
    op( 40, 1, LOAD,  0, "xd",  "dload_2",          2, 0);
    op( 41, 1, LOAD,  0, "xd",  "dload_3",          3, 0);
    op( 25, 2, LOAD,  0, "a",   "aload",            0, U8);
    op( 42, 1, LOAD,  0, "a",   "aload_0",          0, 0);
    op( 43, 1, LOAD,  0, "a",   "aload_1",          1, 0);
    op( 44, 1, LOAD,  0, "a",   "aload_2",          2, 0);
    op( 45, 1, LOAD,  0, "a",   "aload_3",          3, 0);

    op( 54, 2, STORE, 1, "",    "istore",           0, U8);
    op( 59, 1, STORE, 1, "",    "istore_0",         0, 0);
    op( 60, 1, STORE, 1, "",    "istore_1",         1, 0);
    op( 61, 1, STORE, 1, "",    "istore_2",         2, 0);
    op( 62, 1, STORE, 1, "",    "istore_3",         3, 0);
    op( 55, 2, STORE, 2, "",    "lstore",           0, U8);
    op( 63, 1, STORE, 2, "",    "lstore_0",         0, 0);
    op( 64, 1, STORE, 2, "",    "lstore_1",         1, 0);
    op( 65, 1, STORE, 2, "",    "lstore_2",         2, 0);
    op( 66, 1, STORE, 2, "",    "lstore_3",         3, 0);
    op( 56, 2, STORE, 1, "",    "fstore",           0, U8);
    op( 67, 1, STORE, 1, "",    "fstore_0",         0, 0);
    op( 68, 1, STORE, 1, "",    "fstore_1",         1, 0);
    op( 69, 1, STORE, 1, "",    "fstore_2",         2, 0);
    op( 70, 1, STORE, 1, "",    "fstore_3",         3, 0);
    op( 57, 2, STORE, 2, "",    "dstore",           0, U8);
    op( 71, 1, STORE, 2, "",    "dstore_0",         0, 0);
    op( 72, 1, STORE, 2, "",    "dstore_1",         1, 0);
    op( 73, 1, STORE, 2, "",    "dstore_2",         2, 0);
    op( 74, 1, STORE, 2, "",    "dstore_3",         3, 0);
    op( 58, 2, STORE, 1, "",    "astore",           0, U8);
    op( 75, 1, STORE, 1, "",    "astore_0",         0, 0);
    op( 76, 1, STORE, 1, "",    "astore_1",         1, 0);
    op( 77, 1, STORE, 1, "",    "astore_2",         2, 0);
    op( 78, 1, STORE, 1, "",    "astore_3",         3, 0);

    op(132, 3, IINC,  0, "",    "iinc",             0, U8 + MORE);
    op(178, 3, GETS,  0, "*",   "getstatic",        0, CTAB + U16);
    op(179, 3, PUTS,  1, "",    "putstatic",        0, CTAB + U16);
    op(180, 3, GETF,  1, "*",   "getfield",         0, CTAB + U16);
    op(181, 3, PUTF,  2, "",    "putfield",         0, CTAB + U16);

    op(187, 3, NEW,   0, "a",   "new",              0, CTAB + U16);
    op(192, 3, ACAST, 0, "",    "checkcast",        0, INST + CTAB + U16);
    op(193, 3, INSTC, 1, "i",   "instanceof",       0, INST + CTAB + U16);

    op(188, 2, NEWA,  1, "a",   "newarray",         0, U8);
    op(189, 3, ANEWA, 1, "a",   "anewarray",        0, CTAB + U16);
    op(197, 4, MNEWA, 0, "a",   "multianewarray",   0, CTAB + U16 + MORE);
    op(190, 1, ALEN,  1, "i",   "arraylength",      0, 0);
    op( 46, 1, ALOAD, 2, "i",   "iaload",           0, 0);
    op( 47, 1, ALOAD, 2, "xl",  "laload",           0, 0);
    op( 48, 1, ALOAD, 2, "f",   "faload",           0, 0);
    op( 49, 1, ALOAD, 2, "xd",  "daload",           0, 0);
    op( 50, 1, ALOAD, 2, "a",   "aaload",           0, 0);
    op( 51, 1, ALOAD, 2, "i",   "baload",           0, 0);
    op( 52, 1, ALOAD, 2, "i",   "caload",           0, 0);
    op( 53, 1, ALOAD, 2, "i",   "saload",           0, 0);
    op( 79, 1, ASTOR, 3, "",    "iastore",          0, 0);
    op( 80, 1, ASTOR, 4, "",    "lastore",          0, 0);
    op( 81, 1, ASTOR, 3, "",    "fastore",          0, 0);
    op( 82, 1, ASTOR, 4, "",    "dastore",          0, 0);
    op( 83, 1, ASTOR, 3, "",    "aastore",          0, INST);
    op( 84, 1, ASTOR, 3, "",    "bastore",          0, 0);
    op( 85, 1, ASTOR, 3, "",    "castore",          0, 0);
    op( 86, 1, ASTOR, 3, "",    "sastore",          0, 0);

    op(  0, 1, NOP,   0, "",    "nop",              0, 0);
    op( 87, 1, NOP,   1, "",    "pop",              0, 0);
    op( 88, 1, NOP,   2, "",    "pop2",             0, 0);
    op( 89, 1, DUP,   0, "*",   "dup",              0, 0);
    op( 90, 1, DUPX1, 0, "*",   "dup_x1",           0, 0);
    op( 91, 1, DUPX2, 0, "*",   "dup_x2",           0, 0);
    op( 92, 1, DUP2,  0, "**",  "dup2",             0, 0);
    op( 93, 1, D2X1,  0, "**",  "dup2_x1",          0, 0);
    op( 94, 1, D2X2,  0, "**",  "dup2_x2",          0, 0);
    op( 95, 1, SWAP,  0, "**",  "swap",             0, 0);

    op(116, 1, UNOP,  1, "i",   "ineg",             "-", 0);
    op(117, 1, UNOP,  2, "xl",  "lneg",             "-", 0);
    op(118, 1, UNOP,  1, "f",   "fneg",             "-", 0);
    op(119, 1, UNOP,  2, "xd",  "dneg",             "-", 0);
    op(133, 1, UNOP,  1, "xl",  "i2l",              "", 0);
    op(134, 1, UNOP,  1, "f",   "i2f",              "", 0);
    op(135, 1, UNOP,  1, "xd",  "i2d",              "", 0);
    op(136, 1, UNOP,  2, "i",   "l2i",              "", 0);
    op(137, 1, UNOP,  2, "f",   "l2f",              "", 0);
    op(138, 1, UNOP,  2, "xd",  "l2d",              "", 0);
    op(139, 1, FTOI,  1, "i",   "f2i",              "dtoi", 0);
    op(140, 1, FTOI,  1, "xl",  "f2l",              "dtol", 0);
    op(141, 1, UNOP,  1, "xd",  "f2d",              "", 0);
    op(142, 1, FTOI,  2, "i",   "d2i",              "dtoi", 0);
    op(143, 1, FTOI,  2, "xl",  "d2l",              "dtol", 0);
    op(144, 1, UNOP,  2, "f",   "d2f",              "", 0);
    op(145, 1, UNOP,  1, "i",   "int2byte",         "(Byte)", 0);
    op(146, 1, UNOP,  1, "i",   "int2char",         "(Char)", 0);
    op(147, 1, UNOP,  1, "i",   "int2short",        "(Short)", 0);

    op( 96, 1, BINOP, 2, "i",   "iadd",             " + ", 0);
    op( 97, 1, BINOP, 4, "xl",  "ladd",             " + ", 0);
    op( 98, 1, BINOP, 2, "f",   "fadd",             " + ", 0);
    op( 99, 1, BINOP, 4, "xd",  "dadd",             " + ", 0);
    op(100, 1, BINOP, 2, "i",   "isub",             " - ", 0);
    op(101, 1, BINOP, 4, "xl",  "lsub",             " - ", 0);
    op(102, 1, BINOP, 2, "f",   "fsub",             " - ", 0);
    op(103, 1, BINOP, 4, "xd",  "dsub",             " - ", 0);
    op(104, 1, BINOP, 2, "i",   "imul",             " * ", 0);
    op(105, 1, BINOP, 4, "xl",  "lmul",             " * ", 0);
    op(106, 1, BINOP, 2, "f",   "fmul",             " * ", 0);
    op(107, 1, BINOP, 4, "xd",  "dmul",             " * ", 0);
    op(108, 1, DIVOP, 2, "i",   "idiv",             " / ", 0);
    op(109, 1, DIVOP, 4, "xl",  "ldiv",             " / ", 0);
    op(110, 1, BINOP, 2, "f",   "fdiv",             " / ", 0);
    op(111, 1, BINOP, 4, "xd",  "ddiv",             " / ", 0);
    op(112, 1, DIVOP, 2, "i",   "irem",             " % ", 0);
    op(113, 1, DIVOP, 4, "xl",  "lrem",             " % ", 0);
    op(114, 1, FREM,  2, "f",   "frem",             " % ", 0);
    op(115, 1, FREM,  4, "xd",  "drem",             " % ", 0);
    op(126, 1, BINOP, 2, "i",   "iand",             " & ", 0);
    op(127, 1, BINOP, 4, "xl",  "land",             " & ", 0);
    op(128, 1, BINOP, 2, "i",   "ior",              " | ", 0);
    op(129, 1, BINOP, 4, "xl",  "lor",              " | ", 0);
    op(130, 1, BINOP, 2, "i",   "ixor",             " ^ ", 0);
    op(131, 1, BINOP, 4, "xl",  "lxor",             " ^ ", 0);
    op(120, 1, SHIFT, 2, "i",   "ishl",             " << ", 0);
    op(122, 1, SHIFT, 2, "i",   "ishr",             " >> ", 0);
    op(124, 1, SHIFT, 2, "i",   "iushr",            " >> ", UNS);
    op(121, 1, SHIFT, 3, "xl",  "lshl",             " << ", 0);
    op(123, 1, SHIFT, 3, "xl",  "lshr",             " >> ", 0);
    op(125, 1, SHIFT, 3, "xl",  "lushr",            " >> ", UNS);

    op(148, 1, CMP,   4, "i",   "lcmp",             0, 0);
    op(149, 1, CMP,   2, "i",   "fcmpl",            -1, 0);
    op(150, 1, CMP,   2, "i",   "fcmpg",            +1, 0);
    op(151, 1, CMP,   4, "i",   "dcmpl",            -1, 0);
    op(152, 1, CMP,   4, "i",   "dcmpg",            +1, 0);
    op(198, 3, IFZRO, 1, "",    "ifnull",           " == ", PC + I16);
    op(153, 3, IFZRO, 1, "",    "ifeq",             " == ", PC + I16);
    op(155, 3, IFZRO, 1, "",    "iflt",             " < ",  PC + I16);
    op(158, 3, IFZRO, 1, "",    "ifle",             " <= ", PC + I16);
    op(154, 3, IFZRO, 1, "",    "ifne",             " != ", PC + I16);
    op(199, 3, IFZRO, 1, "",    "ifnonnull",        " != ", PC + I16);
    op(157, 3, IFZRO, 1, "",    "ifgt",             " > ",  PC + I16);
    op(156, 3, IFZRO, 1, "",    "ifge",             " >= ", PC + I16);
    op(159, 3, IFCMP, 2, "",    "if_icmpeq",        " == ", PC + I16);
    op(160, 3, IFCMP, 2, "",    "if_icmpne",        " != ", PC + I16);
    op(161, 3, IFCMP, 2, "",    "if_icmplt",        " < ",  PC + I16);
    op(163, 3, IFCMP, 2, "",    "if_icmpgt",        " > ",  PC + I16);
    op(164, 3, IFCMP, 2, "",    "if_icmple",        " <= ", PC + I16);
    op(162, 3, IFCMP, 2, "",    "if_icmpge",        " >= ", PC + I16);
    op(165, 3, IFCMP, 2, "",    "if_acmpeq",        " == ", PC + I16);
    op(166, 3, IFCMP, 2, "",    "if_acmpne",        " != ", PC + I16);

    op(170, 0, TBLSW, 1, "",    "tableswitch",      0, NFT + SWCH + MORE);
    op(171, 0, LKPSW, 1, "",    "lookupswitch",     0, NFT + SWCH + MORE);
    op(167, 3, GOTO,  0, "",    "goto",             0, NFT + PC + I16);
    op(200, 5, GOTO,  0, "",    "goto_w",           0, NFT + PC + I32);
    op(168, 3, JSR,   0, "",    "jsr",              0, JSRI + PC + I16);
    op(201, 5, JSR,   0, "",    "jsr_w",            0, JSRI + PC + I32);
    op(169, 2, RET,   0, "",    "ret",              0, NFT + U8);
    op(209, 3, RET,   0, "",    "ret_w",            0, NFT + U16);

    op(182, 3, IVIRT, 1, "",    "invokevirtual",    0, CTAB + U16);
    op(183, 3, INONV, 1, "",    "invokenonvirtual", 0, CTAB + U16);
    op(184, 3, ISTAT, 0, "",    "invokestatic",     0, CTAB + U16);
    op(185, 5, IINTR, 1, "",    "invokeinterface",  0, CTAB + U16);
    op(172, 1, RETV,  0, "",    "ireturn",          0, NFT);
    op(173, 1, RETV,  0, "",    "lreturn",          0, NFT);
    op(174, 1, RETV,  0, "",    "freturn",          0, NFT);
    op(175, 1, RETV,  0, "",    "dreturn",          0, NFT);
    op(176, 1, RETV,  0, "",    "areturn",          0, NFT);
    op(177, 1, RETRN, 0, "",    "return",           0, NFT);

    op(191, 1, THROW, 1, "",    "athrow",           0, NFT);
    op(202, 1, NOP,   0, "",    "breakpoint",       0, 0);
    op(194, 1, MENTR, 1, "",    "monitorenter",     0, 0);
    op(195, 1, MEXIT, 1, "",    "monitorexit",      0, 0);

    op(196, 0, WIDE,  0, "",    "wide",	            0, 0);
}


//  op(code,length,kind,pop,push,name,var,flags) -- initialize opcode entry

static private void op (int code, int length, int kind, int pop, String push,
    String name, int var, int flags)
{
    Opcode o = new Opcode();
    table[code] = o;

    o.length = length;
    o.kind = kind;
    o.pop = pop;
    o.push = push;
    o.name = name;
    o.var = var;
    o.flags = flags;
}


//  op(code,length,kind,pop,push,name,opr,flags) -- initialize special entry
//
//  Used with a String "opr" in place of the "var" parameter.

static private void op (int code, int length, int kind, int pop, String push,
    String name, String opr, int flags)
{
    op(code, length, kind, pop, push, name, 0, flags);
    table[code].opr = opr;
}



} // class Opcode
