//  Repr.java -- methods dealing with C representations

//  Copyright 1996, 1997 Arizona Board of Regents;
//  see COPYRIGHT file for details.




package toba;

import java.io.*;

class Repr {



static final int DECLS_PER_LINE = 15;	// variables per declaration line
static final int GCHARS_PER_LINE = 18;	// generated char consts on one line
static final int GCHARS_PER_CMMT = 40;	// max chars of string gen in comment


//  isQuotable(c) -- is character c reasonably specified as 'c'?

static boolean isQuotable(char c)
{
    return c >= ' ' && c <= '~' && c != '\'' && c != '\\';
}



//  con(m, c) -- return C representation of constant c in method m.
//
//  Valid for INT, LONG, FLOAT, DOUBLE, or STRING constants
//  (assuming that necessary static constants were generated earlier).

static String con(Method m, Constant c)
{
    switch (c.tag) {

	case Constant.INTEGER:
	    if (((Integer)c.value).intValue() == Integer.MIN_VALUE)
		return "0x80000000";
	    else
		return c.value.toString();

	case Constant.LONG:
	    long v = ((Long)c.value).longValue();
	    if (v > (long)Integer.MIN_VALUE && v < (long)Integer.MAX_VALUE)
		return c.value.toString();
	    // now we must use "ANSI C" to construct a possibly "long long" val
	    int lh = (int)(v >>> 32);
	    int rh = (int)v;
	    return "((((Long)0x" + Integer.toHexString(lh) +
		"u) << 32) | ((Long)0x" + Integer.toHexString(rh) + "u))";

	case Constant.FLOAT:
	case Constant.DOUBLE:
	    if (simplefloat(((Number)c.value).doubleValue()))
		return c.value.toString();
	    else if (c.tag == Constant.FLOAT)
		return "fc" + c.index + ".v";
	    else /* c.tag == Constant.DOUBLE */
		return "dc" + c.index + ".v";

	case Constant.STRING:
	    return "(Object)self_P->" + m.cl.cname + "_V->strings[" + CFile.strref(c) + "]";

	default:
	    return c.value.toString();
    }
}



//  simplefloat(d) -- is d represented accurately by toString()?
//
//  This method errs on the side of caution, but is good enough
//  to accept many exact numbers such as 23.0, 1.5, and 0.125.

static boolean simplefloat(double d)
{
    if (Double.isNaN(d) || Double.isInfinite(d))
	return false;
    if (Double.doubleToLongBits(d) == 0x8000000000000000L)
	return false;			// negative zero

    if (d % 1.0 == 0.0 && d < 1000000.0 && d > -1000000.0)
	return true;
    if (d % 0.03125 == 0.0 && d < 10.0 && d > -10.0)
	return true;

    return false;
}



//  ctype_local_vars(c) -- return C type indicated by character c.
//
//  upper case characters are Java signature characters.
//  lower case characters are JVM datatypes.
//
// These C types are used for local vars in the generated C code

static String ctype_local_vars(char c)
{
    switch (c) {
	case 'B':  case 'b':  return "Byte";
	case 'C':  case 'c':  return "Char";
	case 'D':  case 'd':  return "Double";
	case 'F':  case 'f':  return "Float";
	case 'I':  case 'i':  return "Int";
	case 'J':  case 'l':  return "Long";
	case 'S':             return "Short";
	case 'V':             return "Void";
	case 'Z':             return "Boolean";
	default:              return "Object";
    }
}



// ctype_actual(f) - Similar to ctype_local_vars, but returns actual object
// type instead of just "Object"
//
// These are the actual types of objects.  They can be passed as parameters
// where some type is expected.

public static final String ctype_actual(String f) 
{


  switch (f.charAt(0)) {
    
  case 'L': return(Names.hashclass(f.substring(1, f.indexOf(';'))));

  case '[': return "java_lang_Object";

  case '(': case ')': // skip these
    try { return ctype_actual(f.substring(1)); }
    catch (StringIndexOutOfBoundsException e) { return ""; }

  case 'B': case 'b': case 'C': case 'c': case 'D': case 'd': case 'F': 
  case 'f': case 'I': case 'i': case 'J': case 'l': case 'S': case 'V': 
  case 'Z': 
    return(ctype_local_vars(f.charAt(0)));

  default: Trans.abort("Unknown field type"); return("");
  }
}


// ctype_field_slots(f) - Return the type to use in fields structure in Thor
//               Basically, return Shortp for pointers, else do as in ctype_actual


public static final String ctype_field_slots(String f) 
{
  switch (f.charAt(0)) {
    
  case 'L': 
  case '[': return("Shortp");    // Byte  

  default: return(ctype_local_vars(f.charAt(0)));
  }
}



// nextSignature(f) - Return the remaining portion of the string f 
//                    after the C Type
// See ctype_actual(f) above


public static final String nextSignature(String f) 
{
  switch (f.charAt(0)) {
    
  case 'B': 
  case 'C': 
  case 'D': 
  case 'F': 
  case 'I': 
  case 'J': 
  case 'S': 
  case 'V': 
  case 'Z': return f.substring(1);

  case 'L': return f.substring(f.indexOf(';')+1);

  case '[': 
    {
      int i = 0;
      while (f.charAt(i) == '[')
	i++;
      if (f.charAt(i) == 'L')
	i = f.indexOf(';', i);
      return f.substring(i+1);
    }

  case '(': 
  case ')': 
    try {
      return nextSignature(f.substring(1));
    } catch (StringIndexOutOfBoundsException e) {
      return "";
    }

  default: Trans.abort("***: Unknown field descriptor1=" + f);
    return "";   
  }
}


//  rettype(s) -- Compute C return type of Java method given signature.

static String rettype_local_vars(String s)
{
    return ctype_local_vars(s.charAt(s.indexOf(')') + 1));
}




//  retType(s) -- Compute C return type of method with given signature.

static String retType(String s)
{
    return ctype_actual(s.substring(s.lastIndexOf(')') + 1));
}


//  retType(s) -- Compute C return type of given method 

static String retType(Field m)
{
  String s = m.signature;
  return ctype_actual(s.substring(s.lastIndexOf(')') + 1));
}


//  argTypes(m) -- return string of C arg types for method given signature

static String argTypes(String s)
{

  s = s.substring(1, s.lastIndexOf(')'));

  String cSig = "";
  
  if (! s.equals("")) {
    cSig = ctype_actual(s);
    s = nextSignature(s);
  }
  while (! s.equals("")) {
    cSig = cSig.concat(", " + ctype_actual(s));
    s = nextSignature(s);
  }

  return cSig;
}

//  argTypes(m) -- return string of C arg types for given method

static String argTypes(Field m)
{
  String sig;
  if ((m.access & ClassData.ACC_STATIC) != 0) {
    sig = ("(L" + m.selfClass + "_static;");
  } else {
    sig = ("(L" + m.selfClass + ";");
  }
  sig = sig.concat(m.signature.substring(1));
  return argTypes(sig);
}


} // class Repr


