#ifndef _CACHEDIR_H
#define _CACHEDIR_H

// CacheDir tracks the state of segment copies at various FEs.
// Uses a monitor style read-write lock for concurrent access.

#include <iostream.h>
#include "utils/basic.h"
#include "utils/bits.h"
#include "utils/bool_vector.h"
#include "utils/hashfuncs.h"
#include "utils/map.h"
#include "utils/set.h"
#include "utils/valuekey.h"
#include "common/oref.h"
#include "rwlock.h"

class RW_lock;
class Segment_map_info;
class FE_manager;

typedef Ubits8 FE_state;
typedef PtrSet<FE_manager> FE_set; // Set of FEs

typedef Map<IntKey, Segment_map_info*> Seg_map;

// A segment cached at an fe could be in any one of these states
const FE_state Page_absent      = 0;
const FE_state Page_unreparable = 1;
const FE_state Page_reparable   = 2;
const FE_state Page_complete    = 3;

class CacheDir {
  public:

    CacheDir();
    // effects: Creates an empty cache dir.

    FE_state lookup(Uint pageid, FE_manager *fe) const;
    // effects:  Returns the state of "pagid" at "fe".

    Seg_bmap enter(Uint segid, Seg_bmap bmap, FE_manager *fe, FE_state status,
		   bool alloc_if_possible);
    // modifies: this.
    // effects:  Sets the state of all pages in segid that have their bits set
    //           in  "bmap" to be "status" (for the FE fe).
    //           The state could be "Page_absent".
    //           If "alloc_if_possible" is TRUE, allocation rights are granted
    //           to "fe" if possible. A bitmap indicating the allocation
    //           rights of this fe for segid are returned.

    void enter_single_page(Uint pageid, FE_manager *fe, FE_state status);
    // requires: status is not Page_absent
    // modifies: this.
    // effects:  Sets the state of pageid for fe to be status


    void alter(Uint pageid, FE_state old, FE_state now);
    // requires: Neither "old" nor "now" is "absent".
    // modifies: this.
    // effects: Converts the entry for "pageid" in "old" state to "now" state
    //          for all the FEs that have cached pageid in state "old"

    FE_manager* alloc_rights(Uint pageid) const;
    // effects: Returns the FE_manager that has the allocation rights for
    //          pageid. Returns NULL if no FE has allocation rights for the page

    void remove_fe(FE_manager *fe);
    // effects: Removes all entries for "fe". The allocation rights of
    //          fe are also withdrawn

    bool search(Uint pageid, FE_state state, FE_set *avoid_fes,
		FE_manager* &fe) const;
    // requires: state not be Page_absent
    // effects: Sets "fe" to FE not in "avoid_fes" that has an entry for
    //          "pageid" in the given "state".  Returns false iff no such fe.

    void print(FILE *fp = NULL);
    // modifies: fp
    // effects:  Prints human readable contents on "fp"

    /* Methods that may be required later:
    void alter_fe(FE_manager *fe, FE_State old, FE_state now);
    // Requires neither "old" nor "now" is "absent".
    // Modifies this.
    // Converts all entries for "fe" in "old" state to "now" state.

    XXX const* where(Uint seg) const;
    // Returns a table of fe-state pairs for fe's with non-absent "seg".
    // May return 0 if no such fe.
    */

    ~CacheDir();
    
  private:

    Seg_map* segmap; // maps seg to sets of FEs caching it.
    RW_lock *lock;   // read-write lock for concurrent access, used internally

    bool get_seg_bmap(Uint segid, FE_manager *fe, Segment_map_info* &sinfo,
		      Uint &index) const;
    // requires: The cache directory has been locked.
    // modifies: index, sinfo
    // effects:  Searches the cachedir for information on segment "segid"
    //           corresponding to FE "fe". Modifies sinfo to point to refer
    //           to the information for "segid" and index to be the index
    //           in sinfo.states for FE "fe". Returns TRUE if succeeds
    //
    //           If finds information on segment but not on FE, returns FALSE
    //           and sets sinfo to the right information,
    //           Else if does not find any information, returns FALSE and 
    //           sets sinfo to NULL

    void remove_fe_entry(Uint segid, Segment_map_info *sinfo, Uint i);
    // requires: i be a valid entry into the fe_states array of sinfo
    //           sinfo is the entry corresponding to segid in this
    // effects:  Removes the ith entry from fe_states. If the fe_states
    //           array is empty and no FE has allocation rights, removes
    //           the information about the segment segid


    void create_fe_entry(Uint segid, FE_manager *fe, Segment_map_info* &sinfo,
			 Uint &index);
    // requires: There is no entry for <segid, fe> in this. If there is an entry
    //           for segid, then sinfo points to that else is NULL
    //           The cache directory has been locked
    // effects:  Creates an entry for <segid, fe> with all pages initialized to
    //           Page_state. If sinfo is NULL, allocates space for it and sets
    //           it. Also sets index to the index of the array in
    //           sinfo->fe_states where the <segid, fe> entry is


};

#endif /* _CACHEDIR_H */
