// Copyright 1995 Barbara Liskov

/*
\section{Two-phase commit protocol}

This file is the part of the TM that deals with the two-phase commit.

To do:
\begin{itemize}
\item Try to retransmit if sending a message to an OR fails.
\end{itemize}
 */
#include <iostream.h>

#include "utils/fail.h"
#include "utils/Timer.h"
#include "common/transaction.h"
#include "common/or_message.h"

#include "message_stats.h"
#include "or_or_msg.h"
#include "or_send_message.h"
#include "fe_manager.h"
#include "or.h"
#include "or_config.h"
#include "or_manager.h"
#include "tm.h"
#include "tstatus.h"
#include "mm/log.h"

#include <sys/uio.h>

void TM::commit_single_phase(FE_manager *fe, Transaction *tx, Ubits32 req_id) {
    Ubits32 result = commit_local(fe, tx);
    send_commit_result(fe, result, req_id);
}

void TM::commit_multiple(FE_manager *fe, Transaction *tx, Ubits32 req_id) {
    int res;

    if (tx->coordinator == orx->config->ornum) {
	// We are the coordinator of transaction tx
	fe->result = 0;
	res = prepare_coord(fe, tx);

	// If we already know outcome of transaction, 
	// send message to FE, and we are done.
	if (res != 0) {
	    finish_transaction(fe, tx->tid, res, req_id);
	    return;
	}

	// Wait for transaction to complete
	fe->mutex_commit->grab(); {
	    while (fe->result == 0) 
		fe->cond_commit->wait(); // inform_fe signals this
	    res = fe->result;
	} fe->mutex_commit->release();

	if (orx->config->debug_level > 1)
	    printf("FE thread got transaction result; sending it to FE\n");
	finish_transaction(fe, tx->tid, res, req_id);
    } else {
	// We are a participant

	or_or_message or_msg;
	bool forced = TRUE;
	res = prepare_part(fe, tx, forced);

	// Send vote to coordinator
	or_msg.tid = tx->tid;
	if (res == OR_committed) {
	    // Send read only or read/write vote to coordinator
	    if (tx->mos->count() == 0) {
		or_msg.msgtype = OR_VOTE_READ;
	    } 
	    else {
		// Send OK vote and new objects
		or_msg.msgtype = OR_VOTE_OK;

		// If prepare record not forced, send to coordinator
		if (!forced) {
		    Log_Index index  = orx->tstatus->get_log_index(tx->tid);
		    
		    Prepared_Log_Record *pr = 
			(Prepared_Log_Record *) orx->log->fetch(index);
		    
		    if (pr != 0) {
			if (orx->config->debug_level > 1)
			    printf("Sending prepare record to coordinator\n");
			or_msg.msgtype = OR_VOTE_OK_CL;
			or_msg.u.vote_ok.pr = pr;
			or_msg.u.vote_ok.index = index;
		    }
		}
	    }
	    
	    // XXX Assume that sending message succeeds
	    orx->or_managers->send_message(tx->coordinator, &or_msg);
	    if (orx->config->debug_level > 1)
		printf("Sent ok vote to coordinator\n");
	}
	else {
	    // Send abort vote
	    or_msg.msgtype = OR_VOTE_ABORT;
	    
	    // XXX Assume that sending message succeeds
	    orx->or_managers->send_message(tx->coordinator, &or_msg);

	    if (orx->config->debug_level > 1)
		printf("Sent abort vote to coordinator\n");

	    orx->tm->truncate_vqueue_if_needed();
	}
    }
}

void TM::finish_transaction(FE_manager *fe, Tid const& tid, Ubits32 result,
			    Ubits32 reqid) {
    bool done;

    switch (result) {
	case OR_committed:
	    // Send commit to FE
	    done = orx->tstatus->get_fe_objs(tid);
	    send_commit_result(fe, result, reqid);
	    // Remove from tstatus if all acks have arrived
	    if (done)
		orx->tstatus->remove(tid);
	    break;

	default:
	    send_commit_result(fe, result, reqid);
	    break;
    }
}

bool TM::send_commit_result(FE_manager *fe, Ubits32 result, Ubits32 req_id) {
    cumm_send_timer.start();
    send_timer.start();

    if (result == OR_committed)
	msg_stats.or_committed++;
    else
	msg_stats.or_abort++;

    if (orx->config->debug_level > 1) {
	fe->id().print();
	if (result == OR_committed)
	    fprintf(stderr, ": Transaction %d committed\n", fe->tno);
	else 
	    fprintf(stderr, ": Transaction %d aborted. Reason = %d\n",
		    result, fe->tno);
    }

    OR_send_commit_reply_msg msg(result);
    bool success = Send_net_message(&msg, fe->net, req_id);
    send_timer.stop(); send_time = send_timer.elapsed();
    cumm_send_timer.stop();
    return success;
}
