#include <iostream.h>
#include "utils/network.h"
#include "utils/Timer.h"
#include "common/oref.h"
#include "common/ros.h"
#include "common/or_set.h"
#include "common/transaction.h"

#include "or.h"
#include "or_config.h"
#include "fe_manager.h"
#include "tm.h"
#include "inv_set.h"
#include "cachedir.h"
#include "message_stats.h"
#include "or_send_data_msg.h"
#include "or_send_message.h"
#include "fe_recv_message.h"
#include "mm/mm.h"
#include "mm/log.h"
#include "gc/gc.h"

extern Message_Stats msg_stats;

bool FE_recv_get_root_msg::decode(Network* net, Ubits32 msg_id) {
    OR_send_root_msg root_sender;
    return Send_net_message(&root_sender, net, msg_id);
}

bool FE_recv_fetch_msg::decode(Network *net, Ubits32 msg_id) {
    // Struct to read in msg in one shot. XXX assume consistent with decode.
    struct {
	Oref oref;
	Seg_bmap prefetch_bmap;
	Ubits32 alloc_request;
    } msg;

    if (! net->recv_buffer(&msg, sizeof(msg))) return FALSE;
    OR_send_data_msg data_sender(mgr, msg.oref, msg.prefetch_bmap,
				 msg.alloc_request, msg_id);
    return Send_net_message(&data_sender, mgr->net, msg_id);
}

bool FE_recv_commit_msg::decode(Network *net, Ubits32 msg_id) {
    // Receive the commit message
    trans.ros = new Ros(0); // Do not know the size of incoming ROS
    // Maybe we should use some history information?
    // How does the TM on the OR side get to know all this? XXX
    // Maybe FE_commit_msg should be split into send and recv also

    // int bytes = net->bytes_received; // scratch var to record bytes received

    trans.mos = new Modset;
    trans.nos = new Modset;
    trans.participants = new OR_set;
    
    total_timer.reset(); total_timer.start();
    cumm_total_timer.start();
    cumm_recv_timer.start();
    recv_timer.reset(); recv_timer.start();

    if (! (net->recv_ubits32(&is_coord) && trans.decode(net))) {
	perror("Reception of Commit message failed");
	delete trans.ros;
	delete trans.mos;
	delete trans.nos;
	delete trans.participants;
	return TRUE;
    }

    bool read_only = (trans.mos->count() == 0);

    if (orx->config->debug_level > 0) {
	mgr->id().print();
	if (orx->config->debug_level > 1)
	    trans.print(orx->config->debug_level > 2);
	fprintf(stderr, "Commit number %d:\n", mgr->tno);
	fprintf(stderr, "  Read:  %8d objs\n", trans.ros->size());
	fprintf(stderr, "  Write: %8d objs, %8d bytes\n",
		trans.mos->count(),
		trans.mos->rep_size());
	fprintf(stderr, "  New:   %8d objs, %8d bytes\n",
		trans.nos->count(),
		trans.nos->rep_size());
    }

    recv_timer.stop(); recv_time = recv_timer.elapsed();
    cumm_recv_timer.stop();

    if (!mgr) {
	net->address.print();
	fprintf(stderr, " :FE died probably\n");
    }

    if (trans.coordinator == 0 || trans.participants->size() == 1)
	orx->tm->commit_single_phase(mgr, &trans, msg_id);
    else // Multi-OR transaction
	orx->tm->commit_multiple(mgr, &trans, msg_id);

    // Mos and nos are being deleted by  for this transaction
    // when the objects are no longer needed in the log. They
    // are not valid after this point for read/write transactions.
    delete trans.ros;
    delete trans.participants;

    // Delete empty mos and nos for read-only transactions.
    if (read_only) {
      delete trans.mos;
      delete trans.nos;
    }

    // msg_stats.fe_prepare_coord_bytes += net->bytes_received - bytes;
    msg_stats.fe_prepares++;
    total_timer.stop(); total_time = total_timer.elapsed();
    cumm_total_timer.stop();

    if (orx->config->print_time)
	fprintf(stderr, "Recv = %9.5f, Validate = %9.5f, Send = %9.5f, "
	"Total = %9.5f\n", recv_time, validate_time, send_time, total_time);
    return TRUE;
}

bool FE_recv_invalid_ack_msg::decode(Network *net, Ubits32) {
    Ubits32 last_message_seen;  // Last invalidation message seen by FE

    bool success = net->recv_ubits32(&last_message_seen);
    if (!success)
	perror("Invalidation ack not receivedd correctly");
    // Lock the FE information and process the invalidation ack message
    // received by removing the relevant objects from the invalid object set


    mgr->write_lock();   {
        th_assert(last_message_seen <= mgr->max_message,
                  "FE acked bad number for invalid objects");
        mgr->invalid_objs->remove_before(last_message_seen);
    }
    mgr->write_unlock();

    if (orx->config->debug_level)
	fprintf(stderr, "Received Inv Ack. Msg seen: %d\n", last_message_seen);
    return TRUE;
}

bool FE_recv_alloc_msg::decode(Network *net, Ubits32 msg_id) {
    // Nothing to be received from the net currently

    // Get a new segment from the FE and enter it into the cachedir
    int segid =  orx->mm->alloc_new_segment();
    Seg_bmap full_bmap(TRUE);
    Seg_bmap bmap = orx->cachedir->enter(segid, full_bmap, mgr,
					Page_complete, TRUE);

    // Send the allocation grant rights to the FE
    OR_send_alloc_reply_msg amsg(segid, bmap, full_bmap);
    Send_net_message(&amsg, net, msg_id);
    return TRUE;
}

bool FE_recv_trigger_msg::decode(Network *net, Ubits32 msg_id) {
    if (! net->recv_ubits32(&code)) return FALSE;
    switch(code) {
	case FE_trigger_stats:
	    Handle_stat_message(net, msg_id);
	    break;
	case FE_trigger_gc_stats_start:
	    if (gc) gc->start_stats();
	    break;
	case FE_trigger_gc_stats_print:
	    if (gc) gc->print_stats(&cout);
	    break;
	case FE_trigger_or_kill:
	    fprintf(stderr, "NOT draining the log\n");
	    exit(0);
    }
    return TRUE;
}
	
	
