#include "inoutlists.h"
#include "gc_transaction.h"
#include "partition.h"
#include "allocator.h"

#include "common/page.h"
// #include "common/wk_xref.h"
#include "fe/boot/fe_wk_xref.h"

#include "or/or.h"
#include "or/tm.h"
#include "or/mm/mm.h"
#include "or/mm/handle.h"

// DATA STRUCTURE
// An in/outlist is stored as a linked list of block objects.
// An inlist block stores a number of <source_partition, oref> pairs, 
// where oref identifies the translist from source_partition to the
// inlist's partition. 
// Similarly, an outlist block stores <target_partition, oref> pairs. 
// The pair <NULL_PARTITION, NULL_OREF> indicates end of list, while
// <NULL_PARTITION, oref> indicates that the next block is oref.

Inoutlists::Inoutlists() {
    allocator = new Allocator(WK_GCinoutlist_OREF, gc->inoutlist_block_size);
}

Inoutlists::Elements::Elements(Inoutlists const* inoutlists, 
			       bool of_inlist, unsigned for_partition) {
    next_oref = NULL_OREF;
    next_partition = NULL_PARTITION;
    handle = 0;

    Orefs orefs = of_inlist ? inoutlists->inlists : inoutlists->outlists;
    if (for_partition >= (unsigned)orefs.size()) return;
    next_oref = orefs[for_partition];
}

Oref Inoutlists::Elements::get(unsigned &partition) {
    Oref oref; // to return
    do {
	if (next_partition == NULL_PARTITION) {
	    release();
	    if (next_oref == NULL_OREF) return NULL_OREF;
	    handle = orx->mm->fetch(next_oref);
	    if (handle == 0) 
	      fail("Inlist-block not fetched: %d", next_oref);
	    fields = handle->obj()->fields();
	    i = 0;
	    next_partition = fields[i++].uvalue32;
	    next_oref = fields[i++].get_oref();
	}
	partition = next_partition;
	oref = next_oref;
	next_partition = fields[i++].uvalue32;
	next_oref = fields[i++].get_oref();
    } while (oref == NULL_OREF); // skip over dead entries
    return oref;
}

void Inoutlists::Elements::release() {
	if (handle) orx->mm->release(handle);
	handle = 0;
}

Oref Inoutlists::get_translist(unsigned source_p, unsigned target_p) const {
    Elements elems(this, TRUE, target_p);
    Oref oref;
    unsigned partition;
    while ((oref = elems.get(partition)) != NULL_OREF 
	   && partition != source_p);
    elems.release();
    return oref;
}

void Inoutlists::add_translist(unsigned source_p, unsigned target_p, 
			       Oref translist_oref) {
    printf(">>> add_translist, source_p=%d, target_p=%d, \
translist_oref=%d\n", source_p, target_p, translist_oref);//DEBUG
    add_inoutlist(inlists, target_p, source_p, translist_oref);
    add_inoutlist(outlists, source_p, target_p, translist_oref);
    printf("<<< add_translist\n"); //DEBUG
}

void Inoutlists::add_inoutlist(Orefs &inoutlists, unsigned partition1,
			       unsigned partition2, Oref translist_oref) {
    printf(">>> add_inoutlist: partition1=%d, partition2=%d, \
translist_oref=%d, inoutlists.size()=%d\n", //DEBUG
	   partition1, partition2, translist_oref, //DEBUG
	   inoutlists.size()); //DEBUG
    if (inoutlists.size() <= 0) //DEBUG
	printf("(empty)\n"); //DEBUG
    else //DEBUG
	for (int j = 0; j < inoutlists.size(); j++) //DEBUG
	    printf("j=%d, inoutlists[j]=%d\n", j, inoutlists[j]); //DEBUG
    if (st.on) 
	st.set_translist_timer.start();
    if (partition1 >= (unsigned)inoutlists.size()) 
	inoutlists.append(NULL_OREF, partition1 - inoutlists.size() + 1);
    Oref block_oref = inoutlists[partition1], 
	last_block_oref = block_oref;
    int block_size = gc->inoutlist_block_size, i = block_size - 1;
    MM_Handle* handle = 0;
    OR_obj* block = 0;
    OR_slot* fields = 0;
    // Traverse until the end of the list.
    while (block_oref != NULL_OREF) {
	printf("traversing in/outlist oref=%d, NULL_PARTITION=%u(%d)\n", 
	       block_oref, NULL_PARTITION, NULL_PARTITION); //DEBUG
	last_block_oref = block_oref;
	if (handle) 
	    orx->mm->release(handle); // release previous block
	handle = orx->mm->fetch(block_oref);
	if (handle == 0) 
	  fail("In/outlist-block not fetched: %d", block_oref); // fails here
	block = handle->obj();
	fields = block->fields();
	i = 0;
	while (fields[i].uvalue32 != NULL_PARTITION) {
	    printf("in/outlist entry: partition=%u(%d), oref=%d\n", 
		   fields[i].uvalue32, (int)fields[i].uvalue32, 
		   fields[i+1].get_oref()); //DEBUG
	    i += 2;
	}
	block_oref = fields[++i].get_oref();
	printf("reached NULL_PARTITION, oref=%d\n", block_oref); //DEBUG
    } // while (block_oref != NULL_OREF)
    th_assert(((fields == NULL) && (i == block_size - 1)) ||
	      ((fields[i - 1].uvalue32 == NULL_PARTITION) &&
			(fields[i].get_oref() == NULL_OREF)),
	      "Inoutlists assertion");
    if (block) {
	block = allocator->copy(block, last_block_oref);
	fields = block->fields();
	orx->mm->release(handle);
    }
    if (i == block_size - 1) {
	st.blocks++;
	block = allocator->alloc(block_oref);
	printf("inoutlists[partition1]=%d\n", 
	       inoutlists[partition1]); //DEBUG
	if (inoutlists[partition1] == NULL_OREF) 
	    inoutlists[partition1] = block_oref;
	else 
	    fields[i].set_oref(block_oref);
	fields = block->fields();
	i = 1;
    }
    fields[i-1].uvalue32 = partition2;
    fields[i].set_oref(translist_oref);
    fields[i+1].uvalue32 = NULL_PARTITION;
    fields[i+2].set_oref(NULL_OREF);

    if (st.on) st.set_translist_timer.stop();
///////////////////////////////////////////////////////////////////////////////
    if (inoutlists.size() <= 0)
	printf("(empty)\n"); //DEBUG
    else
	for (int j = 0; j < inoutlists.size(); j++)
	    printf("j=%d, inoutlists[j]=%d\n", j, inoutlists[j]); //DEBUG
    printf("<<< add_inoutlist\n"); //DEBUG
}

void Inoutlists::remove_translist(unsigned source_p, unsigned target_p) {
    remove_inoutlist(inlists, target_p, source_p);
    remove_inoutlist(outlists, source_p, target_p);
}

void Inoutlists::remove_inoutlist(Orefs &inoutlists, unsigned partition1,
				  unsigned partition2) {
    if (partition1 >= (unsigned)inoutlists.size()) return;
    Oref oref = inoutlists[partition1];
    while (oref != NULL_OREF) {
	MM_Handle *handle = orx->mm->fetch(oref);
	if (handle == 0) 
	  fail("Inoutlist-block not fetched: %d", oref);
	OR_obj *block = handle->obj();
	OR_slot *fields = block->fields();
	unsigned partition;
	int i = 0;
	do {
	    partition = fields[i].uvalue32;
	    i += 2;
	} while (partition != partition2 && partition != NULL_PARTITION);
	oref = fields[--i].get_oref();
	if (partition == partition2 && oref != NULL_OREF) { 
	    // Found it; nullify the oref.
	    block = allocator->copy(block, oref);
	    fields = block->fields();
	    fields[i].set_oref(NULL_OREF);
	    oref = NULL_OREF;
	}
	orx->mm->release(handle);
    }
}


Inoutlists::Stats::Stats() {
    on = FALSE;
}

void Inoutlists::Stats::start() {
    on = TRUE;
    blocks = 0;
}

void Inoutlists::Stats::print(ostream *out) {
    *out << "Inoutlists blocks byte usage: " 
	 << blocks*gc->inoutlist_block_size*sizeof(Slot) << endl;
    *out << "Time to get translist: " << get_translist_timer.elapsed() << endl;
    *out << "Time to set translist: " << set_translist_timer.elapsed() << endl;
}
