# Copyright 1995 Barbara Liskov

source init/or.tcl

# This controls whether the database is built outside the OR
# (slow but debugged) or inside the OR (fast but still being developed).

set gen_inside_or 1

# This is used in generating oo7objects and has nothing to do with
# the oids used internally in Thor.

set the_object_id 0

if {!$gen_inside_or} {
    proc object_id {} {
	global the_object_id
	return $the_object_id
    }

    proc next_id {} {
	global the_object_id
	set my_id $the_object_id
	incr the_object_id
	return $my_id
    }
}

# Track how many objects of different kinds have been created
set composite_num 0
set assembly_num 0

# These are "distinctive values" for date, x, and y that are used
# repeatedly rather than passing them around all the time.

set gx 34567
set gy -432
set gdate 9087

# These are OO7 parameters

set NumAtomicPerComp 	20
set NumConnPerAtomic 	3
set DocumentSize 	2000
set ManualSize		[expr 100*1024]
set NumCompPerModule	500
set NumAssmPerAssm	3
set NumAssmLevels	7
set NumCompPerAssm	3
set NumModules		1

# A handy empty Thor list, initialized by CreateOO7Database.

set nil "This is a bogus value that should be overwritten"

# How to produce a string of a particular length (for Document)

proc long_text {size} {
  global DocumentSize
  set text "The total length of this text is exactly $size characters.  "
  while {[string length $text] < $DocumentSize} {
    set text [format "%s%s" $text $text]
  }
  set text [string range $text 0 [expr $DocumentSize-1]]
  return $text
}

# The string used to generate documents

set DocumentText [long_text $DocumentSize]

# The current segment being used to store the OO7 database.
# This variable is obsolete and will be eliminated soon.

set oo7segment [segment]

# The otypes of various kinds of OO7 objects, shared among the instances
set AtomicPartOtype [newstring $oo7segment "atomic"]
set ConnectionOtype [newstring $oo7segment "connection"]
set ComplexAssemblyOtype [newstring $oo7segment "complex assembly"]
set BaseAssemblyOtype [newstring $oo7segment "base assembly"]
set CompositePartOtype [newstring $oo7segment "composite part"]

# Accumulate the orefs of assemblies here

set allAssemblies {}

# Create the OO7 database using the parameters above (possibly
# modified by the user program).  NumModules is currently ignored.
# Return the oref of the single module.  Sets global variable nil.
# Puts the assembly hierarchy in different segments from the library.
# XXX ManualSize ignored.

proc CreateOO7Database {} {
  global nil
  global oo7segment
  global allAssemblies
  global gen_inside_or
  set allAssemblies {}
  set man [Manual "A test manual" "This should be a lot of text"] 
  set nil [thornil $oo7segment]
  set mod [Module "module" 1989 $man $nil $nil]
  if {$gen_inside_or} {
    puts stdout "Generating composite parts inside OR"
  } else {
    puts stdout "Generating composite parts outside OR"
  }
  flush stdout
  set lib [GenerateOO7Library $mod]
  set root [NormalComplexAssembly 1 $lib $mod 0]
  ModuleSetRoot $mod $root
  if {0} {
    puts stdout "Assembly list = $allAssemblies"
    flush stdout
  }
  set numAssemblies [llength $allAssemblies]
  if {$numAssemblies < 16000} {
    set assemblies [tcl_list_to_thor_vec $allAssemblies]
    ModuleSetAssemblies $mod $assemblies
  } else {
  puts stdout "Too many assemblies for current implementation, building incomplete Module"
  }
  puts stdout "$numAssemblies assemblies in module"
  puts stdout "Database module generation complete, [object_id] benchmark objects"
  flush stdout
  return $mod
}

# Create a library of composite parts.  Changes segment after each
# 3 composite parts.

proc GenerateOO7Library {mod} {
  global NumCompPerModule
  set lib {}
  for {set i 0} {$i < $NumCompPerModule} {incr i} {
    set cp [CompositePart]
    lappend lib $cp
  }
  return $lib
}

proc ChooseFromLibrary {lib} {
  set choice [rand [expr [llength $lib]-1]]
  return [lindex $lib $choice]
}

# Create a BaseAssembly using the library of composite parts.

proc BaseAssemblyFromLibrary {lib mod super} {
  global NumCompPerAssm
  global nil
  global allAssemblies
  if {0} {
    puts stdout "BaseAssemblyFromLibrary S = $super"
    flush stdout
  }
  set parts {}
  for {set i 0} {$i < $NumCompPerAssm} {incr i} {
    lappend parts [ChooseFromLibrary $lib]
  }
  if {0} {
    puts stdout "Base assembly contains parts: $parts"
    flush stdout
  }
  set thorparts [tcl_list_to_thor_vec $parts]
  set result [BaseAssembly $super $mod $thorparts $nil]
  lappend allAssemblies $result
  return $result
}

# Create a ComplexAssembly containing BaseAssemblies

proc LowestComplexAssembly {lib mod super} {
  global NumAssmPerAssm
  global nil
  global allAssemblies
  if {0} {
    puts stdout "LowestComplexAssembly S = $super"
    flush stdout
  }
  set result [ComplexAssembly $super $mod $nil]
  set children {}
  for {set i 0} {$i < $NumAssmPerAssm} {incr i} {
    lappend children [BaseAssemblyFromLibrary $lib $mod $result]
  }
  set thorchildren [tcl_list_to_thor_vec $children]
  ComplexAssemblySetChildren $result $thorchildren
  lappend allAssemblies $result
  return $result
}

# Create a ComplexAssembly containing /depth/ levels of other 
# ComplexAssemblies, with BaseAssemblies at the leaves (i.e. 
# when depth = 0, creates BaseAssemblies instead of ComplexAssemblies)
# Call with 0 for super when there is no superassembly.
#
# The root ComplexAssembly (level = 1, super = 0) may have a different
# branching factor if Root_NumAssmPerAssm is set elsewhere.

set Root_NumAssmPerAssm $NumAssmPerAssm

proc NormalComplexAssembly {level lib mod super} {
  global NumAssmPerAssm
  global Root_NumAssmPerAssm
  global NumAssmLevels
  global nil
  if {0} {
    puts stdout "NormalComplexAssembly L = $level S = $super"
    flush stdout
  }
  if {$level == [expr $NumAssmLevels - 1]} {
    return [LowestComplexAssembly $lib $mod $super]
  } else {
    set result [ComplexAssembly $super $mod $nil]
    set children {}
    if {$level == 1} {
      set numChildren $Root_NumAssmPerAssm
    } else {
      set numChildren $NumAssmPerAssm
    }
    incr level
    for {set i 0} {$i < $numChildren} {incr i} {
      lappend children [NormalComplexAssembly $level $lib $mod $result]
    }
    set thorchildren [tcl_list_to_thor_vec $children]
    ComplexAssemblySetChildren $result $thorchildren
    lappend allAssemblies $result
    return $result
  }
}

# Create an atomic part.  The x, y, and date fields are determined
# by the global values.  The id is determined by the global object_id.
# Typically the "out" and "in" lists are empty Thor lists
# (i.e. [thornil], not {}) and are filled in later.

proc AtomicPart {docID out in parent} {
  global gx
  global gy
  global gdate
  global oo7segment
  global AtomicPartOtype
  set result [new $oo7segment atomicpart {ref data data data ref ref ref}]
  set otype $AtomicPartOtype
  init_designobj $result [next_id] $otype $gdate
  store $result 2 [list int $gx int $gy]
  store $result 3 [list int $docID int 0]
  AtomicPartSetOutgoing $result $out
  AtomicPartSetIncoming $result $in
  store $result 6 $parent
  return $result
}

# Replaces the list of outgoing connections for atomic part /ap/.
# The list /out/ should be the oref of a Thor list, not a Tcl list.

proc AtomicPartSetOutgoing {ap out} {
  if {0} {
    puts stdout "Atomic Part (oref = $ap) outgoing list oref = $out"
    flush stdout  
  }
  store $ap 4 $out
}

# Replaces the list of incoming connections for atomic part /ap/.
# The list /in/ should be the oref of a Thor list, not a Tcl list.

proc AtomicPartSetIncoming {ap in} {
  if {0} {
    puts stdout "Atomic Part (oref = $ap) incoming list oref = $in"
    flush stdout  
  }
  store $ap 5 $in
}

# To make a composite part, first the object itself is allocated
# and the simple fields filled in.  Then connmap computes the connections
# between atomic parts.  The atomic parts are created without connections.
# Finally, the atomic parts are connected.

# Routine to generate a composite part outside of the OR
proc outer_comppart {num_atomic num_conn} {
    global gdate
    global NumAtomicPerComp
    global oo7segment
    global nil
    global CompositePartOtype

    set result [new $oo7segment compositepart {ref data ref ref ref ref ref}]
    set otype $CompositePartOtype
    init_designobj $result [next_id] $otype $gdate
    connmap to_list from_list $num_conn $num_atomic

    set atomics {}
    for {set i 0} {$i < $NumAtomicPerComp} {incr i} {
	lappend atomics [AtomicPart 42 $nil $nil $result]
    }
    do_connections $atomics to_list from_list
    store $result 5 [tcl_list_to_thor_vec $atomics]
    store $result 6 [lindex $atomics 0]

    return $result
}

# Create a special "comppart" routine if we want generation outside the OR.
if {!$gen_inside_or} {
    rename outer_comppart comppart
}

proc CompositePart {} {
  global NumConnPerAtomic
  global NumAtomicPerComp
  global composite_num
  global nil

  set result [comppart $NumAtomicPerComp $NumConnPerAtomic]
  store $result 2 [Document "Title[object_id]"]
  store $result 3 $nil
  store $result 4 $nil

  incr composite_num
  if {($composite_num % 10) == 0} {
    puts stdout "Composite $composite_num (oref $result) created, [object_id] benchmark objects so far"
    flush stdout
  }
  return $result 
}

# Actually make the connections among the objects in /atomics/, 
# based on the information in /to_list_/ and /from_list_/.  Note that
# the "lists" are both array variables containing a number of Tcl lists
# of integer indexes, whereas /atomics/ is a Tcl list of orefs of atomic 
# parts.  
#
# A partial explanation of the abstraction function:
#
# The term "part $x" means "the part whose oref is stored in atomics($x)"
#
# If [lindex $to_list($i) $j] == $k, then part $i has an outgoing
# connection to part $k.  Note that $j is NOT the number of a part, but
# only an index into what should logically be a bag.
#
# Invariant:
#
# ([lindex $to_list($i) $j] == $k) iff there exists some $t such that
# ([lindex $from_list($k) $t] == $i)
#

proc do_connections {atomics to_list_ from_list_} {
  upvar $to_list_ to_list
  upvar $from_list_ from_list
  global NumAtomicPerComp
  for {set i 0} {$i < $NumAtomicPerComp} {incr i} {
    if {0} {
      puts stdout "Connecting atomic part $i"
      flush stdout
    }
    set this [lindex $atomics $i]
    set tol $to_list($i)
    set len [llength $tol]
    for {set j 0} {$j < $len} {incr j} {
      set k [lindex $tol $j]
      set dest [lindex $atomics $k]
      set conn [Connection $this $dest]
      set connmatrix($i,$k) $conn
    }
  }
  for {set i 0} {$i < $NumAtomicPerComp} {incr i} {
    set this_to {}
    set this [lindex $atomics $i]
    set tol $to_list($i)
    set len [llength $tol]
    for {set j 0} {$j < $len} {incr j} {
      set k [lindex $tol $j]
      set c $connmatrix($i,$k) 
      if {0} {
        puts stdout "Connection from $i out to $k oref = $c"
        flush stdout
      }
      lappend this_to $c
    }
    AtomicPartSetOutgoing $this [tcl_list_to_thor_vec $this_to]
  }
  for {set i 0} {$i < $NumAtomicPerComp} {incr i} {
    set this_from {}
    set this [lindex $atomics $i]
    set frl $from_list($i)
    set len [llength $frl]
    for {set j 0} {$j < $len} {incr j} {
      set k [lindex $frl $j]
      set c $connmatrix($k,$i) 
      if {0} {
        puts stdout "Connection from $k in to $i oref = $c"
        flush stdout
      }
      lappend this_from $c
    }
    AtomicPartSetIncoming $this [tcl_list_to_thor_vec $this_from]
  }
}
  
proc tcl_list_to_thor_list {oref_list} {
  global oo7segment
  global nil
  set len [llength $oref_list]
  set result $nil
  for {set i [expr $len-1]} {$i >= 0} {incr i -1} {
    set curr [new $oo7segment fulllist {allrefs 2}]
    store $curr 0 [lindex $oref_list $i]
    store $curr 1 $result
    set result $curr
  }
  return $result
}

proc tcl_list_to_thor_vec {oref_list} {
  set len [llength $oref_list]
  set result [newvec $len]
  for {set i 0} {$i < $len} {incr i} {
    vecstore $result $i [lindex $oref_list $i]
  }
  return $result
}

proc newvec {len} {
  global oo7segment
  set result [new $oo7segment vecofany [list allrefs $len]]
  return $result
}

proc vecstore {vec index oref} {
  store $vec $index $oref
}

# Do bookkeeping associated with creating an assembly

proc CountAssembly {} {
  global assembly_num
  incr assembly_num
  if {[expr $assembly_num%100] == 0} {
    puts stdout "Assembly $assembly_num initialized, [object_id] total objects so far"
    flush stdout
  }
}

# create a complex assembly.  Call with 0 as super if there is 
# no superassembly.  Children is the oref of a Thor list of 
# ComplexAssemblies or BaseAssemblies.

proc ComplexAssembly {super module children} {
  global gdate
  global oo7segment
  global ComplexAssemblyOtype
  CountAssembly
  set result [new $oo7segment complexassembly {ref data ref ref ref}]
  set otype $ComplexAssemblyOtype
  if {$super == 0} {set super $result}
  init_assembly $result [next_id] $otype $gdate $super $module
  ComplexAssemblySetChildren $result $children
  return $result 
}

# Replaces the list of children for complex assembly /ca/.
# The list /ch/ should be the oref of a Thor list, not a Tcl list.

proc ComplexAssemblySetChildren {ca ch} {
  if {0} {
    puts stdout "Complex Assembly (oref = $ca) Children list oref = $ch"
    flush stdout  
  }
  store $ca 4 $ch
}

# create a connection between two atomic parts.  Its id is 
# generated automatically, and its length is zero.

proc Connection {from to} {
  global oo7segment
  global ConnectionOtype
  set result [new $oo7segment connection {ref data ref ref}]
  store $result 0 $ConnectionOtype
  store $result 1 [list int [next_id] int 0]
  store $result 2 $from
  store $result 3 $to
  return $result
}

# Create a base assembly.  Call with 0 as super if there is no superassembly.
# priv and shar are orefs of Thor lists of composite parts contained
# in this assembly.

proc BaseAssembly {super module priv shar} {
  global gdate
  global oo7segment
  global BaseAssemblyOtype
  CountAssembly
  set result [new $oo7segment baseassembly {ref data ref ref ref ref}]
  set otype $BaseAssemblyOtype
  if {$super == 0} {set super $result}
  init_assembly $result [next_id] $otype $gdate $super $module
  BaseAssemblySetPrivate $result $priv
  BaseAssemblySetShared $result $shar
  return $result 
}

# Replaces the list of private children for base assembly /ba/.
# The list /priv/ should be the oref of a Thor list, not a Tcl list.

proc BaseAssemblySetPrivate {ba priv} {
  if {0} {
    puts stdout "Base Assembly (oref = $ba) Private list oref = $priv"
    flush stdout  
  }
  store $ba 4 $priv
}

# Replaces the list of Shared connections for base assembly /ba/.
# The list /shar/ should be the oref of a Thor list, not a Tcl list.

proc BaseAssemblySetShared {ba shar} {
  if {0} {
    puts stdout "Base Assembly (oref = $ba) Shared list oref = $shar"
    flush stdout  
  }
  store $ba 5 $shar
}

# create an assembly.  This should never be used except for testing.
# Call with 0 as super if there is no superassembly

proc Assembly {type super module} {
  global gdate
  global oo7segment
  puts stdout "Warning: created instance of supertype Assembly"
  flush stdout
  set result [new $oo7segment assembly {ref data ref ref}]
  set otype [newstring $oo7segment $type]
  if {$super == 0} {set super $result}
  init_assembly $result [next_id] $otype $gdate $super $module
  return $result 
}

# initialize an assembly.

proc init_assembly {obj id type date super module} {
  if {0} {
    puts stdout "init assembly super = $super module = $module"
    flush stdout
  }
  init_designobj $obj $id $type $date
  store $obj 2 $super
  store $obj 3 $module
}

# create a module.  This is just a placeholder for the
# full implementation to come later.

proc Module {type date manual root assms} {
  global allAssemblies
  global oo7segment
  set result [new $oo7segment module {ref data ref ref ref}]
  set otype [newstring $oo7segment $type]
  init_designobj $result [next_id] $otype $date
  store $result 2 $manual
  ModuleSetAssemblies $result $assms
  ModuleSetRoot $result $root
  return $result 
}

# set the root ComplexAssembly for a module

proc ModuleSetRoot {mod root} {
  store $mod 4 $root
}

# set the assemblies field for a module.  
# Assms should be the oref of a Thor list, not a Tcl list.

proc ModuleSetAssemblies {mod assms} {
  store $mod 3 $assms
}

# create a document.  

proc Document {title} {
  global DocumentText
  return [real_document $title [next_id] $DocumentText]
}

# create a manual.

proc Manual {title text} {
  return [makeoo7text $title [next_id] $text manual]
}

# initialize the fields that are common to all DesignObj types.

proc init_designobj {obj id type date} {
  store $obj 0 $type
  store $obj 1 [list int $id int $date]
}

proc alloc_designobj {class} {
  global oo7segment
  set result [new $oo7segment $class {ref data}]
  return $result
}

# Note that a document has one more ref field, pointing to a composite object

proc makeoo7text {title id text class} {
  global oo7segment
  set result [new $oo7segment $class {ref data ref}]
  store $result 0 [newstring $oo7segment $title]
  store $result 1 [list int $id int 0]
  store $result 2 [newstring $oo7segment $text]
  return $result
}

proc real_document {title id text} {
  return [makeoo7text $title $id $text document]
}

# effects - sets to_ and from_ to be arrays of lists.  The indices
#           of to_ go from 0 to $objs-1, and each list is $conns long.
#           Each element of each list is a random number x such that 
#	    0 <= x <= $objs-1.
#           _from($q) contains $z iff _to($z) contains $q

proc connmap {to_ from_ conns objs} {
  upvar $to_ to
  upvar $from_ from
  for {set i 0} {$i < $objs} {incr i} {
    set to($i) {}
    set from($i) {}
  }
  for {set i 0} {$i < $objs} {incr i} {
    for {set j 0} {$j < $conns} {incr j} {
      set conn [rand $objs]
      set to($i) [concat $to($i) $conn]
      set from($conn) [concat $from($conn) $i]
    }
  }
}
