// Copyright 1995 Barbara Liskov

/*
  \section{Invalid object set}

  This file contains the header for a set of invalid objects.  This set is 
  kept on a per-FE basis.
  
  Although there is no read/write lock associated with the invalid set
  itself, the set is included in the FE_manager,
  which contains a lock for each FE.  In order to call the methods in an
  invalid set, the caller should have the appropriate lock in the FE_manager.

  Invalid objects are generated for this frontend F due to modifications by
  a transaction T from other frontends. The invalid object set keeps
  collections of invalid objects -- one collection for each transaction
  that invalidates F's objects. Each collection is denoted by a unique
  sequence number (for this server-frontend pair) that is used by the TM for
  sending messages to F. We have kept the sequence number to be
  monotonically increasing so that F just has to acknowledge the last
  collection seen instead of the complete list (this property is also useful
  for recovery purposes). Note that we could the log sequence number over
  here but that is currently not being done.

  Here, a collection is just like a set, i.e. no duplicate entries for an
  object.
  
  XXX Do we need to say about objects in earlier collections? The curent
  methods in the interface do not allow you to detect that. But we need to
  change the AF for this.
  
  \subsection{Abstraction function}
  
  A(r) = 
  \begin{itemize}
  \item The objects in the invalid set are keys of the openhashmap (hash
  table) r.invalid_set.  The value associated with the object o
  indicates the latest collection to which it belongs.

  \item	 highest_collection_num is the highest collection number returned
  by the new_collection operation.
  \end{itemize}

  \subsection{Rep invariant}
  \begin{itemize}
  \item No object has a collection number higher than highest_collection_num.
  \item Each object has exactly one collection number associated with it.
  \end{itemize}   

  \subsection{Notes}

  The two obvious implementations for the invalid list are an array of
  collections, or a hash table from orefs to collection numbers.  The
  array representation can remove a collection quickly when an
  acknowledgment arrives from the FE; however, determining whether a
  particular object is present requires a search of the entire array.
  We have chosen instead to use a hash table, which requires a search
  when an acknowledgment arrives, but can quickly determine whether an
  object is present.  

*/

#ifndef _INV_SET_H
#define _INV_SET_H

#include "utils/basic.h"
#include "utils/openhashmap.h"
#include "common/oref.h"

class Orefs;

// Structure for invalid set
//          Hash table from Orefs to Colection_num (monotonically increasing
//          sequence number)

typedef Ubits32 Collection_num;
declareOpenHashMap(Inv_set, Oref, Collection_num, Oref_hash, Oref_equal)

class Invalid_set {
public:
    Invalid_set(void);
    // effects:  Create a new set of invalid objects.  
    //           Initially, the highest allowable collection number is 0.
    
    ~Invalid_set(void);
    // effects:  Destroy invalid set

    bool empty();
    // requires: caller has a read or write-lock on this
    // effects: Return TRUE iff the invalid list is empty
    
    bool member(Oref x);
    // requires: caller has a read or write-lock on this
    // effects: Return TRUE iff x is in some collection in the invalid list 

    void add_obj(Oref x, Collection_num n);
    // requires: caller has a write-lock on this
    //           n was returned by a call to new_collection.
    // modifies: this
    // effects:  Add x to the collection denoted with collection number n.
    
    Orefs *invalidated_since(Collection_num n);
    // requires: caller has a read or write-lock on this
    // effects:  Returns the set of all objects belonging to collections
    //           with collection numbers > n. 
    //           This set is newly allocated and can be freed by caller.
    
    void remove_before(Collection_num n);
    // requires: caller has a write-lock on this
    // modifies: this
    // effects:  Remove all collections with collection numbers <= n.
    
    Collection_num new_collection(void);
    // requires: caller has a write-lock on this
    // modifies: this
    // effects:  Increments and then returns highest allowable collection 
    //           number in invalid set.
    
    void reset(void);
    // requires: caller has a write-lock on this
    // modifies: this
    // effects:  Make the set of invalid objects empty, and reset the
    //           highest allowable collection number to 0.

private:
    Collection_num highest_collection_num;
    Inv_set *invalid_set;
};

#endif /* _INV_SET_H */
