// Copyright 1995 Barbara Liskov

/*
\section{Main Program}

This is the "main" routine for an OR.
If you modify the argument processing code here, please update the
corresponding documentation in "doc/HOW_TO_RUN".
*/

#include <stdio.h>
#include <stdlib.h>

#include "utils/compat.h"
#include "utils/fail.h"
#include "common/locator.h"

#include "mm/disk.h"
#include "gc/gc.h"

#include "init.h"
#include "or_config.h"
#include "or.h"
#include "server.h"
#include "or_or_msg.h"
#include "config/vdefs/REPLICATION.h"

static OR_config* parse_arguments(int argc, char* argv[]);
    // requires	argc is length of argv.  argv[0] is program name.
    // effects	Creates or configuration from command line and returns it.
    //		The caller should delete the returned object when it is
    //		no longer needed.  Dies after printing an error message
    //		if the command line is not well-formed.

static void usage(char const* prog);
    // requires	prog is program name.
    // effects	Print usage message and die.

#include "message_stats.h"
Message_Stats msg_stats;

int
main(int argc, char* argv[]) {

    orx = new OR;
    orx->config = parse_arguments(argc, argv);

    if (orx->config->init != 0)
	OR_init();
    else
	OR_recover();

    // Provide service to FEs
    serve_connections();

    return 0;
}

// \subsection{Argument parsing code}
//
// If you modify this code, please update the corresponding documentation in
// "doc/HOW_TO_RUN".

static OR_config* parse_arguments(int argc, char* argv[]) {
    OR_config* config		= new OR_config;
    config->fe_socket		= OR_DEFAULT_FE_PORT;
    config->or_socket		= OR_DEFAULT_OR_PORT;
    config->ornum		= 1;
    config->cache_size		= OR_DEFAULT_CACHE_SIZE;
    config->log_size		= OR_DEFAULT_LOG_SIZE;
    config->apply_threshold	= OR_DEFAULT_APPLY_PERCENT;
    config->target_percent	= OR_DEFAULT_TARGET_PERCENT;
    config->simulate_log_flush  = FALSE;
#if REPLICATION
    config->with_backups        = FALSE;
    config->two_backups        = FALSE;
#endif    
    config->print_time          = FALSE; 
    config->debug_level		= 0;
    config->debug_coop		= FALSE;
    config->num_clients         = 8;
    config->eager_prefetch      = 0;

    // gc options
    gc                          = NULL;

    config->init		= new OR_init_config;
    config->init->size		= (200 * 1024 * 1024) >> DISK_UNIT_SHIFT;
    config->init->input_file    = 0;

    config->fetch_mods = TRUE;
    config->fetch_cache = TRUE;
    config->fetch_complete = TRUE;
    config->fetch_reparable = TRUE;
    config->fetch_disk = TRUE;

    config->iread_complete = TRUE;
    config->iread_reparable = TRUE;
    config->iread_mods = TRUE;

    config->send_updates = FALSE;
    config->send_updates_at_commit = FALSE;

    int have_db      = 0;
    int have_size    = 0;
    int opt;
    int number;
    char* endptr;
    // The B and b options are for replication only but have not
    // been set in the getopt string.

    while ((opt = getopt(argc, argv, "ip:o:s:S:D:l:f:r:n:u:P:gLbBa:t:")) != EOF) {
	switch (opt) {
	  case 'o':
	    number = strtol(optarg, &endptr, 0);
	    if ((endptr == optarg) || (number < 0)) usage(argv[0]);
	    config->ornum = number;
	    break;
#if REPLICATION
          case 'B':
	    config->with_backups = TRUE;
            break;
          case 'b':
	    config->two_backups = TRUE;
            break;
#endif
	  case 'n':
	    config->num_clients = strtol(optarg, &endptr, 0);
            break;
	  case 'f': 
	    if (!strcmp(optarg, "mods")) 
		config->fetch_mods = FALSE;
      	    else if (!strcmp(optarg, "cache")) 
		config->fetch_cache = FALSE;
      	    else if (!strcmp(optarg, "complete")) 
		config->fetch_complete = FALSE;
      	    else if (!strcmp(optarg, "reparable")) 
		config->fetch_reparable = FALSE;
      	    else if (!strcmp(optarg, "disk")) 
		config->fetch_disk = FALSE;
	    else usage(argv[0]);
	    break;
	  case 'u':  
	    if (0 == strcmp(optarg, "lazy")) 
		config->send_updates = TRUE;
      	    else if (0 == strcmp(optarg, "commit")) {
		config->send_updates = TRUE;
		config->send_updates_at_commit = TRUE;
	    } else usage(argv[0]);
            break;  
	  case 'r': 
	    if (endptr == optarg) usage(argv[0]);
      	    else if (!strcmp(optarg, "complete")) 
		config->iread_complete = FALSE;
      	    else if (!strcmp(optarg, "reparable")) 
		config->iread_reparable = FALSE;
      	    else if (!strcmp(optarg, "mods")) 
		config->iread_mods = FALSE;
	    else usage(argv[0]);
	    break;
	  case 'l':
	    number = strtol(optarg, &endptr, 0);
	    if ((endptr == optarg) || (number <= 0)) usage(argv[0]);
	    config->log_size = number * 1024;
	    break;
          case 'a':
	    number = strtol(optarg, &endptr, 0);
	    if ((endptr == optarg) || (number < 0) || number > 100) usage(argv[0]);
	    config->apply_threshold = number;
	    break;
          case 't':
	    number = strtol(optarg, &endptr, 0);
	    if ((endptr == optarg) || (number < 0) || number > 100) usage(argv[0]);
	    config->target_percent = number;
	    break;
          case 'L':
            config->simulate_log_flush = TRUE;
            break;
          case 'P':
	    number = strtol(optarg, &endptr, 0);
	    if ((endptr == optarg) || (number <= 0)) usage(argv[0]);
            config->eager_prefetch = number;
            break;
          case 'g':
            gc = new GC();
            break;
          case 'i':
            have_db = 1;
            break;
	  case 's':
	    number = strtol(optarg, &endptr, 0);
	    if ((endptr == optarg) || (number <= 0)) usage(argv[0]);
	    config->init->size = (number * 1024 * 1024) >> DISK_UNIT_SHIFT;
	    have_size = 1;
	    break;
	  case 'S':
	    number = strtol(optarg, &endptr, 0);
	    if ((endptr == optarg) || (number <= 0)) usage(argv[0]);
	    config->cache_size = number * 1024;
	    break;
	  case 'D':
	    number = strtol(optarg, &endptr, 0);
	    if (endptr == optarg) {
		if (0 == strcmp(optarg, "coop")) config->debug_coop = TRUE;
		else usage(argv[0]);
	    } else {
		config->debug_level = number;
	    }
	    break;
	  default:
	    usage(argv[0]);
	    break;
	}
    }

    if ((argc - optind) != 1) {
	// set config->disk to ./or_$USER
	char const * prefix = "./or_";
	char const * user =  getenv("USER");
	config->disk = new char[strlen(prefix)+strlen(user)+2];
	strcpy(const_cast<char *>(config->disk), prefix);
	strcat(const_cast<char *>(config->disk), user);
	fprintf(stderr, "No data file specified. Creating new db in %s.\n", config->disk);
	have_db = 1; // Assume a new db. This may be changed.
    }
    else {
	config->disk = argv[optind];
    }

    // Check for conflicting options
    if (!have_db && have_size) usage(argv[0]);

    if (!have_db) {
	delete config->init;
	config->init = 0;
    }

    return config;
}

static void usage(char const* p) {
    warn("Usage: %s [-i [init options]] [options] [<file>]", p);
    warn("  <file>               Name of OR backing store. [/tmp/or_$USER]");
    warn("");

    warn("  General options (default values are in []):");
    warn("    -o <ornum>         The or number                  [1]");
    warn("    -l <logsize>       Max log size in kilobytes      [%d KB]",
	 OR_DEFAULT_LOG_SIZE);
    warn("    -L                 Whether simulate log flush     [off]");
    warn("    -B                 Whether run or with backups    [off]");
    warn("    -b                 Whether run with two backups   [one backup]");
    warn("    -P <Num pages>     Num pages in eager prefetch    [0]");
    warn("    -S <cachesize>     Max cache size in kilobytes    [%d KB]",
	 OR_DEFAULT_CACHE_SIZE);
    warn("    -D <debug_level>   Print debugging data           [off]");
    warn("    -D <coop>          Specific debugging             [off]");
    warn("    -g                 Turn GC on                     [off]");
    warn("");

    warn("  Initialize options (default values are in []):");
    warn("    -s <size>          Database size in megabytes     [20]");
    warn("");

    warn("  Fetch policy:");
    warn("    -f <mods cache reparable complete disk>");
    warn("    -f xyz suppresses fetching using the xyz policy.");
    warn("    E.g., \"-f mods -f disk suppresses\" ");
    warn("    suppresses fetching only modifications and fetching from disk.");
    warn("");

    warn("  Iread policy:");
    warn("    -r <reparable complete>");
    warn("    -r xyz suppresses ireads using the xyz policy.");
    warn("    E.g., \"-r reparable -r complete\" ");
    warn("    suppresses ireads from FEs with reparable or complete segments");
    warn("");

    warn("  Update propagation policy:");
    warn("    -u <lazy commit>   Send updates with invalidations [off]");
    warn("                       either lazily, or at commit points");

    fail("*** exit ***");
}

