// Copyright 1995 Barbara Liskov

// \section{Disk Format}

#ifndef _DFORMAT_H
#define _DFORMAT_H

// The following is used to find the UFS superblock address.
// We use a similar address for our superblock so that we can
// skip any disk labelling information.
#ifdef __linux__
#define BBOFF           ((off_t)(2048))
#define SBOFF           ((off_t)(BBOFF + BBSIZE))
#define SBSIZE 8192 // from /usr/include/linux/ufs_fs.h
#define BBSIZE 8192 // from /usr/include/linux/ufs_fs.h
#else
#include <ufs/fs.h>
#endif
#include <iostream.h>

#include "utils/basic.h"
#include "utils/bits.h"
#include "common/oref.h"
#include "common/or_obj.h"
#include "common/page.h"
#include "disk.h"

// \subsection{Super-blocks}
//
// Each OR maintains a pair of super-blocks in disk with
// checksums and timestamps.   The super-block with the higher time-stamp
// is used at recovery time.
//
// \begin{center}
// \begin{tabular}{|l|p{3in}|}
//	\hline
//	"DISK_SB_SIZE"&		Super-block size (in disk units)
//	"DISK_SB_1"&		Address of first super-block
//	"DISK_SB_2"&		Address of second super-block
//	\hline
// \end{tabular}
// \end{center}

#define DISK_SB_SIZE	1

#define DISK_SB_1	(SBOFF >> DISK_UNIT_SHIFT)
#define DISK_SB_2	(DISK_SB_1 + 1)

// Magic number to identify OR super-block
#define DISK_SB_MAGIC	0xA93C1E47

// Maximum length of log-file name
#define DISK_MAX_LOGLEN	100

// The super-block contains enough information to allow OR initialization.
// \begin{center}
// \begin{tabular}{|l|p{3in}|}
//	\hline
//	magic&		Contains magic number that can be used during
//			OR initialization to check that this is actually
//			the persistent storage for an OR.
//	checksum&	Checksum is used during recovery to detect
//			a partially updated superblock.  This situation
//			arises when an OR crashes while updating the block.
//	timestamp&	Used to find the newer of the two super-blocks.
//	\hline
//	size&		Number of blocks in disk (includes superblocks).
//	segprefsize&	Preferred segment size.
//	segtable&	Disk range occupied by segment table.
//	root&		Root oref.
//	log&		Log file name
//	\hline
// \end{tabular}
// \end{center}

#define DISK_SB_HEADER							      \
    Ubits32	magic;							      \
    Ubits32	checksum;						      \
    Ubits32	timestamp;						      \
    Disk_Count	size;							      \
    Disk_Count	segprefsize;						      \
    Disk_Range	segtable;						      \
    Oref	root;							      \
    char	log[DISK_MAX_LOGLEN+1]

struct Disk_Sb_Header {
    DISK_SB_HEADER;
};

// The super-block structure
struct Disk_SB {
    DISK_SB_HEADER;

    // Pad until the superblock occupies the right number of disk blocks.
    char pad[DISK_UNIT*DISK_SB_SIZE - sizeof(Disk_Sb_Header)];
};

// \subsection{Segment Table}
//
// The segment table is maintained as a two-level map formed out
// of segments.  The root of the map is a segment stored at a
// wellknown location on disk.  This root segment consists of
// one object which has some small integer info followed by
// an array of disk ranges for the second level map entries.
//
// Each second-level map entry is a segment with a single object.
// The second-level object has some integers followed by an
// array of disk ranges for the segments mapped by this 2nd-level
// entry.

// Format of the single object contained in the root segment:
// (A disk range is encoded into two "OR_slots" by putting the
// address in the first slot and count in the second one)
struct Disk_RootObj {
    // This part matches an OR_obj
    OR_slot	header[OR_obj_headers];

    Ubits32	max_segments;	// Maximum number of segments allowed in db
    Ubits32	num_leaves;	// # of leaf entries
    Ubits32	segs_per_leaf;	// # of entries per leaf segment
    Disk_Range	ranges[1];	// Disk ranges for leaf segments
};

// Format of the single object contained in a leaf segment:
// (A disk range is encoded into two "OR_slots" by putting the
// address in the first slot and count in the second one)
struct Disk_LeafObj {
    // This part matches an OR_obj
    OR_slot	header[OR_obj_headers];

    Ubits32	first;		// Segment number of first mapped entry
    Ubits32	count;		// Number of entries in here
    Disk_Range	ranges[1];	// Disk ranges for mapped segments
};


// \subsection{Segments}
//
// Each segment is considered an array of "OR_slot"s (each
// "OR_slot" is a four-byte quantity) followed by pages (each of size Page_size)
//
// The first few OR_slots in the segment consist various pieces of
// information described below.

// Segment magic number
#define DISK_SEG_MAGIC	0xA93C1E4A

// Since the number of pages being kept track of in terms of pages, we can
// have at most 2^8 pages in a segment
#define DISK_SEG_MAX_SIZE (Page_size * (1 << 8))

// Maximum object index for a segment
// #define DISK_SEG_MAX_INDEX ((1 << 16) - 1)

// Segment types.  "Normal" holds Thor universe objects.
// "Special" holds OR runtime objects (these are not in the Thor universe).

#define DISK_SEG_NORMAL		0
#define DISK_SEG_SPECIAL	1

struct Disk_Segment {
    // Initial "OR_slot"s
    //struct Segment_header {
    //	Ubits32 magic;	   // Magic number
    //	Ubits16 id;	   // Segment id
    //	Ubits16 type;      // Segment type
    //	Ubits8  num_pages; // Number of pages in the entire segment
    //    } h;

    // The padding is needed to ensure that the pages are disk block aligned
    // char pad[DISK_UNIT - sizeof(Disk_Segment::Segment_header)];

    Page pages[1];	// Array of pages

    void unparse(ostream &o);
    // prints segment on stderr
};

inline void Disk_Segment::unparse(ostream &out) {

    out << "SEGMENT" << endl;
    //    out << "id = " << h.id << endl;
    //    out << "num_pages = " << h.num_pages << endl;
    //    out << "type = " << h.type << endl;

    // XXX Should unparse all the pages
}

extern bool Disk_ReadSuper(Disk* disk, Disk_SB* super);
// modifies - super
// effects  - Reads superblock from "disk" into "super".
//	      Returns true iff successful.

#endif /* _DFORMAT_H */
