// Copyright 1995 Barbara Liskov

/*
\section{MM Handles}

A handle encapsulates an "Oref" and an object pointer into
MM data structures.  The "MM" pin operations return handles.
The client of the MM should "delete" this handle as soon as
the handle is no longer required.

XXX {\em Do not make subclasses of this class.}
*/

#ifndef _HANDLE_H
#define _HANDLE_H

#include <stddef.h>
#include "utils/basic.h"
#include "utils/array.h"
#include "common/oref.h"

struct  OR_obj;
class   CacheEntry;
class	Itable_Mod;
class   MM;
class   Prefetch_Info;

class MM_Handle {
  public:
    Oref      oref()   const;		// Return oref of this object
    OR_obj*   obj()   const;		// Return pointer to pinned memory

    bool fetched_from_itable() const; // whether fetched from itable
    bool fetched_from_disk() const; // whether fetched from disk

  private:
    bool in_itable;			// Is this object in the itable?
    bool from_disk;			// Whether fetching involved disk access
    Oref     oref_;			// The oref that names this object 
    OR_obj*  obj_;			// Object contents

    union {
	CacheEntry*	entry;	// Cache entry pointer
	Itable_Mod*	mod;	// Itable modification
	MM_Handle*	next;	// Next free handle
    };

    // Do not call these routines.  They are defined to prevent the
    // compiler from providing default values for these routines.
    MM_Handle(MM_Handle const&);
    MM_Handle& operator = (MM_Handle const&);

    MM_Handle();
    // effects	Create uninitialized handle.  Used by friend classes
    //		to create handles.

    ~MM_Handle();
    // effects	Unpin the contained object and release any storage
    //		for this handle.  This is made private because an
    //		MM handle should only be deleted when "mm->mutex"
    //		is held.

    // For fast allocation/deallocation
    void operator delete(void*);
    void* operator new(size_t);

    friend MM;
    friend Prefetch_Info; 
};

// Array of handles
declareArray(MM_HandleList,MM_Handle*)

// Inline definition of various methods for speed.
inline MM_Handle::MM_Handle() {
}

inline Oref MM_Handle::oref() const {
    return oref_;
}

inline OR_obj* MM_Handle::obj() const {
    return obj_;
}

inline bool MM_Handle::fetched_from_itable() const {return in_itable;}
inline bool MM_Handle::fetched_from_disk() const {return from_disk;}

#endif /* _HANDLE_H */
