// Copyright 1995 Barbara Liskov

#include <stdio.h>
#include "utils/intcontrol.h"
#include "utils/sort.h"
#include "common/oref.h"
#include "common/slot.h"

#include "or/or.h"

#include "itable.h"
#include "logrecord.h"
#include "mm.h"

// Do NOT provide definitions for the private constructors/destructors
// to let the linker catch any attempts to use them.

#if 0
Itable_Mod::Itable_Mod() {assert(0);}
Itable_Mod::Itable_Mod(Itable_Mod const& x) {assert(0);}
void Itable_Mod::operator = (Itable_Mod const& x) {assert(0);}
#endif


// For debugging ItableMod::unref
Itable_Mod* checkmod(int o) {
   return orx->mm->itable->lookup(o);
}


// The following are actually used for searching hash tables.
Itable_Mod::Itable_Mod(Oref o) {id = o;}
Itable_Mod::~Itable_Mod() {}

// "Itable_Mod" allocation: allocate a chunk big enough to
// hold object contents.
Itable_Mod* Itable_Mod::alloc(Oref oref, Data_Log_Record* lr, OR_obj* object,
			      int num_slots) {
    // "Itable_Mod" definition already has space for one object slot.
    // How many extra slots do we need?
    int object_size = (num_slots + OR_obj_headers) * Slot_size;
    int size = sizeof(Itable_Mod) - Slot_size + object_size;

    Itable_Mod* mod = (Itable_Mod*) new char[size];
    mod->id	= oref;
    mod->pin	= 0;
    mod->lr	= lr;
    mod->fslots = num_slots;

    // Copy object contents
    memcpy((void*) mod->object(), (const void*) object, object_size);

    return mod;
}

// Control information to build a set of "Itable_Mod".

// An "Itable_Mod*" that indicates the hash table slot is obselete
static char junky_value;
static Itable_Mod* obsolete_mod = (Itable_Mod*) &junky_value;

class IM_Control {
  public:
    static int   is_full(Itable_Mod*);
    static int   is_empty(Itable_Mod*);
    static int   is_obsolete(Itable_Mod*);
    static void  make_empty(Itable_Mod*&);
    static void  make_obsolete(Itable_Mod*&);
    static int   hash(Itable_Mod*);
    static int   equal(Itable_Mod*, Itable_Mod*);
};

inline int IM_Control::is_full(Itable_Mod* v) {
    return (v != 0) && (v != obsolete_mod);
}

inline int IM_Control::is_empty(Itable_Mod* v) {
    return (v == 0);
}

inline int IM_Control::is_obsolete(Itable_Mod* v) {
    return (v == obsolete_mod);
}

inline void IM_Control::make_empty(Itable_Mod*& v) {
    v = 0;
}

inline void IM_Control::make_obsolete(Itable_Mod*& v) {
    v = obsolete_mod;
}

inline int IM_Control::hash(Itable_Mod* v) {
    return Oref_sindex(v->oref());
}

inline int IM_Control::equal(Itable_Mod* a, Itable_Mod* b) {
    return (Oref_sindex(a->oref()) == Oref_sindex(b->oref()));
}

// Set of "Itable_Mod"s indexed by "Oref_sindex"
#define HTABLE    IM_Set
#define HTYPE     Itable_Mod*
#define HCONTROL  IM_Control
#include "utils/htable.h"

#define HTABLE    IM_Set
#define HTYPE     Itable_Mod*
#define HCONTROL  IM_Control
#include "utils/htable_p.h"

// The rep is a map from segment number to "IM_Set*".
// It is an extensible array

declareArray(Itable_Rep, IM_Set*)
implementArray(Itable_Rep, IM_Set*)

implementArray(Itable_Mods,Itable_Mod*)

Itable::Itable() {
    rep = new Itable_Rep;
}

Itable::~Itable() {
    delete rep;
}

int Itable::has_modifications(int seg) {
  if (seg >= rep->size() || rep->slot(seg) == NULL)
    return 0;
  else
    return  rep->slot(seg)->size();
}

inline bool Itable::get_segment_info(int segnum, IM_Set*& set) {
    // effects: if information for segnum is present in the Itable rep,
    //          put the info in set and return TRUE. Else return FALSE
    //          and leave set unchanged

    if (segnum >= rep->size() || rep->slot(segnum) == NULL)
	return FALSE;
    set = rep->slot(segnum);
    return TRUE;
}


Itable_Mod* Itable::enter(Oref o, Data_Log_Record* record, OR_obj* obj,
		   int num_slots) {
    IM_Set* set;
    if (get_segment_info(Oref_segment(o), set)) {
	// Cancel old entry if any
	Itable_Mod* mod;
	Itable_Mod key(o);
	if (set->find(&key, mod)) {
	    orx->mm->absorbed++;
            // Remove old mod from the table 
            set->remove(mod);
	    kill_modification(mod);
	}
    }
    else {
	// Create a segment specific map
	set = new IM_Set;
	// Add information about this segment in rep
	int segnum = Oref_segment(o);
	if (segnum >= rep->size())
	    rep->append(NULL, segnum - rep->size() + 1);
	rep->slot(segnum) = set;
    }

    // Make new modification
    Itable_Mod* mod = Itable_Mod::alloc(o, record, obj, num_slots);
    mod->ref();
    set->insert(mod);
    orx->mm->mods++;
    return mod;
}



bool Itable::check_rep(void) {
  for (int segnum=0; segnum < rep->size(); segnum++) {
    if ( rep->slot(segnum) != NULL) {
      Itable_Mod* mod;
      IM_Set::Elements x = rep->slot(segnum);
      while (x.get(mod)) {
	if (Oref_segment(mod->oref()) !=(Uint) segnum)
	  return FALSE;
      }
    }
  }
  return TRUE;
}
 

void Itable::remove(Itable_Mod* mod) {
    // Check whether this modification has been overridden
    Oref oref = mod->oref();
    IM_Set* set;
    if (get_segment_info(Oref_segment(oref), set)) {
	Itable_Mod* current;
	if (set->find(mod, current) && (mod == current)) {
	    // Modification has not been overridden
	    set->remove(mod);
	    if (set->size() == 0) {
		rep->slot(Oref_segment(oref)) = NULL;
		delete set;
	    }
	    kill_modification(mod);
	}
    }
}

void Itable::kill_modification(Itable_Mod* mod) {
    Oref oref = mod->oref();
    OR_obj* object = mod->object();

    // Shrink log size usage
    mod->record()->cancel(oref, object, mod->fslots);
    orx->log->shrink((mod->fslots + OR_obj_headers) * Slot_size);

    // Now we can allow the object to die (provided nobody else is holding it)
    mod->unref();
}

Itable_Mod* Itable::lookup(Oref o) {
    IM_Set* set;
    if (!get_segment_info(Oref_segment(o), set))
	return 0;

    Itable_Mod* mod = 0;
    Itable_Mod key(o);
    set->find(&key, mod);
    return mod;
}

bool Itable::segment_needs(int s, Data_Log_Record* record) {
    IM_Set* set;
    if (!get_segment_info(s, set)) return FALSE;

    Itable_Mod* mod;
    IM_Set::Elements x = set;
    while (x.get(mod)) {
	if (mod->record() == record) return TRUE;
    }
    return FALSE;
}

void Itable::get_modifications(int seg, Itable_Mods* array) {
    IM_Set* set;
    if (!get_segment_info(seg, set))
	// No pending modifications
	return;

    // Loop over the modification set and append all mods to array
    Itable_Mod* mod;
    IM_Set::Elements x = set;
    while (x.get(mod)) {
	mod->ref();
        th_assert(Oref_segment(mod->oref()) == (Uint)seg,
		  "Bogus mod in Itable\n");
	array->append(mod);
    }
}

void Itable::dump() {
    Itable_Mod* mod;

    fprintf(stderr, "Itable:\n");
    for (int segment = 0; segment < rep->size(); segment++) {
	IM_Set* set = rep->slot(segment);
	if (!set) continue;
	fprintf(stderr, "  segment %3d\n", segment);
	IM_Set::Elements m = set;
	while (m.get(mod)) {
	    fprintf(stderr, "    %4d: log record = %p\n",
		    Oref_sindex(mod->oref()), mod->record());
	}
    }
}
