// Cleaner routines

#include "common/arraysort.h"
#include "common/openhashmap.h"
#include "common/hashfuncs.h"
#include "common/intarray.h"
#include "or/or.h"

#include "itable.h"
#include "mm.h"
#include "disk.h"

#include "lmm/finfo.h"
#include "lmm/frag.h"

// Clean when number of empty regions falls to this value.
static const int clean_threshold = 12;

// Clean this many regions at a time.
static const int clean_count = 8;

// List of regions being cleaned
declareArray(Regions,Ldisk_Region*)
implementArray(Regions,Ldisk_Region*)

// Map from segment to object list.  (Used while cleaning)
#define same_seg(a,b) ((a) == (b))
declareOpenHashMap(CleanerMap,int,Itable_Mods*,hash_int,same_seg)
implementOpenHashMap(CleanerMap,int,Itable_Mods*,hash_int,same_seg)

// Sorting regions by region number.
static int compare_regions(void const* p1, void const* p2) {
    int r1 = *((int const*) p1);
    int r2 = *((int const*) p2);
    return r1 - r2;
}

bool SegInfo::clean_disk() {
    // Clean if # of empty regions < cleaning threshold, or if we are
    // supposed to ignore all of the currently empty regions because
    // of disk space utilization constraints.
    int start_empty = empties();
    int threshold = (clean_threshold > ignore) ? clean_threshold : ignore;
    if (start_empty > threshold) return FALSE;

    // Pick regions based on cleaning order.
    IntArray rnums;
    pick_regions(clean_count, &rnums);
    if (rnums.size() == 0) return FALSE;

    // Sort by region number to avoid excessive seeking.
    ArraySort(rnums, compare_regions);

    // Read the regions and get the live objects.
    Regions rlist;
    CleanerMap map;
    for (int i = 0; i < rnums.size(); i++) {
	Disk_Range range;
	range.address = r2b(rnums[i]);
	range.count   = region_size >> DISK_UNIT_SHIFT;

	Ldisk_Region* r = make_region(0);
	bool result = or->mm->disk->read(r, range, Disk_CRead);
	th_assert(result, "could not read region for cleaning");

	or->mm->clean_count++;
	or->mm->clean_live += live[rnums[i]] * sizeof(OR_slot);

	rlist.append(r);
	extract_objects(rnums[i], r, &map);
    }

    write_objects(&map);

    // Delete map contents
    for (CleanerMap::Bindings b = &map; b.ok(); b.next()) {
	Itable_Mods* mods = b.val();
	int num = mods->size();
	for (int i = 0; i < num; i++) {
	    Itable_Mod* m = mods->slot(i);
	    m->unref();
	}
	delete mods;
    }

    // Delete the regions
    for (i = 0; i < rnums.size(); i++)
	delete_region(rlist[i]);

    int finish_empty = empties();
    th_assert(finish_empty > 0, "cleaner did not make empty regions");
    fprintf(stderr, "cleaned %2d regions: %2d -> %2d empty regions\n",
	    rnums.size(), start_empty, finish_empty);
    return TRUE;
}

void SegInfo::extract_objects(int rnum, Ldisk_Region* r, CleanerMap* map) {
    OR_slot* addr = r->buf;
    int start = r2s(rnum) + ((char*)addr - (char*)r) / sizeof(OR_slot);

    int segcount = r->frags;
    for (int i = 0; i < segcount; i++) {
	// Process next fragment
	Ldisk_Fragment* frag = (Ldisk_Fragment*) addr;
	FragRange f;
	f.address = start;
	f.size    = frag->slots;

	addr += frag->slots;
	start += frag->slots;
	
	int objcount = frag->count;
	Itable_Mods* mlist = 0;
	for (int j = 0; j < objcount; j++) {
	    Oref oref;
	    OREF_SET(oref,frag->id,frag->header[j].id);
	    if (find_region(oref) != rnum) {
		// This version of the object is not live.
		continue;
	    }

	    if (mlist == 0) {
		// Get modification list for this segment.
		if (!map->fetch(frag->id, mlist)) {
		    mlist = new Itable_Mods(frag->count);
		    map->store(frag->id, mlist);
		}
	    }

	    // Add object to cleaner map
	    OR_obj* obj = (OR_obj*) (((OR_slot*) frag)+frag->header[j].offset);
	    Itable_Mod* m = Itable_Mod::alloc(oref, 0, obj);
	    m->ref();
	    mlist->append(m);
	}
    }
}

static long* segment_ages = 0;

// Routine for comparing segment ages
static int compare_segment_age(void const* p1, void const* p2) {
    int s1 = *((int const*) p1);
    int s2 = *((int const*) p2);

    return segment_ages[s1] - segment_ages[s2];
}

void SegInfo::write_objects(CleanerMap* map) {
    // Get list of segments
    IntArray segs(map->size());
    for (CleanerMap::Bindings b = map; b.ok(); b.next())
	segs.append(b.key());

    // Sort segments by age
    segment_ages = mtime;
    ArraySort(segs, compare_segment_age);
    segment_ages = 0;

    // Fill regions with fragments sorted by age.
    Ldisk_Region* dest = 0;
    int segcount = segs.size();
    for (int i = 0; i < segcount; i++) {
	int seg = segs[i];
	Itable_Mods* mods;
	if (!map->fetch(seg, mods))
	    th_assert(0, "cleaner could not find objects");

	if (dest == 0) dest = make_region(or->mm->new_stamp());
	if (!add_fragment(dest, seg, mtime[seg], mods)) {
	    // Move to next region
	    write_region(dest);
	    delete_region(dest);
	    dest = make_region(or->mm->new_stamp());
	    bool result = add_fragment(dest, seg, mtime[seg], mods);
	    th_assert(result, "fragment contents do not fit in region");
	}
    }

    if (dest != 0) {
	write_region(dest);
	delete_region(dest);
    }
}

void SegInfo::write_region(Ldisk_Region* r) {
    // Pick an empty region
    int region;
    bool result = pick_empty(region);
    th_assert(result, "could not find empty region");

    // Write region contents to disk
    Disk_Range range;
    range.address = r2b(region);
    range.count   = region_size >> DISK_UNIT_SHIFT;
    result = or->mm->disk->write(r, range, Disk_CWrite);
    th_assert(result, "region write failed");

    // Now update object location info
    OR_slot* addr = r->buf;
    int start = r2s(region) + ((char*)addr - (char*)r) / sizeof(OR_slot);

    int segcount = r->frags;
    for (int i = 0; i < segcount; i++) {
	// Process next fragment
	Ldisk_Fragment* frag = (Ldisk_Fragment*) addr;
	FragRange f;
	f.address = start;
	f.size    = frag->slots;

	addr += frag->slots;
	start += frag->slots;
	add_seg(region, frag->id);

	int objcount = frag->count;
	for (int j = 0; j < objcount; j++) {
	    Oref oref;
	    OR_obj* obj = (OR_obj*) (((OR_slot*) frag)+frag->header[j].offset);
	    OREF_SET(oref, frag->id, frag->header[j].id);
	    moved(oref, OR_obj_full_size(obj), f);
	}
    }
}
