#ifndef _FINFO_H
#define _FINFO_H

#include "common/bits.h"
#include "common/oref.h"
#include "common/array.h"
#include "ldformat.h"

class orefs;
class RegionList;

// Information kept about fragments for log-structured layout policy.

// A fragment id is encoded within an integer to allow us to
// reuse the "mm/scache" code.  We have at most one fragment
// for a given segment in one region, and therefore we combine
// the segment id with the region # to get the fragment id.
//
// The segment id is currently at most 16 bits.  This leaves
// another 16 bits for the region id.  This should be sufficient
// because with half-megabyte regions, even a large disk will
// not have very many regions.

inline int make_fid(int segment, int region) {
    // effects	Create fragment id for "<segment, region>"
    return (int) ((((unsigned int) region) << 16) |
		  (((unsigned int) segment)));
}

inline int f2seg(int id) {
    // effects	Return segment # for named fragment
    return (id & 0xFFFF);
}

inline int f2region(int id) {
    // effects	Return region # for named fragment
    return ((int) (((unsigned int) id) >> 16));
}

// Information kept per fragment is the address of the
// fragment on the disk, and its size.
struct FragRange {
    int  address;		// Disk address (in slots)
    int  size;			// Size (in slots);
};

class FragRanges;
class CleanerMap;

// Map from objects to fragment range.
class FragMap {
  public:
    FragMap();
    ~FragMap();

    FragRange& operator [] (Oref);

    int max_index(int seg);
    // effects	Return the index of the largest object that belongs
    //		"seg" and has a non-null entry in this map.
  private:
    FragRanges* rep;
};

// Segment information
//	* Mapping from object -> fragment info
//	* Most recent fragment info

class SegInfo {
  public:
    SegInfo();
    // effects	Initialize info with simple clustered layout

    void init();
    // effects	Initialize the disk.

    void recover();
    // effects	Recover location info from disk.

    int num_regions();
    // effects	Return number of regions on disk.

    int alloc_seg();
    // effects	Allocate segment number

    void add_seg(int r, int seg);
    // requires	Region "r" does not currently contain "seg".
    // effects	Mark "r" as containing "seg".

    void moved(Oref oref, int slots, FragRange f);
    // effects	Mark object with specified number of slots and named by "oref"
    //		as being stored at "f".

    void modified(int seg, long t);
    // effects	Mark "seg" as modified at time "t".

    void find_object(Oref obj, FragRange& f);
    // effects	Return fragment info for "obj".

    int find_region(Oref obj);
    // effects	Return # of region that contains "obj".

    bool pick_empty(int& r);
    // modifies	"r"
    // effects	If an empty region exists, set "r" to that region and
    //		return true.  Else return false.

    int empties();
    // effects	Return number of empty regions.

    void pick_regions(int n, IntArray* list);
    // modifies	"list"
    // effects	Picks "n" disk regions to clean based on some disk
    //		cleaning policy.  Appends the picked regions to "list".

    void live_segments(int region, IntArray* segments);
    // effects	Append list of segments with live data on "region",
    //		to "segments".

    void live_objects(int region, int segment, orefs* objects);
    // effects	Append the set of live objects that belong to "segment"
    //		and are stored on "region" to "objects".

    int live_ocount(int region, int segment);
    // effects	Return the number of live that belong to "segment"
    //		and are stored on "region".

    long age(int region);
    // effects	Return the time of the latest modification to any
    //		segment that has live data in "region".

    double goodness(int r);
    // effects	Return "goodness" of region "r" for cleaning.
    //		I.e. return a large number if "r" is suitable for
    //		cleaning, a small number otherwise.

    void check();
    // effects	Rep invariant checks

    bool clean_disk();
    // effects	Make room on the disk if necessary.  Returns true
    //		iff the cleaner ran.

    void resize_dspace(int p);
    // effects	Change disk space utilization.
  private:
    FragMap	map;		// Segment -> Object -> Fragment map
    long*	mtime;		// Modify time per segment

    int		regions;	// Number of regions in disk
    int		ignore;		// Number of regions to ignore
    int		next_seg;	// Next segment to allocate
    int*	live;		// Number of live slots per region
    IntArray*	segs;		// Per region set of contained segments

    // The current cleaning policy evaluator
    double (*evaluator)(SegInfo*, int);

    // Evaluators for cleaning policy
    static double eval_greedy		(SegInfo* info, int region);
    static double eval_cost_benefit	(SegInfo* info, int region);
    static double eval_worst		(SegInfo* info, int region);

    // Methods for recovery

    void get_region_info(int regions, RegionList&);
    // effects	Return array of region headers.

    void recover_region(int rnum, Ldisk_Region* r);
    // effects	Recover from specified region.

    // Methods for cleaning

    void extract_objects(int region, Ldisk_Region* r, CleanerMap* map);
    // modifies	"map"
    // effects	Extract live objects from "r" into "map".

    void write_objects(CleanerMap* map);
    // effects	Write out the objects stored in "map".

    void write_region(Ldisk_Region* r);
    // effects	Write out region contents and update location info.

    void clear_region(int rnum);
    // requires	No more live data in region named by "rnum".
    // effects	Clear corresponding info and invalidate corresponding
    //		cache entries.
};

#endif /* _FINFO_H */
