// Copyright 1995 Barbara Liskov

// \section{Log Implementation}
//
// The contents of the log are managed in a circular buffer.
// Usually the log should not exceed a certain number of entries
// and therefore the log should wrap around in the circular
// buffer as entries are added and removed from the log.
// However, if entries are not being deleted from the log for
// some reason, the circular buffer is grown as needed.

// The log is protected from concurrent access by a mutex.

#include <iostream.h>
#include <stdio.h>
#include "utils/th_assert.h"
#include <errno.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/param.h>
// #ifdef __linux__
// #include <pth.h>
// #endif

// XXX The apply process is too eager at the moment.  It is woken up
// very often.  Plus it only applies a record at a time.

#include "or/thread.h"
#include "or/or.h"
#include "or/or_config.h"

#include "or/gc/gc.h"

#include "log.h"
#include "logrecord.h"
#include "mm.h"


#include "utils/other_unix.h"
#include "utils/compat.h"
#include "utils/network.h"
#include "utils/bits.h"
#include "utils/Timer.h"
#include "common/locator.h"
#include "utils/communication.h"
#include "utils/intarray.h"
#include "or/mm/log.h"
#include "or/mm/mm.h"
#include "config/vdefs/REPLICATION.h"

#if REPLICATION
 #include "rep.h"
 #include "back_send_message.h"
 #include "back_recv_message.h"
 #include "prim_send_message.h"
 #include "prim_recv_message.h"
#endif

Uint const Default_net_bufsize = 50*1024; // For the network buffers

// Each log entry contains a pointer to the actual log record,
// and the log record state
    


struct Log_Entry {
    Log_Record* record;			// The actual record
    int		installed : 1;		// Is record installed?
    int		applied   : 1;		// Is record applied?
};

// Routine to calculate percentage of a number
inline int percent(int num, int percent) {
    return (int) (num * (((double) percent) / 100.0));
}

// \subsection{Log Threads}

typedef void (Log::*Log_Proc)();

class Log_Thread : public Thread {
  public:
    Log_Thread(Log* l, Log_Proc p);
    virtual void main();
  private:
    Log* log;
    Log_Proc proc;
};

Log_Thread::Log_Thread(Log* l, Log_Proc p) {
    log = l;
    proc = p;
}

void Log_Thread::main() {
    // All the work is done inside a "Log" method
    (log->*proc)();
}

/* \subsection{Exported Operations} */

Log::Log() {
    mutex = new Mutex;
    apply_wait = new Condition(mutex);
    flush_wait = new Condition(mutex);
    flushers   = new Condition(mutex);
    space_wait = new Condition(mutex);
    death_wait = new Condition(mutex);

    // Create a small array initially
    list = new Log_Entry[1];
    size = 1;

    logsize = 0;
    max_logsize = orx->config->log_size;
    apply_start = percent(max_logsize, orx->config->apply_threshold);
    apply_stop  = percent(max_logsize, orx->config->target_percent);

    start = 0;
    count = 0;
    next_index = 0;
    next_flush = 0;

    
    draining = 0;
    dying = 0;
    drain_wait = new Condition(mutex);

    // stats
    flush_size = 0;
    flush_time = 0;
    max_logrecord = 0;

#if REPLICATION
    fmutex = new Mutex;
    flush_thread_sync_wait = new Condition(fmutex);

    next_tobe_marshalled = 0;
    immediate_flush = false;
    allow_flushing = false;

    char local_host_name[MAXHOSTNAMELEN];
    if ( gethostname(local_host_name, MAXHOSTNAMELEN) ) 
        fail("error getting local host name\n");

    //assign backup_or_num depending on which host it is
    if ( strcmp(local_host_name,BACKUP1_NAME)== 0 ) {
        local_or_num = BACKUP1_OR_NUM;
    }
    else if ( strcmp(local_host_name,BACKUP2_NAME) == 0 ) {
        local_or_num = BACKUP2_OR_NUM;
    }
    else if ( strcmp(local_host_name, PRIMARY_NAME) == 0 ) {
        local_or_num = PRIMARY_OR_NUM;
    }
    else fail("Backup not in replication setup\n");
#endif


    // Fork off the apply thread
    Thread* a = new Log_Thread(this, &Log::apply_thread);
    printf("Starting apply thread\n"); //DEBUG
    a->start(TH_MIN_PRIO);

    // Fork off the flush thread
    Thread* f = new Log_Thread(this, &Log::flush_thread);
    printf("Starting flush thread\n"); //DEBUG
    f->start(TH_MAX_PRIO);

#if REPLICATION
    if ( local_or_num != PRIMARY_OR_NUM ) {
      Thread* b = new Log_Thread(this, &Log::backup_flush_thread);
      printf("Starting backup flush thread\n"); //DEBUG
      b->start();
    }
    
    if (orx->config->with_backups) {
      fprintf(stderr,"Waiting for flush threads to synchronise....");
      // wait for flush threads to sync
      fmutex->grab();
      flush_thread_sync_wait->wait();
      fmutex->release();
      fprintf(stderr,"done\n");
    }
#endif    

}

Log::~Log() {
    delete [] list;
    delete apply_wait;
    delete flush_wait;
    delete drain_wait;
    delete death_wait;
    delete flushers;
    delete mutex;

#if REPLICATION 
    delete fmutex;
#endif

    // XXX Figure out how to kill the apply/flush threads
}

void Log::resize(int bytes) {
    mutex->grab(); {
	max_logsize = bytes;
	apply_start = percent(max_logsize, orx->config->apply_threshold);
	apply_stop  = percent(max_logsize, orx->config->target_percent);

	if (ready_to_apply()) apply_wait->signal();
    } mutex->release();
    // synchronous_apply(); // only for experiments
}

Log_Index Log::append(Log_Record* rec, bool client) {
    Log_Index result;

    mutex->grab(); {
	while (client && dying) {
	    // Client threads are not allowed to append stuff while OR
	    // is shutting down.  Just put the thread to sleep.  It
	    // will die when the OR dies.
	    death_wait->wait();
	}

	int recsize = rec->size();
	if (recsize > (int)max_logrecord) max_logrecord = recsize; // stats
	while ((logsize + recsize) > max_logsize) {
	    if (recsize > max_logsize) 
		fail("Record bigger than max log size, %d bytes", recsize);
	    if (logsize < apply_start) {
		// the apply thread would think there is nothing to do
		must_apply = TRUE;
		if (apply_stop > max_logsize - recsize) 
		    // make sure that enough space would be freed
		    apply_stop = max_logsize - recsize;
		apply_wait->signal();
	    }
//  #ifdef __linux__
//  	    pth_yield(NULL);
//  #endif
	    space_wait->wait();
	}

	if (count >= size)
	    enlarge();

	count++;
	Log_Entry* e = &(list[(start + count - 1) % size]);
	e->record = rec;
	e->installed = 0;
	e->applied = 0;
	next_index++;
	logsize += rec->size();
	
	if (ready_to_flush()) flush_wait->signal();
	if (ready_to_apply()) apply_wait->signal();

	result = next_index-1;

    } mutex->release();
    // synchronous_apply(); // only for experiments

    return result;
}

Log_Index Log::high() const {
    Log_Index h;
    mutex->grab(); {
	h = next_index-1;
    } mutex->release();
    return h;
}

Log_Index Log::low() const {
    Log_Index l;
    mutex->grab(); {
	l = next_index - count;
    } mutex->release();
    return l;
}

bool Log::is_stable(Log_Index index) const {
    bool result;
    mutex->grab(); {
	result = (index < next_flush);
    } mutex->release();
    return result;
}

Log_Index Log::get_next_flush() {
  Log_Index result;

  mutex->grab();
    result= next_flush;
  mutex->release();

  return result;
}

bool Log::is_installed(Log_Index index) const {
    bool result;
    mutex->grab(); {
	int pos = find(index);
	if (pos < 0) {
	    // Index is out of range.  Has been installed only if index
	    // is old as opposed to new.
	    result = (index < next_flush);
	}
	else {
	    result = list[pos].installed;
	}
    } mutex->release();
    return result;
}

Log_Record* Log::fetch(Log_Index i) const {
    Log_Record* result;

    mutex->grab(); {
	int index = find(i);
	if (index < 0) {
	    result = 0;
	}
	else {
	    result = list[index].record;
	}
    } mutex->release();

    return result;
}

#if REPLICATION
Log_Record* Log::get_next_record_tobe_marshalled() {
        Log_Record* result;

    mutex->grab(); {
	if ( next_tobe_marshalled < next_flush || next_tobe_marshalled > next_index) {
	  fprintf(stderr,"Log:next_tobe_marshalled out of sync\n");
 
	  result =  0;
	} 
        else {
	    int index = find(next_tobe_marshalled);
	    if (index < 0) {
	        result = 0;
	     }
	     else {
	        result = list[index].record;
	     }
	     next_tobe_marshalled++;
	} 
     } mutex->release();

     return result;
}

void Log::set_marshall_number(Log_Index i) {
  mutex->grab(); {
        next_tobe_marshalled = i;
  } mutex->release();

}

void Log::set_allow_flushing() {
  mutex->grab(); {
        allow_flushing = true;
  } mutex->release();

}

#endif

void Log::flush(Log_Index i) {
#if REPLICATION
  if (allow_flushing) {
#endif
    mutex->grab(); {
#if REPLICATION
        immediate_flush = true;
#endif
	while (i >= next_flush) {
	    // Wakeup flush thread if necessary

	    flush_wait->signal();

	    // Wait for flush thread to do something

	    flushers->wait();


	}
#if REPLICATION
        immediate_flush = false;
#endif
    } mutex->release();
#if REPLICATION
  }
  else {
    next_flush = next_index;
    next_tobe_marshalled = next_flush;
    if (ready_to_apply()) apply_wait->signal();
    flushers->broadcast();
  }
#endif
}

void Log::installed(Log_Index i) {
    mutex->grab(); {
	int index = find(i);
	if ((index >= 0) && !list[index].installed) {
	    list[index].installed = 1;
	    if (ready_to_apply()) apply_wait->signal();
	}
    } mutex->release();
    // synchronous_apply(); // only for experiments
}

void Log::applied(Log_Index start_index, Log_Index finish_index) {
    mutex->grab(); {
	// Mark these records as applied

	for (int i = start_index; (Uint) i <= finish_index; i++) {
	    int index = find(i);
	    if ((index >= 0) && !list[index].applied) {
		list[index].applied = 1;
	    }
	}

	// We now try to delete all applied records at the low end
	int removed = 0;
	int oldsize = logsize;
	while ((count > 0) && list[start].applied) {
	    logsize -= list[start].record->size();
	    delete list[start].record;
	    list[start].record = 0;
	    start = (start + 1) % size;
	    count--;
	    removed++;
	}

	if (removed > 0) {
	    if (orx->config->debug_level > 0)
		fprintf(stderr, "log: %8d -> %8d bytes\n", oldsize, logsize);

	    th_assert(count >= 0, "log count is invalid after applying");

	    // Wake-up drainer if present

	    if (draining && (count == 0)) {
		draining = 0;
		drain_wait->broadcast();
	    }

	    // Wake-up threads waiting for log space
	    space_wait->broadcast();
	}
    } mutex->release();
}

void Log::drain() {

    flush(high());
    
    mutex->grab(); {

	while (count > 0) {
	    draining = 1;
	    apply_wait->signal();
	    drain_wait->wait();

	}
    } mutex->release();
}

void Log::shutdown() {
    if (gc) {
	gc->print_stats(&cout);
	orx->mm->set_gc(FALSE); // stop the GC
    }
    fprintf(stderr, "Draining the log...");
    mutex->grab(); {
	dying = 1;

    } mutex->release();
    drain();

#if REPLICATION
   if (orx->config->with_backups && local_or_num == PRIMARY_OR_NUM) { 
     Prim_send_shutdown_msg* down_msg = new Prim_send_shutdown_msg(1);
     Prim_recv_shutack_msg* downack_msg = new Prim_recv_shutack_msg();

     comm.send_recv(OR_address(BACKUP1_OR_NUM), down_msg, downack_msg );
     if (orx->config->two_backups) {
       comm.send_recv(OR_address(BACKUP2_OR_NUM), down_msg, downack_msg );
     }

  }
#endif

    fprintf(stderr, "done.\n");
    exit(0);
}

int Log::current_size() const {
    int result;
    mutex->grab(); {
	result = logsize;
    } mutex->release();
    return result;
}

int Log::target_size() const {
    int result;
    mutex->grab(); {
      // If the log is being drained, then the target size is 0.
      result = (draining ? 0 : apply_stop);
    } mutex->release();
    return result;
}

void Log::shrink(int bytes) {
    mutex->grab(); {
	assert(logsize >= bytes);
	logsize -= bytes;
    } mutex->release();
}

/* \subsection{Internal Operations} */

int Log::find(Log_Index i) const {
    Log_Index l = next_index - count;
    Log_Index h = next_index - 1;

    if ((i < l) || (i > h)) {
	return -1;
    }
    else {
	return (start + (i - l)) % size;
    }
}

void Log::enlarge() {
    // Enlarge the buffer
    int new_size = size*2;
    Log_Entry* new_list = new Log_Entry[new_size];
    for (int i = 0; i < count; i++) {
	new_list[i] = list[(start + i) % size];
    }

    delete [] list;
    list = new_list;
    size = new_size;
    start = 0;
}

bool Log::ready_to_apply() {
    // First make sure at least something is ready to be applied
    Log_Index lowest = next_index - count;

    if (!((count > 0) && (lowest < next_flush) && list[start].installed))
	return 0;

    if (must_apply) {
	must_apply = FALSE;
	return TRUE;
    }

    return (draining || (logsize >= apply_start));
}

bool Log::ready_to_flush() {
    // XXX Currently, just see if there is an unflushed record.

    // It might make sense to not get ready to flush until
    // we accumulate a bunch of log records, or somebody is
    // waiting for a flush.

#if REPLICATION    
  if ( immediate_flush ) 
#endif
    return ((count > 0) && (next_index > next_flush));
#if REPLICATION    
  else 
    return ((count > 0) && (next_index > next_flush + 10));
  // XXX unless immediate flush, wait for 10 records before flushing
#endif

}

/* \subsection{Threads Attached to the Log} */

void Log::apply_thread() {
    while (1) {
	mutex->grab(); {

	    // Wait for something to apply
	  while (!ready_to_apply()) {
		apply_wait->wait();

	  }
//  #ifdef __linux__
//  	  pth_yield(NULL);
//  #endif
	} mutex->release();
	printf("Log::apply_thread calling orx->mm->clean_log\n"); //DEBUG
	orx->mm->clean_log();

    }
}

void Log::synchronous_apply() {
    bool to_apply;
    mutex->grab();
    to_apply = ready_to_apply();
    mutex->release();
    if (to_apply) {
	orx->mm->clean_log();
    }
}

// Network error handler that prevents OR crashes.
static void print_warning(Device*, const char* msg) {
    warn(msg);
}

#if REPLICATION

void Log::flush_thread() {

 if ( orx->config->with_backups) {

    OR_num backup1_or_num = BACKUP1_OR_NUM;
    Address backup1_address = OR_address(backup1_or_num);
    Ubits32 backup1_hostid; int backup1_port;

    OR_num backup2_or_num = BACKUP2_OR_NUM;
    Address backup2_address = OR_address(backup2_or_num);
    Ubits32 backup2_hostid; int backup2_port;

    Address local_address;

    local_address = OR_address(local_or_num);

    // If PRIMARY then do this
    if (local_or_num == PRIMARY_OR_NUM ) { 

      //Setup connection to backup2
      //get address and port of backup2
      bool success;

      if ( orx->config->two_backups) {
	if (!locator.lookup(backup2_address, &backup2_hostid, &backup2_port))
	  fail("OR Address not found in locator");

	backup2_port++; // the main OR processes listen in on backup2_port

	// setup network connection to backup2
	Network *net_backup2 = new Network(backup2_address, backup2_hostid, backup2_port, success);
	th_assert(net_backup2 && success, "Could not connect to the backup 2");
	net_backup2->set_nodelay();
	net_backup2->address = backup2_address;
	comm.include(net_backup2);        
      }

      // Setup connection to backup1
 
      //get address and port of backup1
      if (!locator.lookup(backup1_address, &backup1_hostid, &backup1_port))
     	fail("OR Address not found in locator");

      backup1_port++; // The main or processes listen in on the backup1_port

      // setup network connection to backup1
      Network *net_backup1 = new Network(backup1_address, backup1_hostid, backup1_port, success);
      th_assert(net_backup1 && success, "Could not connect to the backup1");
      net_backup1->set_nodelay();
      net_backup1->address = backup1_address;
      comm.include(net_backup1);


      // signal the main thread to continue after log setup.
    
      flush_thread_sync_wait->signal();

      Ubits32 no_to_flush;
      Prim_send_data_msg *data_msg;
      Prim_recv_ack_msg *register_msg;
      Log_Index old_next_index;

      while (1) {
	   // Wait for something to flush
	   mutex->grab();

           while (!ready_to_flush()) {
	     flush_wait->wait();
	   }

//  #ifdef __linux__
//  	   pth_yield(NULL);
//  #endif

           no_to_flush = next_index - next_flush;
	   old_next_index = next_index;
	   mutex->release();

	   data_msg = new Prim_send_data_msg(this, no_to_flush);
           register_msg = new Prim_recv_ack_msg();

           comm.send_recv(backup1_address, data_msg, register_msg);
           if (orx->config->two_backups) {

	     set_marshall_number(next_flush);
	     comm.send_recv(backup2_address, data_msg, register_msg);
	   }
      
	   delete [] data_msg;
	   delete [] register_msg;

       
	    mutex->grab();
	    next_flush = old_next_index;
	    mutex->release();

	    if (ready_to_apply()) apply_wait->signal();
	     
	    // Wakeup all threads waiting for a log record
	    // to get flushed.  Hopefully, not too many
	    // threads will be waiting for a flush.
	    // We might have to do something more complicated
	    // if this assumption is incorrect.

	    flushers->broadcast();
      
      } 

    }
    // if backup then flush should return immediately   
    else {
        
	flush_thread_sync_wait->signal();

        mutex->grab(); {
	  while (1) {
	    while ( !ready_to_flush() )  
	      flush_wait->wait();
//  #ifdef __linux__
//  	    pth_yield(NULL);
//  #endif
            next_flush = next_index;
	    if (ready_to_apply()) apply_wait->signal();
	    flushers->broadcast();
	  }
	}mutex->release();
    }
  }
 // if OR is configured without backups then flush call at primary
 // returns immediately
  else {
       mutex->grab(); {
	  while (1) {
	    while ( !ready_to_flush() )  
	      flush_wait->wait();
//  #ifdef __linux__
//  	    pth_yield(NULL);
//  #endif
	    if (orx->config->simulate_log_flush) {
	      // put statistics here
	       ;
	    }
            next_flush = next_index;
	    if (ready_to_apply()) apply_wait->signal();
	    flushers->broadcast();
	  }
	}mutex->release();
 }
}

void Log::backup_flush_thread() {
    int backup_sock;

    OR_num backup1_or_num = BACKUP1_OR_NUM;
    Address backup1_address = OR_address(backup1_or_num);

    OR_num backup2_or_num = BACKUP2_OR_NUM;
    Address backup2_address = OR_address(backup2_or_num);
    Ubits32 backup2_hostid; int backup2_port;

    OR_num primary_or_num = PRIMARY_OR_NUM;
    Address primary_address = OR_address(primary_or_num);

    Address local_address;
    Ubits32 local_hostid; int local_port_no;

    local_address = OR_address(local_or_num);

    // if the local OR is a backup
    if ( local_or_num == BACKUP1_OR_NUM || local_or_num == BACKUP2_OR_NUM) { 
       
      fprintf(stderr,"Local OR is configured as backup. Starting receipt driven flushing.\n");
     //given backup_or_num get backup_hostid and backup_port_no
     if (!locator.lookup(local_address, &local_hostid, &local_port_no))
     	fail("OR Address not found in locator");

     local_port_no++; // The main or processes listen in on the local_port_no

    // set up socket
     if ((backup_sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	sysfail("socket");
     }

     int opt=1;
     if ( setsockopt(backup_sock, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt)) < 0 ) {
         sysfail("setsockopt");
     }
    

     // Bind address
     struct sockaddr_in addr;
     addr.sin_family = AF_INET;
     addr.sin_addr.s_addr = INADDR_ANY;
     addr.sin_port = htons(local_port_no); 
     if (bind(backup_sock, (struct sockaddr*) &addr, sizeof(addr)) < 0) {
	sysfail("bind");
     }
    

     // Allow connections
     if (listen(backup_sock, 5) < 0) {
	sysfail("listen");
     }

     // **********************************************************
     // Accept connection from the primary

     struct sockaddr_in from;
     int from_size = sizeof(from);
     int primary_connection;


     if ((primary_connection = accept(backup_sock, (struct sockaddr*) &from, &from_size)) < 0)
        {
	    if ((errno == EINTR) )
	       perror("errno == eintr\n");

	    syswarn("accept");
     	}

	// setup the network connection to the primary

     Network* primary_net = new Network(primary_address, primary_connection);
     primary_net->set_handler(print_warning);
     primary_net->set_nodelay();
     primary_net->set_buffsizes(Default_net_bufsize, Default_net_bufsize);
     primary_net->address = primary_address;
     comm.include(primary_net);
    


     // ***************************************************
     // setup the backup-backup connection

    if (orx->config->two_backups) { 
      if (local_or_num == BACKUP1_OR_NUM) {
        // if this process is backup1 then setup the connection to the
        // other backup
    
         // connect to the backup2
         if (!locator.lookup(backup2_address, &backup2_hostid, &backup2_port))
	     fail("OR Address not found in locator");

         backup2_port++;

         bool success;

         // setup the network connection to the other backup
         Network *backup_net = new Network(backup2_address, backup2_hostid, backup2_port, success);
         th_assert(backup_net && success, "Could not connect to the backup1");
         backup_net->set_nodelay();
         backup_net->address = backup2_address;
         comm.include(backup_net);
      
      }
      else {
              
         // If this process is backup2 then accept connection from 
        // the other backup
         
         int backup_connection;              
 
         if ( (backup_connection = accept(backup_sock, (struct sockaddr*) &from, &from_size)) < 0 ) {
	      if ( errno == EINTR ) 
		   perror("errno == EINTR\n");
               syswarn("accept");
	 }

         Network *backup_net = new Network(backup1_address, backup_connection);
	 backup_net->set_handler(print_warning);
	 backup_net->set_nodelay();
         backup_net->set_buffsizes(Default_net_bufsize, Default_net_bufsize);
	 backup_net->address = backup1_address;
         comm.include(backup_net);

      } 
    }
    
     fprintf(stderr, "OR is ready as backup.\n");

     Back_recv_data_msg* data_msg = new Back_recv_data_msg(or);
     comm.register_handler(data_msg);

     Back_recv_shutdown_msg* down_msg = new Back_recv_shutdown_msg();
     comm.register_handler(down_msg);

     int t;

      while (1) {
          bool temp=TRUE; 
          if ( (t=comm.handle_message (-1, temp)) < 0 ) { 
	       break;
          }
//  #ifdef __linux__
//  	  pth_yield(NULL);
//  #endif
      }

  } //end backup

}

#endif

// XXX how to comment this out if REPLICATION is defined?

#if !REPLICATION
void Log::flush_thread() {
    // XXX This thread just marks everything as flushed right away.
    //	   This means that the log really is not stable.

    mutex->grab(); {
	while (1) {
	    // Wait for something to flush
	    while (! ready_to_flush())
		flush_wait->wait();
//  #ifdef __linux__
//  	    pth_yield(NULL);
//  #endif
	    // XXX Obviously, the following should be made real.
	    if (orx->config->simulate_log_flush) {
		// Simulate the time taken for a flush.
		// First compute cummulative size of records to be flushed.
		for (unsigned i = next_flush; i < next_index; i++) {
		    int index = find(i);
		    if (index < 0) continue;
		    int recsize = list[index].record->size();
		    flush_size += recsize;
		}
		simulate_flush();
	    }

	    next_flush = next_index;
	    if (ready_to_apply()) apply_wait->signal();

	    // Wakeup all threads waiting for a log record
	    // to get flushed.  Hopefully, not too many
	    // threads will be waiting for a flush.
	    // We might have to do something more complicated
	    // if this assumption is incorrect.
	    flushers->broadcast();
	}
    } mutex->release();
    // synchronous_apply(); // only for experiments
}

void Log::simulate_flush() {
    int usecs = 5000;  // rotational latency at 5ms
    // No seek latency because log is written sequentially.
    usecs += flush_size/5; // transfer time at 5MB/s
    usleep(usecs);
    flush_time += usecs; // statistics
    flush_size = 0;
}

#endif

void Log::stat(OR_stat& s) {
    s.logflush_time = flush_time;
    s.max_logrecord = max_logrecord;
}
