// Copyright 1995 Barbara Liskov

/*
\section{Log Interface}

The log maintains information about committing transactions as well as
committed transactions whose modifications have not been applied to
the database proper.  The log is designed so that it can be easily
flushed to a backup for replication.

\subsection{Caution}

The log may be used by multiple threads.  Therefore the results
of a previous call may no longer be valid.
*/

#ifndef _LOG_H
#define _LOG_H

#include "utils/basic.h"
#include "common/or_stat.h"
#include "logindex.h"
#include "or/thread.h"
#include "common/locator.h"
#include "utils/communication.h"
#include "config/vdefs/REPLICATION.h"

class Log_Entry;
class Log_Record;
class Mutex;
class Condition;

#if REPLICATION
class Locator;
class Communication;
#endif

class Log {
  public:
    Log();
    ~Log();

/*
\subsection{Adding Records}

Log records are heap-allocated by the caller and stored in the log.  Once a
log record is put into the log, the log assumes responsbility for
deleting the log record.  (The apply process will most likely actually
delete the log record, but at this level the apply process is considered
part of the log.)
*/
    Log_Index append(Log_Record* rec, bool client = TRUE);
    // requires	- rec is a heap-allocate log record.
    //		  Client threads should always pass in "true" for
    //		  the second argument.
    // modifies	- this
    // effects	- Add rec to the end of the log and return its
    //		  index within the log.  The log is now responsible
    //		  for deleting rec.  The log may delete rec as soon
    //		  as its state changes to "installed".  The "client"
    //		  argument is set to false by internal users of the
    //		  log.  Internal users of the log are allowed to add
    //		  records to the log even while the log is being
    //		  drained during shutdown.

/*
\subsection{Scanning Through a Log}

The following operations allow us to scan through the Log.
"high() < low()" for an empty log.
*/

    Log_Index high() const;
    // effects	- Return index of newest log entry

    Log_Index low() const;
    // effects	- Return index of oldest log entry

    Log_Record* fetch(Log_Index i) const;
    // effects	- Return the log record corresponding to index i.
    //		  Returns 0 if (high() < i) or (low() > i).

#if REPLICATION
    Log_Record* get_next_record_tobe_marshalled();
    // Important the thread in which this is called must have already
    // grabbed the mutex.
    //effects -  returns the next record that is to be marshalled into
    // a message. Called by functions that are sending log records to backup
#endif
    	
/*
\subsection{Log Record State}

Each log record has a number of boolean properties.
\begin{description}
\item[valid]
	The record exists.  This is true iff the record index is
	in the range "low()..high()".
\item[stable]
	The record has been flushed successfully to the backing store
	for the log.
\item[installed]
	The contents of the record have been installed in the local
	object location structure.  Therefore fetches at this OR will
	observe the modifications recorded in this log record.
\item[applied]
	The changes have been applied to disk.
\item[applied_all]
	The contents of the record have been applied at all replicas.
	Therefore this record is free to be deleted.
\end{description}
 */

    // Temporarily disabled
    //bool is_valid	(Log_Index i) const;
    //bool is_applied	(Log_Index i) const;
    //bool is_applied_all	(Log_Index i) const;

    bool is_stable(Log_Index i) const;
    // effects	- These routines test the property of the log record
    //		  named by "i".  The caller should take care that the
    //		  system contains various threads that change these
    //		  properties.  E.g., the "apply" thread marks log
    //		  records as "applied".

    bool is_installed(Log_Index i) const;
    // effects	- These routines test the property of the log record
    //		  named by "i".  The caller should take care that the
    //		  system contains various threads that change these
    //		  properties.  E.g., the "apply" thread marks log
    //		  records as "applied".

    Log_Index get_next_flush();
    // effects - returns the value of next_flush after the log mutex has been
    // grabbed. Accessor function for next_flush

    void flush(Log_Index index);
    // requires	- index was the result of a previous append call on this log.
    // modifies	- this
    // effects	- Flush specified log record and all earlier records.
    //		  Returns when the log records have all reached the
    //		  log backing store.  On return, all of the records
    //		  with indices <= index will be stable.

    void installed(Log_Index index);
    // modifies	- this
    // effects	- Marks the specified log index as "installed".
    //		  The log is now free to write out these modifications
    //		  (probably by handing it off to the apply process).

    void applied(Log_Index start, Log_Index finished);
    // modifies	- this
    // effects	- Marks the log indices "[start..finished]" as applied.
    //		  The log is now free to delete the corresponding
    //		  log records.

    void drain();
    // modifies - this
    // effects	- Waits until the current contents of the log have
    //		  been applied to disk.

    void shutdown();
    // modifies - this
    // effects	- Waits until the current contents of the log have
    //		  been applied to disk, and then kills the OR.

    int current_size() const;
    // effects	- Return the number of bytes currently occupied by the log
    //		  contents.

    int target_size() const;
    // effects	- Return the size that the cleaner should attempt to reduce
    //		  the log to.

#if REPLICATION
    void set_allow_flushing();
    // effects : sets allow_flushing to true
    //           disables flushing of records to backups when the OR is being
    //           setup
#endif

    void shrink(int bytes);
    // effects	- Shrink log size by specified number of bytes.
    //		  This routine will typically be called either when
    //		  a logged modification is overwritten, or when the
    //		  modification is written out.

    void resize(int bytes);
    // effects	Resize log to specified size.

    void stat(OR_stat& stat);
    // effects	Store log performance info in "stat".

/* \subsection{Rep} */
  private:
    Mutex*	mutex;		// Controls access to the data structure.
    Condition*	apply_wait;	// Apply thread waits on this.
    Condition*  flush_wait;	// Flush thread waits on this.
    Condition*	flushers;	// Threads waiting for flushes block here.
    Condition*  space_wait;	// Threads waiting for space block here.
    Condition*  death_wait;	// Clients sleep here waiting for OR to die.
    Log_Entry*	list;		// Log buffer (used in a circular fashion)

    int	size;			// Number of entries in "list".
    int	logsize;		// Number of bytes currently in log
    int max_logsize;		// Maximum log size
    int	apply_start;		// Number of bytes at which apply should start
    int apply_stop;		// Number of bytes at which apply should stop

    bool        must_apply;     // Must apply some records to make space in log
    // Useful when the logsize is < apply_start but a thread T is still
    // waiting for log space. This is possible when T wants to append 
    // a record that is greater than maxlogsize - apply_start.

    bool	draining;	// Is log being drained?
    bool	dying;		// Is OR shutting down?
    Condition*	drain_wait;	// Thread executing "drain" wait on this

				    
    int		start;		// Location of first valid entry in list
    int		count;		// Number of valid entries in list
    Log_Index	next_index;	// Next index to allocate
    Log_Index	next_flush;	// Index of oldest non-stable entry

#if REPLICATION
    Mutex*      fmutex; // Used by the primary to ensure that the flush threads
			// of primary and backup are connected.
    Condition*  flush_thread_sync_wait;

    Log_Index   next_tobe_marshalled; // Index of next log record to be 
// marshalled into message to be sent to backups
    bool        immediate_flush; // is set to true when the flush is to 
// take place immediately. The idea of this is to flush when we have accumulat
// ed enough records. This is not happenning now.

    bool        allow_flushing; // turns off flushing when it is false
// Used when the OR is being setup.

    Locator locator;   // Both used to set up the connection to the backups
    Communication comm;

    OR_num local_or_num; // used to hold the OR_num of the OR. Used in
// identifying if it is a primary or backup.

#endif

    // statistics
    int         flush_size;     // size of data in bytes to be flushed
    Ubits32     flush_time;     // Time taken in flushes, in microsecs
    Ubits32     max_logrecord;  // biggest logrecord appended
/*
\subsection{Rep Invariant}

\begin{verbatim}
    size >= 0
    size >= count
    only first count entries in (list[start..size-1],list[0..start-1])
	are "non-empty".
    next_flush <= next_index
    next_flush >= low()
    logsize = sum of the sizes of all log records in log
\end{verbatim}

\subsection{Abstraction Function}

A() = first count entries in "list[start..size-1],list[0..start-1]".
*/

/*\subsection{Auxiliary Threads}*/
    void apply_thread();
    // requires	- called by apply thread
    // effects	- Loop forever applying log records.

    void flush_thread();
    // requires - called by flush thread
    // effects	- Loop forever flushing log records.

#if REPLICATION
    void backup_flush_thread();
    // requires - called by flush thread
    // effects	- Loop forever flushing log records.
#endif

/*
\subsection{Internal Operations}

All of these operations require that the caller has grabbed the log mutex.
*/
    int find(Log_Index i) const;
    // requires	- Mutex has been grabbed by the current thread.
    // effects	- Return the index in list for the record named by i.
    //		  If the specified record does not exist in the log,
    //		  return -1.

    void enlarge();
    // requires	- Mutex has been grabbed by the current thread.
    // effects	- Enlarges list by at least one.

    bool ready_to_apply();
    // requires	- Mutex has been grabbed by the current thread.
    // effects	- Return true iff a record is ready to be applied.

    bool ready_to_flush();
    // requires	- Mutex has been grabbed by the current thread.
    // effects	- Return true iff a record is ready to be flushed.

    void synchronous_apply();
    // requires - Mutex has not been grabbed by the current thread.
    // effects  - applies log if ready_to_apply.
    // Does the same work as apply_thread, but synchronously.
    // Useful for performance measurement.

#if REPLICATION
    void set_marshall_number(Log_Index i);
    // thread has not grabbed mutex. 
    // effects-  sets next_tobe_marshalled to be i
    // Used to reset the next_tobe_marshalled so that the identical set of
    // records can be sent to the second backup.
#endif

#if !REPLICATION
    void simulate_flush();
    // Sleeps for time needed to flush "flush_size" bytes.
    // Currently, uses a disk model to calculate sleep time.
    // Sets "flush_size" to zero.
#endif

};


#endif /* _LOG_H */






