// Copyright 1995 Barbara Liskov

/*
  \section{Log Record Implementation}

  This file contains the implementation of various log records.

  XXX _We should think about splitting the different log record
  implementations into separate files._
  */

#include <iostream.h>
#include <pthread.h>

#include "utils/intset.h"
#include "utils/intarray.h"
#include "utils/th_assert.h"
#include "utils/device.h"
#include "common/slot.h"
#include "common/or_obj.h"
#include "common/modset.h"
#include "common/transaction.h"

#include "or/gc/gc.h"
#include "or/gc/mod_list.h"
#include "or/gc/collector.h"
#include "or/or.h"
#include "or/or_config.h"
#include "or/thread.h"

#include "itable.h"
#include "logrecord.h"
#include "mm.h"
#include "segment.h"
#include "config/vdefs/REPLICATION.h"

/*  Abstract base class.  This does not do much of anything. */

// to just create a logrecord without any value
Log_Record::Log_Record() {}

Log_Record::Log_Record(Tid t) : tid_(t) {
}

Log_Record::~Log_Record() {
}

Tid Log_Record::tid() const {
    return tid_;
}

bool Log_Record::absorbed() const {
    return FALSE;
}

int Log_Record::size() {
    return sizeof(Log_Record);
}

#if REPLICATION
int Log_Record::type() {
  return 0;
}
#endif

bool Log_Record::encode(Device* dev) {
  return tid_.encode(dev);
    
}

bool Log_Record::decode(Device* dev) {
  bool ret = tid_.decode(dev);
  return ret;
   
}

/* Data records. */

Data_Log_Record::Data_Log_Record(Transaction const *tx) : Log_Record(tx->tid) {
    aborted = FALSE;
    mos     =  tx->mos;
    nos     =  tx->nos;
    pending      = mos->count() + nos->count();
    current_size = rep_size();

    Oref oref;
    OR_obj* obj;
    int num_slots;

    segments = new IntArray;
    IntSet segset;
    Modset::Elements mgen(mos);
    while (mgen.get(obj, oref, num_slots)) {
	int seg = Oref_segment(oref);
	if (! segset.contains(seg)) {
	    segset.insert(seg);
	    segments->append(seg);
	}
    }
    Modset::Elements ngen(nos);
    while (ngen.get(obj, oref, num_slots)) {
	int seg = Oref_segment(oref);
	if (! segset.contains(seg)) {
	    segset.insert(seg);
	    segments->append(seg);
	}
    }

}

#if REPLICATION

// for the backup to create a log record to put the information
// it gets from the net
Data_Log_Record::Data_Log_Record(int dummy) : Log_Record() {
    mos = new Modset;
    nos = new Modset;
    segments = new IntArray;
    pending = 0;
    aborted = FALSE;
    current_size = rep_size();
}

#endif

Data_Log_Record::Data_Log_Record() : Log_Record(tid()) {

    mos = new Modset;
    nos = new Modset;

    pending = 0;
    current_size = rep_size();
    segments = new IntArray;
}

Data_Log_Record::~Data_Log_Record() {
    assert(pending == 0);
    if (mos) delete mos;
    if (nos) delete nos;
    delete segments;
}

int Data_Log_Record::rep_size() const {
    return mos->rep_size() + nos->rep_size();
}

int Data_Log_Record::size() {
    return (Log_Record::size() + current_size);
}

#if REPLICATION
int Data_Log_Record::type() {
  return ( Log_Record::type() + Data_Log_Record_Type);
}
#endif

bool Data_Log_Record::encode(Device* dev) {

  //    fprintf(stderr,"%d ", current_size);
    return (Log_Record::encode(dev) && mos->encode(dev) && nos->encode(dev) );
}

bool Data_Log_Record::decode(Device* dev) {
    if (Log_Record::decode(dev) && mos->decode(dev) && nos->decode(dev)) {
	//set up the value of segments
        
	pending      = mos->count() + nos->count();
	current_size = rep_size();
	//	fprintf(stderr,"%d ", current_size);

	Oref oref;
	OR_obj* obj;
	int num_slots;

	IntSet segset; 
	
	Modset::Elements mgen(mos);
	while (mgen.get(obj, oref, num_slots)) {
	  int seg = Oref_segment(oref);
	  if (! segset.contains(seg)) {
	    segset.insert(seg);
	    segments->append(seg);

	  }
	}
    
	Modset::Elements ngen(nos);
	while (ngen.get(obj, oref, num_slots)) {
	  int seg = Oref_segment(oref);
	  if (! segset.contains(seg)) { 
	    segset.insert(seg); 
	    segments->append(seg);
	  }
	}

	return TRUE;
    }
    return FALSE;
}

void Data_Log_Record::install() {
    // If transaction committed, put new objects in itable.  Otherwise return
    MM* mm = orx->mm;
    
    if (aborted) {
	return;
    } else {
	Itable* itable = mm->itable;

	bool gc_mod_list_initially_empty = TRUE;
	mm->mutex->grab();
//	printf("Data_Log_Record::install grabbed orx->mm->mutex\n"); //DEBUG
	if (gc) {
	    gc->mod_list->mutex->grab();
//	    printf("Data_Log_Record::install grabbed gc->mod_list->mutex\n");
	    //DEBUG
	    gc_mod_list_initially_empty = (gc->mod_list->size() == 0);
	}
	// Enter modified objects
	Oref oref;
	OR_obj* obj;
	int num_slots;
	Modset::Elements mgen(mos);
	while (mgen.get(obj, oref, num_slots)) {
	    Itable_Mod* mod = itable->enter(oref, this, obj, num_slots);
	    if (gc) 
		gc->mod_list->insert(mod);
	    int segnum = Oref_segment(oref);
	    Segment* seg = (Segment*) mm->cache->lookup(segnum);
	    if (seg != 0) seg->uptodate = FALSE;
	}
	    
	// Enter new objects
	int index = 0;
	Modset::Elements ngen(nos);
	while (ngen.get(obj, oref, num_slots)) {
	    Itable_Mod* mod = itable->enter(oref, this, obj, num_slots);
	    if (gc) 
		gc->mod_list->insert(mod);
	    index++;
	    int segnum = Oref_segment(oref);
	    Segment* seg = (Segment*) mm->cache->lookup(segnum);
	    if (seg != 0) seg->uptodate = FALSE;
	}
	if (gc) {
	    bool gc_mod_list_finally_empty = (gc->mod_list->size() == 0);
	    gc->mod_list->mutex->release();
//	    printf("Data_Log_Record::install released gc->mod_list->mutex\n");
	    //DEBUG
	    if (gc_mod_list_initially_empty && !gc_mod_list_finally_empty) {
		gc->mod_list->condition->signal();
	    }
	}
	mm->mutex->release();
//	printf("Data_Log_Record::install released orx->mm->mutex\n"); //DEBUG
    }

    delete mos;
    delete nos;
    mos = 0;
    nos = 0;
}

bool Data_Log_Record::absorbed() const {
    return (pending == 0);
}

void Data_Log_Record::get_modified_segments(IntSet* set) {
    if (pending == 0) return;

    Itable* itable = orx->mm->itable;
    int num = segments->size();

    for (int i = 0; i < num; i++) {
	// Double-check with itable before putting segment
	int seg = segments->slot(i);
	if (!set->contains(seg) && itable->segment_needs(seg, this))
	    set->insert(seg);
    }
}

void Data_Log_Record::cancel(Oref, OR_obj*, int num_slots) {
    // Just ignore which object it is and decrement the value of the
    // pending modification counter.  This implies that
    // "get_modified_segments" will generate some extra segments in
    // many cases, but we can always double-check with the itable to
    // see if there are any pending modifications to these segments.

    pending--;
    th_assert(pending >= 0, "too many cancellations of log record contents");

    current_size -= (num_slots + OR_obj_headers) * Slot_size;
    assert(current_size >= 0);
}

void Data_Log_Record::commit() {}

void Data_Log_Record::abort() {
    aborted = TRUE;
    pending = 0;
}

/* Commit record -- for single-OR transactions only. */

Commit_Log_Record::Commit_Log_Record() : Log_Record() {}

Commit_Log_Record::Commit_Log_Record(Tid tid, Global_tstamp stable_threshold)
     : Log_Record(tid), threshold(stable_threshold) {}

Commit_Log_Record::~Commit_Log_Record() {}

int Commit_Log_Record::size() {
    return Log_Record::size() + sizeof(threshold);
}

#if REPLICATION
int Commit_Log_Record::type() {
    return ( Log_Record::type() + Commit_Log_Record_Type );
}
#endif

bool Commit_Log_Record::encode(Device* dev) {
    return (Log_Record::encode(dev) && threshold.encode(dev));
}

bool Commit_Log_Record::decode(Device* dev) {
    return (Log_Record::decode(dev) && threshold.decode(dev));
}

void Commit_Log_Record::install() {
}

void Commit_Log_Record::get_modified_segments(IntSet*) {}

/* Abort record. */

Abort_Log_Record::Abort_Log_Record(Tid tid) : Log_Record(tid) {
}

Abort_Log_Record::Abort_Log_Record() : Log_Record() {}

Abort_Log_Record::~Abort_Log_Record() {}

int Abort_Log_Record::size() {
    return Log_Record::size();
}

#if REPLICATION
int Abort_Log_Record::type() {
    return ( Log_Record::type() + Abort_Log_Record_Type) ;
}
#endif

bool Abort_Log_Record::encode(Device* dev) {
    return Log_Record::encode(dev);
}

bool Abort_Log_Record::decode(Device* dev) {
    return Log_Record::decode(dev);
}

void Abort_Log_Record::install() {}

void Abort_Log_Record::get_modified_segments(IntSet*) {
    // Nothing to do
}

/* Prepared record --- logged at participants. */

Prepared_Log_Record::Prepared_Log_Record(Transaction const* tx, OR_num coord,
		     Global_tstamp tstamp) : Data_Log_Record(tx) {
    threshold = tstamp;
    coordinator = coord;
}

Prepared_Log_Record::Prepared_Log_Record() : Data_Log_Record(0) {}

Prepared_Log_Record::~Prepared_Log_Record() {}

int Prepared_Log_Record::size() {
    return (Data_Log_Record::size() +
	    sizeof(threshold) +
	    sizeof(coordinator));
}

#if REPLICATION
int Prepared_Log_Record::type() {
    return ( Log_Record::type() + Prepared_Log_Record_Type) ; 

}
#endif

bool Prepared_Log_Record::encode(Device* dev) {
    return (Data_Log_Record::encode(dev) &&
	    threshold.encode(dev) &&
	    dev->send_ubits32(coordinator, TRUE));
}

bool Prepared_Log_Record::decode(Device* dev) {
    return (Data_Log_Record::decode(dev) &&
	    threshold.decode(dev) &&
	    dev->recv_ubits32(&coordinator));
}

/* Committing record --- logged at coordinator. */

Committing_Log_Record::Committing_Log_Record() : Data_Log_Record(0) {}

Committing_Log_Record::Committing_Log_Record(Transaction const* tx,
	     Global_tstamp tstamp, OR_set *parts) : Data_Log_Record(tx) {
    threshold = tstamp;
    participants = *parts;  // Make copy of set of participants
}

Committing_Log_Record::~Committing_Log_Record() {}

int Committing_Log_Record::size() {
    return (Data_Log_Record::size() +
	    sizeof(threshold) +
	    participants.encode_size());
}

#if REPLICATION
int Committing_Log_Record::type() {
    return ( Log_Record::type() + Committing_Log_Record_Type );

}
#endif

bool Committing_Log_Record::encode(Device* dev) {
    return (Data_Log_Record::encode(dev) &&
	    threshold.encode(dev) &&
	    participants.encode(dev));
}

bool Committing_Log_Record::decode(Device* dev) {
    return (Data_Log_Record::decode(dev) &&
	    threshold.decode(dev) &&
	    participants.decode(dev));
}

/* Committed record --- logged at coordinator and participants. */


Committed_Log_Record::Committed_Log_Record() :
  Log_Record() {}


Committed_Log_Record::Committed_Log_Record(Tid tid, OR_set *parts) :
    Log_Record(tid) {
    participants = *parts;  // Make copy of set of participants
}

Committed_Log_Record::~Committed_Log_Record() {}

int Committed_Log_Record::size() {
    return (Log_Record::size() +
	    participants.encode_size());
}

#if REPLICATION
int Committed_Log_Record::type() {
    return ( Log_Record::type() + Committed_Log_Record_Type);
}
#endif

bool Committed_Log_Record::encode(Device* dev) {
    return (Log_Record::encode(dev) &&
	    participants.encode(dev));
}

bool Committed_Log_Record::decode(Device* dev) {
    return (Log_Record::decode(dev) &&
	    participants.decode(dev));
}

void Committed_Log_Record::install() {}
void Committed_Log_Record::get_modified_segments(IntSet*) {}

/* Done record. */

Done_Log_Record::Done_Log_Record() : Log_Record() {}

Done_Log_Record::Done_Log_Record(Tid tid) : Log_Record(tid) {}

Done_Log_Record::~Done_Log_Record() {}

int Done_Log_Record::size() {
    return Log_Record::size();
}

#if REPLICATION
int Done_Log_Record::type() {
    return ( Log_Record::type() + Done_Log_Record_Type );
}
#endif

bool Done_Log_Record::encode(Device* dev) {
    return Log_Record::encode(dev);
}

bool Done_Log_Record::decode(Device* dev) {
    return Log_Record::decode(dev);
}

void Done_Log_Record::install() {}

void Done_Log_Record::get_modified_segments(IntSet*) {}

/* Participant record --- holds participant's prepare record in
   coordinator log protocol */

Participant_Log_Record::Participant_Log_Record() : Log_Record() {}


Participant_Log_Record::Participant_Log_Record(Tid tid,
		     OR_num participant, Prepared_Log_Record *record,
		     Log_Index lindex) : Log_Record(tid) {
    index = lindex;
    pr = record;
    or_num = participant;
}

Participant_Log_Record::~Participant_Log_Record() {
    delete pr;
}

int Participant_Log_Record::size() {
    return (Log_Record::size() +
	    sizeof(Ubits32) +
	    sizeof(Ubits32) +
	    pr->size());
}

#if REPLICATION
int Participant_Log_Record::type() {
    return ( Log_Record::type() + Participant_Log_Record_Type) ;

}
#endif

bool Participant_Log_Record::encode(Device* dev) {
    return (Log_Record::encode(dev) &&
	    dev->send_ubits32(or_num, TRUE) &&
	    dev->send_ubits32(index, TRUE) &&
	    pr->encode(dev));
}

bool Participant_Log_Record::decode(Device* dev) {
    pr = new Prepared_Log_Record;

    return (Log_Record::decode(dev) &&
	    dev->recv_ubits32(&or_num) &&
	    dev->recv_ubits32(&index) &&
	    pr->decode(dev));
}

void Participant_Log_Record::install() {}
void Participant_Log_Record::get_modified_segments(IntSet*) {}

/* Stamos record --- used in coordinator log protocol */

Stamos_Record::Stamos_Record() : Log_Record(){}

Stamos_Record::Stamos_Record(OR_num coord, bool add) : Log_Record(tid()){
    or_num = coord;
    action = add;
}

Stamos_Record::~Stamos_Record() {
}

int Stamos_Record::size() {
    return Log_Record::size() + sizeof(Ubits32) + sizeof(Ubits32);
}

#if REPLICATION
int Stamos_Record::type() {
    return ( Log_Record::type() + Stamos_Record_Type) ;
}
#endif

bool Stamos_Record::encode(Device* dev) {
    return (Log_Record::encode(dev) &&
	    dev->send_ubits32(or_num, TRUE) &&
	    dev->send_ubits32(action, TRUE));
}
bool Stamos_Record::decode(Device* dev) {
    return (Log_Record::decode(dev) &&
	    dev->recv_ubits32(&or_num) &&
	    dev->recv_ubits32(&action));
}

void Stamos_Record::install() {}
void Stamos_Record::get_modified_segments(IntSet*) {}


