// Copyright 1995 Barbara Liskov

/*
\section{Memory Manager Interface}

The "MM" provides access to persistent objects (see "or_obj.h").
The interface is documented in Mercury Design Note 64.
*/

#ifndef _MM_H
#define _MM_H

#include "utils/intset.h"
#include "utils/basic.h"
#include "common/oref.h"
#include "common/tid.h"
#include "log.h"
#include "disk.h"
#include "or/gc/gc.h"

class Cache;			// Segment cache
class Disk_SB;			// Disk super block
class Disk_Segment;		// Disk format for a segment
class IntArray;			// Array of integers (used for segment lists)
class Itable;			// Table of pending object installations
class MM_Handle;		// Handle for pinned "OR_obj"
class MM_HandleList;		// List of handles
class MM_Stats;
class Mos;			// Modified object set
class Mutex;			// For controlling access to MM
class OR_obj;			// Type of an OR object
class Segment;			// A segment
class Segment_Table;		// Segment table
class CacheEntry;		// Type of cache entries
class Pref_set;			// Set of segment bitmaps 
class Prefetch_Info;
class Collector;

struct OR_stat;			// Performance info
struct prefetch_hint;		// Prefetching hint info

class MM {
  public:
    MM(bool init);
    // effects	Create a memory manager for OR.
    //		If "init" is true, then initialize the disk contents.
    //		Else, recover the disk contents.

/* \subsection{Read Operations} */

    MM_Handle* fetch(Oref o);
    // effects - Obtain read lock on object named by o, return a
    //		 handle to the object.  Return 0 if not found.
    //
    //		 The caller should release the handle to release the
    //		 read lock.

    MM_Handle* fast_fetch(Oref o);
    // effects	- Obtain read lock on object named by o and return a
    //		  handle to the object.  Return 0 if not found or if
    //		  fetching the object would require some slow operation
    //		  (a disk read for example).
    //
    //		 The caller should release the handle to release the
    //		 read lock.

    Segment *fetch_segment(int segnum, bool fast, bool grab_mutex=TRUE);
    // requires calller holds this->mutex iff grab_mutex == FALSE
    // modifies	seg, itable
    // effects	Fetches identified segment.
    //          If "fast" and segment is not in cache, returns 0.
    //          Applies any modifications to the segment in itable:
    //          does not remove the mods from the itable and does not write
    //          segment to the disk, leaving it dirty.
    //          Pins the segment.
    //          Returns 0 if segment is not found.
    //          Iff "grab_mutex", attempts to grab this->mutex.

    
    void mods_prefetch(int segnum, MM_HandleList* list);
    // Modifies list
    // effects Appends modified objects in the identified segment to "list".

    void subpage_prefetch(Prefetch_Info* pinfo, Pref_set* set,
		  struct prefetch_hint const* hint); 
    // modifies	"pinfo", "set".
    // effects	Prefetches objects using the subpage policy using "hint"
    //		Any objects initially listed in "set" are not prefetched.
    //		All prefetched objects are appended to "pinfo" and
    //		inserted into "set".

    void pointer_prefetch(MM_HandleList* list, Pref_set* set,
			 struct prefetch_hint const* hint); 
    // modifies	"list", "set".
    // effects	Prefetches objects reachable from list using "hint"
    //		Any objects initially listed in "set" are not prefetched.
    //		All prefetched objects are appended to "list" and
    //		inserted into "set".

    void release(MM_Handle*);
    // effects	Release the read lock obtained on behalf of the specified
    //		handle.

    void release_list(MM_HandleList* list);
    // effects	Release the read locks obtained on behalf of all
    //		handles in "list".

    Oref directory();
    // effects  - Return oref for directory.

/* \subsection{Modification Operations} */
    void set_gc(bool status=TRUE);

    int alloc_new_segment();
    // effects: Creates a new segment in the database and returns the id of
    //          that segment. Returns -1 if cannot allocate space

    void clean_log();
    // effects	Perform IO to move some of the log contents to disk.
    //		Also removes log entries.

    int free_space();
    // requires  Caller holds "mm->mutex".
    // effects	 Return the number of bytes of free space on disk.
    //		 This free-space might be fragmented and therefore
    //		 compaction might be required to access all of
    //		 this space.

    void modify_segment(Segment *seg, bool remove_mods);
    // requires  Caller holds "mm->mutex".
    // effects  applies pending modifications to segment.
    //          without writing out to disk (leaves seg dirty).
    //          If "remove_mods", removes modifications from itable.
    
/*
\subsection{Low-level Object Mutation}

These routines are designed to be used only during database initialization
to create new objects.
*/
    Oref reserve(int num_slots);
    // requires: This method is only called at database initialization
    //           num_slots is less than the maximum size of object on a page
    // effects:  Reserves space for an object with size "num_slots" and returns
    //           the oref of the object. num_slots excludes the header slots
    

    Oref make_object(OR_obj* obj, int num_slots, Oref class_oref);
    // modifies	this
    // effects	Create a new object with a copy of the contents of "obj".
    //		and returns its oref.  The new object state is logged,
    //		but has not necessarily reached the disk by the time
    //		this routine returns. num_slots denotes the number of slots
    //          that are occupied by obj (Excluding the headers)
    //          class_oref is the oref of the class object.

/*
\subsection{Time Stamp Generation}

The following routine can be used to generate monotonically increasing
stamps for various MM related activities.
*/
    long new_stamp();
    // requires	mutex is held by the caller.
    // modifies	this
    // effects	Returns a new stamp that is greater than all previous
    //		returned stamps.  The stamp information is not maintained
    //		across crashes, and therefore after a crash this routine
    //		may return a smaller stamp than a stamp returned before
    //		the crash.

// \subsection{Performance Measurements}
    void stat(OR_stat& stat);
    // effects	Store MM performance info in "stat".

    void resize_cache(int bytes);
    // effects	Resize the cache to the specified size.

    int cache_size();
    // returns cache size in bytes.

    void resize_log(int bytes);
    // effects	Resize the log to the specified size.

    void resize_dspace(int percent_use);
    // effects	Set parameters so that disk space utilization is
    //		at the specified percentage level.  This call is
    //		used pretty much just to experiment with cleaning
    //		in the log structured disk layout policy.
/*
\subsection{Common Pieces of State}

These are exported to various sub-modules.
*/
    Mutex*	mutex;		// Controls access to MM data structures
    Cache*	cache;		// Segment cache
    Itable*	itable;		// Pending installation table
    Disk*	disk;		// The underlying disk
    Disk_SB*	super;		// Cached copy of disk super block
    Segment_Table* segtable;	// Segment table

    int         segment_request_id; // to identify last segment request to FEs
    int         segment_requested;  // To identify last segmented requested

    MM_Stats*   stats; // statistics

    CacheEntry* alloc_entry(int id, long address, long size);
    // requires	Caller holds "mm->mutex"
    // effects	Allocate a cache entry with the specified id/address/size.
    //		This routine is used by the cache module to isolate it
    //		from the details of whether a segment or a fragment is
    //		being allocated.  The address and size are always given
    //		in bytes.

    void collect_segments(IntSet& set, Log_Index& low, Log_Index& high);
    // modifies	"list", "low", "high".
    // effects	Collect set of segments that need to be written out to disk.
    //		Store this set of segments in "set".  Store in "low" and
    //		"high" the range of log records that will be considered
    //		applied to disk when this list of segments has been written
    //		out.

  private:
    friend Itable;

    int      new_arena;		// New and moved objects are put here
    long     stamp_counter;	// For increasing stamp generation
    int	     big_threshold;	// Objects with this many bytes or more get
				// their own segments.
    bool     gc_on;             // FALSE until gc gets turned on.

  public:
    // Performance information.
    // Public because other modules might fill these in...
    long  mods;			// # of committed object modifications
    long  absorbed;		// # of absorbed modifications
    long  installs;		// # of installs
    long  fetches;		// # of cache fetches
    long  misses;		// # of cache misses
    long  frag_writes;		// # of fragment writes
    long  frag_sizes;		// Combined size of written out fragments
    long  clean_count;		// # of cleaned regions
    long  clean_live;		// Amount of live data in cleaned regions
    Timer install_time;         // Time for MOB installs
    
  private:

    MM_Handle* retry;
    // Junky handle returned to indicate that fetch operation should
    // be restarted.

    MM_Handle* do_fetch(Oref oref, bool fast);
    // effects	Perform all of the work of either "fetch" or "fast_fetch"
    //		depending on the value of "fast".

    MM_Handle* read_object(Oref oref, bool fast);
    // effects	Read object named by "oref" from either the cache,
    //		or the disk.  Returns handle normally.
    //		Returns 0 if the object cannot be found,
    //		or if "fast" is TRUE and reading the object would
    //		require some slow operation.  Returns "retry" if
    //		the operation should be retried.

    Segment* find_segment(int segnum, bool fast = FALSE);
    // effects	Wait until specified segment is in memory.

    void install_object(Oref oref, OR_obj* obj, int num_slots);
    // modifies	this
    // effects	Installs "obj" as new value for "oref".
    //          Num_slots is the number of field slots
    //		May release "mutex" if necessary to wait for
    //		the appropriate segment to be read into memory.

    void forward_object(Oref oref, OR_obj* obj);
    // requires	Object does not fit in its current segment
    // effects	Attempt to move it and install a forwarder in the
    //		original location.
    //		May release "mutex" if necessary to wait for
    //		the appropriate segment(s) to be read into memory.

    void sort_segments(IntArray& list);
    // modifies	"list".
    // effects	Sort the set of segments to provide better disk scheduling.

    void write_modifications(IntSet const& list);
    // requires	All of the entries in "list" are segment numbers.
    // effects	Writes out all pending modifications to these segments.

    void init();			// Initialize disk image
    void recover();			// Recover from disk image
    Oref make_root();			// Make root structure for new OR

    void create_seg_table();		// Initialize segment table
    void recover_seg_table();		// Read segment table

    friend class Back_Recv_Message;
    friend Collector;
};

#endif /* _MM_H */
