// Copyright 1995 Barbara Liskov

#include <stdio.h>
#include "utils/basic.h"
#include "utils/fail.h"
#include "utils/th_assert.h"
#include "utils/openhashmap.h"
#include "utils/hashfuncs.h"
#include "utils/link.h"

#include "or/or.h"
#include "or/or_config.h"
#include "or/thread.h"
#include "mm.h"
#include "scache.h"

// The declarations for the hash table

#include "utils/map.h"
#include "utils/valuekey.h"

// Using Intkey for the segment ids
class Cache_Table : public Map<IntKey, CacheEntry*> {
  public:
    Cache_Table(int sizehint = 0) : Map<IntKey, CacheEntry*>(sizehint) {}
};

#define IMPL_KEY IntKey
#define IMPL_VALUE CacheEntry *
#include "utils/impl_map.t"

// Hash table declarations end here


Cache::Cache(int size_in_bytes) {
    int segsize_in_bytes = 32*1024; // XXX Needs to be fixed
    map		= new Cache_Table(size_in_bytes/segsize_in_bytes);
    lru		= new Link;
    size	= 0;
    max_size	= size_in_bytes;
    space	= new Condition(orx->mm->mutex);
}

CacheEntry* Cache::lookup(int id) {
    CacheEntry* entry;
    if (map->find(id, entry))
	return entry;
    else
	return 0;
}

void Cache::invalidate(int id) {
    // No need to "evict" the cache entry.  Just remove it from the
    // map so nobody looks at it, and move it in the LRU chain so that
    // it will be available for eviction soon.

    CacheEntry* entry;
    if (map->find(id, entry)) {
	Link* slink = entry->lru_link();
	slink->remove();
	lru->append(slink);
	map->remove_fast(id);
    }
}

void Cache::used(CacheEntry* entry) {
    Link* slink = entry->lru_link();
    slink->remove();
    lru->prepend(slink);
}

bool Cache::alloc(int id, long address, long entry_size) {
    th_assert(!map->contains(id), "allocating duplicate cache slot");

    if (! create_space(entry_size)) {
	// Not enough room in the cache: wait for a while
	space->wait();
	return FALSE;
    }

    // Allocate a new slot
    CacheEntry* entry = orx->mm->alloc_entry(id, address, entry_size);

    // Enter into map and lru list
    map->store(id, entry);
    lru->append(entry->lru_link());
    size += entry_size;

    return TRUE;
}

void Cache::enter(int id, CacheEntry* entry) {
    th_assert(!map->contains(id), "allocating duplicate cache slot");

    while (! fast_enter(id, entry)) {
	// Wait a little while for cache space.
	space->wait();
    }
}

bool Cache::fast_enter(int id, CacheEntry* entry) {
    th_assert(!map->contains(id), "allocating duplicate cache slot");

    int entry_size = entry->size();
    if (! create_space(entry_size)) {
	// Not enough room in the cache
	return FALSE;
    }

    // Enter into map and lru list
    map->add(id, entry);
    lru->append(entry->lru_link());
    size += entry_size;
    return TRUE;
}

bool Cache::create_space(int entry_size) {
    Link* l = lru->next();
    while (((size + entry_size) > max_size) && (l != lru)) {
	CacheEntry* entry = ((CacheEntry*) l->value);
	l = l->next();

	if (entry->evictable())
	    evict(entry);
    }

    return ((size + entry_size) <= max_size);
}

void Cache::evict(CacheEntry* entry) {
    th_assert(entry->evictable(), "evicting wrong slot");

    map->remove_fast(entry->id());
    size -= entry->size();
    entry->lru_link()->remove();
    delete entry;

    space->broadcast();
}

void Cache::make_space() {
    space->broadcast();
}

void Cache::resize(int bytes) {
    max_size = bytes;

    // Try to evict stuff that does not fit any more
    create_space(0);
}

CacheEntry::CacheEntry(int id) {
    cond	= new Condition(orx->mm->mutex);
    state	= Absent;
    waiting	= 0;
    pins	= 0;
    cid		= id;
    lru.value	= this;
}

CacheEntry::~CacheEntry() {
    th_assert(evictable(), "discarding a non-evictable entry");
    lru.remove();
    delete cond;
}

bool CacheEntry::missing() const {
    return ((state == Absent) || (state == Reading));
}

void CacheEntry::initialized() {
    th_assert(state == Absent, "initializing an existing entry");
    state = Clean;
}

void CacheEntry::mark_dirty() {
    state = Dirty;
}

void CacheEntry::mark_clean() {
    state = Clean;
}

void CacheEntry::write_contents() {
    fail("CacheEntry::write_contents called");
}

bool CacheEntry::evictable() const {
    return (((state == Clean) || (state == Absent)) &&
	    (pins == 0) && (waiting == 0));
}

void CacheEntry::fetch(bool iread) {
    orx->mm->cache->used(this);

    if (state == Absent) {
	// Need to read the entry

	// Fetch from disk
	state = Reading;
	read_contents(iread);

	// Finish the read: tell other threads the entry is available,
	state = Clean;
	release();
    }

    while (state == Reading) {
	// Somebody else is reading.  Wait for it.
	waiting++;
	cond->wait();
	waiting--;
    }
}

void CacheEntry::write_lock() {
    fetch(TRUE);

    while (pins > 0) {
	waiting++;
	cond->wait();
	waiting--;
    }
}

void CacheEntry::write() {
    if (! is_dirty())
	return;

    add_pin();
    write_contents();
    state = Clean;
    remove_pin();
}

void CacheEntry::add_pin() {
    pins++;
}

void CacheEntry::remove_pin() {
    th_assert(!missing(), "unpin called on missing entry");
    th_assert(pins > 0, "extra unpin");

    pins--;
    if (pins == 0) release();
}

void CacheEntry::release() {
    if (waiting > 0) {
	// Wake up any waiters for this entry
	cond->broadcast();
    }
    else if (evictable()) {
	// Entry is now evictable.  Wake up any thread waiting for space.
	orx->mm->cache->make_space();
    }
}

// Routines for debugging

void CacheEntry::dump() {
    char const* st;
    switch (state) {
      case Absent:	st = "absent";	break;
      case Reading:	st = "reading";	break;
      case Clean:	st = "clean";	break;
      case Dirty:	st = "dirty";	break;
      default:		st = "unknown";	break;
    }

    fprintf(stderr, "  %4d: %-9s, w = %d, p = %d\n", cid, st, waiting, pins);
}

void Cache::dump() {
    Link* l = lru->next();
    while (l != lru) {
	CacheEntry* entry = ((CacheEntry*) l->value);
	entry->dump();
	l = l->next();
    }
}

// Wrapper to allow calls by stupid debuggers
void dump_cache() {
    orx->mm->cache->dump();
}
