// Copyright 1995 Barbara Liskov
/*
\section{Segment Implementation}

The representation of a segment consists of an array of "OR_slot"s.
The format of this array is given in "dformat.h".

\subsection{Rep Invariant}
XXX Fill this in

\subsection{Abstraction Function}
XXX Fill this in

\subsection{Todo}

\begin{itemize}
\item	Think about fairness for writers.  Currently, they can be
	continuously overtaken by readers.
\item	Interface to support multiple simulataneous ireads/writes.
\item	\ldots
\end{itemize}

*/

/*
\subsection{Assertion Checking}

The following code performs an assertion check and dumps out the
segment contents and aborts if the check fails.
*/
#define CHECK(cond) if (cond) {;} else { dump(); assert(cond); }

#include <assert.h>
#include <stdio.h>
#include <iostream.h>
#include <string.h>

#include "utils/th_assert.h"
#include "utils/array.h"
#include "utils/fail.h"
#include "common/or_obj.h"
#include "common/oref.h"
#include "or/or.h"
#include "or/or_config.h"
#include "or/thread.h"
#include "or/gc/gc.h"
#include "or/gc/class_map.h"

#include "dformat.h"
#include "disk.h"
#include "mm.h"
#include "mm_stats.h"
#include "scache.h"
#include "segment.h"

/*
Header entries for existing object give the index in the segment at
which the corresponding object starts.  However, if the header entry
is "FREE_OBJECT", then no such object exists.
*/
// #define FREE_OBJECT	0

// \subsection{Segment Creation}

Segment::Segment(int id, Disk_Range r) : CacheEntry(id) {
    th_assert(r.count >= 1, "segment must have at least one block");
    th_assert((r.count << DISK_UNIT_SHIFT) <= DISK_SEG_MAX_SIZE,
	      "segment is too big");

    range	= r;
    contents	= 0;
    uptodate    = FALSE;
    write_stamp	= orx->mm->new_stamp();
}

Segment::~Segment() {
    if (contents != 0) delete [] contents;
}

// \subsection{Segment Writer}
//
// The following special purpose thread is forked off whenever a
// new segment is created.  The job of this thread is to write out
// the new segment contents right away.  The segment contents have
// to be written out to prevent the cache from locking up with
// dirty new segments.  This has to be done in a separate thread to
// avoid releasing "mm->mutex" in the middle of segment initialization..
//
// XXX This is probably inefficient because the new segment will be
// written out again soon when it has been filled up with new objects.

class New_Segment_Writer : public Thread {
  public:
    New_Segment_Writer(Segment* seg);
    virtual void main();
  private:
    Segment* segment;
};

New_Segment_Writer::New_Segment_Writer(Segment* s) {
    segment = s;
}

void New_Segment_Writer::main() {
    orx->mm->mutex->grab(); {
	segment->write();
    } orx->mm->mutex->release();    
}

void Segment::init() {
    int num_pages       = (range.count << DISK_UNIT_SHIFT) / Page_size;

    contents          = (Disk_Segment*) new Page[num_pages];
    //    contents->h.magic = DISK_SEG_MAGIC;
    //    contents->h.id    = id();
    //    contents->h.type  = DISK_SEG_NORMAL;
    //    contents->h.num_pages = num_pages;
    for (int i = 0; i < num_pages; i++) 
	contents->pages[i].set_or(orx->config->ornum);

    write_stamp		= orx->mm->new_stamp();

    initialized();
    mark_dirty();
    init_reservation_info();

    Thread* writer = new New_Segment_Writer(this);
    writer->start();
}

// #include "or/fe_manager.h"
// #include "or/or_config.h"

// void Segment::coop_fetch() {
//     CacheDir* cd = orx->mm->cachedir;
//     MM_Stats* stats = orx->mm->stats;
//     FeSet *tried_fes = new FeSet(); // FEs already approached
//     fe_num helper;

//     stats->iread++;
//     if (!missing()) stats->iread_cache++;

//     while (missing()) {
// 	orx->mm->segment_request_id++; // each fetch attempt increments id
// 	// If a client has the complete segment, forward the request.
// 	if (orx->config->iread_complete &&
// 	    cd->search(id(), CacheDir::complete, tried_fes, helper)) {
// 	    stats->iread_forward++;
// 	    fetch_forward(helper, tried_fes, FALSE); // don\'t send mods
// 	} 
// 	// If a client has a reparable segment, forward the request with mods
//         else if (orx->config->iread_reparable &&
// 		 cd->search(id(), CacheDir::reparable, tried_fes, helper)) {
// 	    stats->iread_forward_mods++;
// 	    bool send_mods = orx->config->iread_mods;
// 	    fetch_forward(helper, tried_fes, send_mods); 
// 	    if (!send_mods && !missing()) { // fetched segment needs repair
// 		orx->mm->fetch_segment(id(), FALSE); // does repair
// 	    }
// 	    // XXX above has a non-osdi-crashing transactional bug since
// 	    // mm->mutex is released between segment arrival and repair
// 	} else { // read from disk
// 	    stats->iread_disk++;
// 	    stats->iread_disk_time.start();
// 	    fetch(TRUE); 
// 	    stats->iread_disk_time.stop();
// 	}
//     }
//     delete tried_fes;
// }

// #include "or/message_stats.h"

// void Segment::fetch_forward(fe_num helper, FeSet *tried_fes, bool send_mods) {
//     waiting++; // Lock segment in cache.

//     orx->mm->segment_requested = id();
//     fe_num zero; // zero fe_num signifies that requester is OR
//     fe_message msg;

//     orx->mm->stats->iread_forward_time.start();

//     orx->mm->mutex->release();
//     forward_fetch(id(), helper, zero, orx->mm->segment_request_id, send_mods);
//     FE_manager *helper_fe = NULL;
//     orx->fe_manager_set->lock->read_lock(); {
//       if (! orx->fe_manager_set->fetch(helper, helper_fe)) {
// 	helper_fe = NULL;
// 	cerr << "helper disappeared" << endl;
//       }
//     } orx->fe_manager_set->lock->read_unlock();

//     // Wait for reply in special network. 
//     if (!helper_fe->iread_net->wait_or_timeout(50000)) 
//       th_fail("Timeout while waiting for iread reply\n");

//     if (!msg.decode(helper_fe->iread_net)) 
//       th_fail("Error reading reply to iread avoidance\n");
 
//     orx->mm->mutex->grab();
//     switch (msg.msgtype) {
//     case FE_SEGMENT: 
//       // a segment arrived from the FE
//       msg_stats.fe_segment++;
//       int bytes = helper_fe->iread_net->bytes_received;
//       segment_arrived(msg, helper_fe->iread_net);
//       msg_stats.fe_segment_bytes += helper_fe->iread_net->bytes_received-bytes;
//       break;
//     case FE_FETCH_DENIED: {
//       msg_stats.fe_fetch_denied++;	
//       fe_num requester(msg.u.fetch_denied.requester);
//       int req_id = msg.u.fetch_denied.request_id;
//       th_assert(requester.is_zero(), "Reply in wrong channel\n");	 
//       break;
//     } 
//     default:
//       /* XXX - Log error */
//       th_fail("FE sent invalid message in reply to iread avoidance\n");
//       break;
//     };
    

//     orx->mm->stats->iread_forward_time.stop();
//     tried_fes->insert(helper);
//     orx->mm->segment_requested = 0;
//     waiting--;

//     if (missing()) {
//       orx->mm->stats->iread_forward_denied++;
//       return;
//     }
    
//     orx->mm->cachedir->enter(id(), helper, CacheDir::complete);    
  
// }


// void Segment::segment_arrived(fe_message &msg, Network *net) {
//   th_assert (msg.u.segment.fullSegment, "Non-full seg from Fe");
//   if (orx->mm->segment_request_id != msg.u.segment.request_id) 
//     th_fail("Old request id in iread reply \n");

//   int segnum = msg.u.segment.u.segnum;
//   int size = OR_SEGMENT_SIZE/sizeof(OR_slot);
//   OR_slot *buf = new OR_slot[size];
//   if (!net->recv_buffer(buf, OR_SEGMENT_SIZE)) {
//       th_fail("Failed to read segment from message\n");
//   }
//   set_contents(buf);
// }


// void Segment::set_contents(OR_slot *dseg) {
//     if (contents)
//         delete contents; // if any exists
//     contents = (Disk_Segment *) dseg;
//     state = Clean;
//     write_stamp = orx->mm->new_stamp();
//     init_reservation_info();
//     release(); // broadcast waiting threads
//}  


void Segment::read_contents(bool iread) {
    Disk_OpType t = iread ? Disk_IRead : Disk_FRead;
    int pages = (range.count << DISK_UNIT_SHIFT) / Page_size;
    contents = (Disk_Segment*) new Page[pages];
    bool result = orx->mm->disk->read(contents, range, t);
    if (result == 0) th_fail("Segment fetch failed");
    write_stamp = orx->mm->new_stamp();
    init_reservation_info();
    uptodate = FALSE;
}

void Segment::write_contents() {
    write_stamp = orx->mm->new_stamp();

    bool result = orx->mm->disk->write(contents, range, Disk_FWrite);
    if (result == 0) th_fail("Segment write failed");
}

void Segment::init_reservation_info() {
    // XXX This method will be removed 
}

Disk_Range Segment::disk_range() const {
    return range;
}

int Segment::size() const {
    return (range.count << DISK_UNIT_SHIFT);
}

bool Segment::modified(long stamp) const {
    return (is_dirty() && (write_stamp <= stamp));
}

void Segment::pin_segment() {
    orx->mm->cache->used(this);
    add_pin();
}

OR_obj* Segment::pin(int index) {
    fetch();

    OR_obj* result = find(index);
    if (result != 0) add_pin();
    return result;
}

void Segment::unpin() {
    remove_pin();
}

bool Segment::install(int index, OR_obj* object, int num_slots) {
    write_lock();

    // We could write a debugging routine here that verifies that the size of
    // the object has not changed XXX

    th_assert(index >= 0, "Bad index passed to segment installation");
    OR_obj* old = find(index);
    Uint page_num, onum;
    if (!old) {
      // The object did not exist before. It needs to be added to this segment
      Pagenum_onum_from_index(index, page_num, onum);
      th_assert(page_num < num_pages(), "Page number too high");      
      old = (OR_obj *) contents->pages[page_num].allocate(onum, num_slots);
    }
    if (!old)
      return FALSE; // object doesn't fit in this page!
    memcpy(old, object, (num_slots + OR_obj_headers) * Slot_size);
    return TRUE;
}

/*
\subsection{Internal Operations}

When these operations are called, the caller should already have
set-up things to prevent conflicting concurrent access to the segment.
These operations do not do any concurrency control.
*/

OR_obj* Segment::find(int index) {
    Uint page_num, onum;

    Pagenum_onum_from_index(index, page_num, onum);
    if (index < 0 || page_num >= num_pages()) return NULL;
    Page *page = &contents->pages[page_num];
    OR_obj *result = (OR_obj *) page->lookup(onum);
    return result;
}

void Segment::print(FILE *fp) {
    char const* f1 = "%-15s = %12d\n";
    char const* f2 = "%-15s = <%d,%d>\n";

    if (!fp) fp = stderr;
    fprintf(fp, f1, "Segment",	id());
    fprintf(fp, f2, "range",	range.address, range.count);
    fprintf(fp, "%-15s = %12lu\n", "Mods wstamp", write_stamp);

    //    if (contents != 0) {
    //	fprintf(fp, f1, "Pages",	contents->h.num_pages);
    //	fprintf(fp, f1, "Type",	        contents->h.type);
    //	fprintf(fp, f1, "Id",	        contents->h.id);
    //    }
}
