// Copyright 1995 Barbara Liskov

/*
\section{Segment Interface}

A segment maps from an object index within a segment to
the corresponding object.
*/

#ifndef _SEGMENT_H
#define _SEGMENT_H

#include <stdio.h>
#include "utils/basic.h"
#include "common/oref.h"

#include "disk.h"
#include "dformat.h"
#include "scache.h"


union Page;
class OR_obj;
class Mutex;
class Condition;
class Network;

class Segment : public CacheEntry {
  public:
// \subsection{Initialization}
    Segment(int segid, Disk_Range range);
    // requires	"range.count >= 1"
    // effects	Creates a segment for mapping objects within the
    //		segment specified by segid.  The segment is
    //		allowed to occupy the given "range".

    virtual ~Segment();
    // requires	"evictable()"
    // effects	Discard segment from memory.

    void init();
    // requires	"missing()"
    // effects	Initializes segment to contain no objects.
    //		After this call, the segment is no longer "missing".

    // \subsection{Segment State}
    Disk_Range disk_range() const;
    // effects	Return disk range occupied by segment

    Uint num_pages() const { 
	return Max_pages_in_seg;
    }
    // effects	Return the number of data pages in segment

    Page* lookup_page(Uint pagenum);
    // requires: pageid < num_pages().
    // effects: Returns the "pagenum"th page in this.

    bool uptodate;
    // Iff there are no unapplied modifications to this segment.
    // Accesser must hold mm->mutex.

    bool modified(long stamp) const;
    // effects	- Returns true iff the segment has any pending modifications
    //		  made on or before "stamp".
    //
    //		  In typical use, a stamp will be generated when
    //		  installing a modification into a segment.  Later on
    //		  "modified(stamp)" can be called to check whether or
    //		  not that modification has been written to the disk.

    int size() const;
    // effects	Return segment size (in bytes)

    // \subsection{Pinning Operations}

    OR_obj* pin(int index);
    // effects In addition to inherited semantics, fetches the segment.

    void unpin();
    // Requires caller hold mm->mutex.

    void pin_segment();
    // Requires caller hold mm->mutex.

    // \subsection{Object Modifications}
    
    bool install(int index, OR_obj* object, int num_slots);
    // modifies	this
    // requires	"index" is a valid object index into the segment.
    // effects	Attempts to store "object" value (with num_slots slots that
    //          does not include header slots) at the
    //          specified index.  Returns TRUE if successful, FALSE if the
    //		segment does not have enough room for the object.

    OR_obj* find(int i);
    // requires	Caller holds "mutex" and segment is cached.
    // effects	Returns the "i"th object within "contents".
    //		Returns 0 if no such object exists.

    //void segment_arrived(fe_message &msg, Network *net);
    // requires	Caller holds "mutex" and segment is cached.
    // handles segments sent in reply to iread avoidance messages
    
    //void coop_fetch();
    // requires	Caller holds "mutex".
    // Fetch contents of the segment: try the cache, then FEs, then disk.
    // May release mutex while waiting.


    void print(FILE *fp = NULL);
    // effects	Print segment rep to fp

// \subsection{Hooks for CacheEntry}
  protected:
    void read_contents(bool iread);
    void write_contents();

// \subsection{Representation}
  private:
// Access to a segment is controlled by the MM mutex and a pin count.
    Disk_Range		range;		// Occupied disk range
    Disk_Segment*	contents;	// Actual contents (see "dformat.h")
    long		write_stamp;	// All mods before this stamp are
					// known to be on disk.

    void set_contents(OR_slot *dseg);
    // requires	Caller holds "mutex" and segment is cached.
    // Sets the contents of this to dseg and signals waiting threads.
    
    void init_reservation_info();
    // requires	!missing()
    // effects	Initialize reservation info for this segment if necessary.

    // void fetch_forward(FE_index helper, FeSet *tried_fes, bool send_mods);
    // Requires caller holds mutex.
    // Modifies tried_fes.
    // Effects sends forward request to helper, adds it to tried fes.
    // Sends mods iff "send_mods".

};

/* See "segment.cc" for rep invariant and abstraction functions. */

inline Page* Segment::lookup_page(Uint pagenum) {
    th_assert(contents, "Segment is not cached");
    th_assert(pagenum < num_pages(), "Bad page number for lookup");
    return &(contents->pages[pagenum]);
}

#endif /* _SEGMENT_H */
