// Copyright 1995 Barbara Liskov

/*
\section{OR connection manager}
*/

/*
\subsection{Todo}

\begin{itemize}
\item Ensure that multiple OR managers aren't started for same OR
\item TCP-related issues (see fe_manager.cc)
\end{itemize}
*/

#include <iostream.h>

#include <signal.h>
#include <unistd.h>
#include <sys/uio.h>


#include "utils/compat.h"
#include "utils/fail.h"
#include "utils/network.h"

#include "common/or_num.h"
#include "common/or_set.h"
#include "common/or_message.h"

#include "or.h"
#include "or_config.h"
#include "fe_map.h"
#include "tm.h"
#include "tstatus.h"
#include "update.h"
#include "fe_manager.h"
#include "or_manager.h"

// Number of milliseconds to wait in the main loop of the or_manager
#define OR_OR_CONN_TIMEOUT 1000
Uint const Default_net_bufsize = 50*1024;


implementOpenHashMap(ORManagerTable, OR_num, OR_manager *, or_num_hash, or_num_equal)

// \subsection{OR manager set implementation}
OR_manager_set::OR_manager_set()
{
    table = new ORManagerTable;
    mutex = new Mutex;
}

OR_manager_set::~OR_manager_set()
{
    delete table;
    delete mutex;
}

bool OR_manager_set::send_message(OR_num or_num, or_or_message *msg)
{
    OR_manager *mgr;
    bool res;

    if (or_num == orx->config->ornum)
    {
	warn("Tried to send message to ourself");
	return FALSE;
    }
    
    // Look for existing OR manager
    mutex->grab(); {
	res = table->fetch(or_num, mgr);
    } mutex->release();
    if (res)
	return mgr->send_message(msg);

    // Set up network connection to new OR
    Ubits32 host_id;
    int port;
    bool success = orx->locator->lookup(OR_address(or_num), &host_id, &port);
    if (!success) return FALSE;

    Network *net = new Network(OR_address(or_num), host_id, port, success);
    if (!success) { delete net;  return FALSE;}

    net->set_nodelay();
    net->set_buffsizes(Default_net_bufsize, Default_net_bufsize);
    // Send my OR number
    if (!net->send_buffer(&orx->config->ornum, sizeof(orx->config->ornum)) 
	|| !net->flush()) {
	delete net;
	return 0;
    }

    // Start new OR manager
    mgr = new OR_manager(or_num, net);
    add(or_num, mgr);
    mgr->start();
    return mgr->send_message(msg);    
}

void OR_manager_set::add(OR_num or_num, OR_manager *mgr)
{
    mutex->grab(); {
	table->store(or_num, mgr);
    } mutex->release();
}

void OR_manager_set::remove(OR_num or_num)
{
    mutex->grab(); {
	table->remove(or_num);
    } mutex->release();
}

// \subsection{OR Manager Implementation}

OR_manager::OR_manager(OR_num id, Network* n) {
    or_num  = id;
    net	    = n;
    mutex_comm = new Mutex;
}

OR_manager::~OR_manager() {
    delete net;
    delete mutex_comm;
}

bool OR_manager::send_message(or_or_message const *msg)
{ 
    bool res;
    mutex_comm->grab(); {
    res = msg->encode(net);
    } mutex_comm->release();
    net->flush();
    return res;
}

void OR_manager::main() {
    or_or_message msg;
    int wait_status;

    // Disable pipe signals
    struct sigaction todo;
    todo.sa_handler = SIG_IGN;
    sigemptyset(&todo.sa_mask);
    todo.sa_flags = 0;
    sigaction(SIGPIPE, &todo, 0);

    while ((wait_status = wait_or_timeout(&msg, OR_OR_CONN_TIMEOUT)) >=0) {
	if (wait_status == 0) continue; // The network timed out.
	
	/* decode OR request */
	switch (msg.msgtype) {
	case OR_COMMIT:
	    if (orx->config->debug_level > 1)
		printf("Got commit message from OR %d\n", or_num);
	    commit_trans(&msg);
	    break;
	case OR_VOTE_ABORT:
	    if (orx->config->debug_level > 1)
		printf("Got abort vote from OR %d\n", or_num);
	    vote_abort(msg.tid);
	    break;
	case OR_VOTE_OK_CL:
	    if (orx->config->debug_level > 1)
		printf("Got prepare record from participant\n");
	    // Log participant's prepare record
	    orx->tm->log_participant(msg.tid, or_num, msg.u.vote_ok.pr, 
				    msg.u.vote_ok.index);
	    // fall through
	case OR_VOTE_OK:
	    if (orx->config->debug_level > 1)
		printf("Got ok vote from OR %d\n", or_num);
	    vote_ok(msg.tid, FALSE);
	    break;
	case OR_VOTE_READ:
	    if (orx->config->debug_level > 1)
		printf("Got read vote from OR %d\n", or_num);
	    vote_ok(msg.tid, TRUE);
	    break;
	case OR_ABORT:
	    if (orx->config->debug_level > 1)
		printf("Got abort message from OR %d\n", or_num);
	    abort_trans(msg.tid);
	    break;
	case OR_ACK:
	    if (orx->config->debug_level > 1)
		printf("Got ack message from OR %d\n", or_num);
	    ack(msg.tid);
	    break;
	    
	default:
	    /* XXX - Log error */
	    warn("OR sent invalid message type %d\n", msg.msgtype);
	    net->shutdown();
	    break;
	};
    }
    if (net->ok()) {
	// Finished because of connection shutdown
	// XXX - Log something?

    }
    // Remove ourselves from set of connected ORs
    orx->or_managers->remove(or_num);
}

int  OR_manager::wait_or_timeout(or_or_message *msg, int msec) {
    // modifies msg
    // effects  waits for a message from the OR or times out after msec
    //          milliseconds. Returns -1 there was an error on the
    //          network connection with this OR. Returns 0 if there was a
    //          timeout. Else decodes the received message from the network,
    //          puts it into msg and returns 1

    int res = net->wait_or_timeout(msec);
    if (res > 0) {
	if (!msg->decode(net)) res = -1;
    }
    return res;
}

// \subsubsection{Message handlers}

void OR_manager::vote_abort(Tid const& tid) {
    // If we haven't prepared this transaction yet, just remember abort.
    if (orx->tstatus->vote_abort(tid) == STATUS_OK)
	return;

    // Inform FE about abort -- ok to do this before logging abort record
    // because of presumed abort
    inform_fe(tid, OR_abort_other); // For the moment using this constant

    orx->tm->abort(tid);
}

void OR_manager::vote_ok(Tid const& tid, bool read_only) {
    int status = orx->tstatus->vote_ok(tid, or_num, read_only);

    // Only commit now if all votes have arrived
    if (status != STATUS_COMMIT)
	return;

    orx->tm->commit(tid);

    // Inform FE about commit. 
    inform_fe(tid, OR_abort_other);
}

void OR_manager::ack(Tid const& tid) {
    // If this is the last acknowledgment, log done record.
    if (orx->tstatus->vote_ack(tid, or_num)) {
	orx->tm->log_done(tid);
    }
}

void OR_manager::abort_trans(Tid const& tid) {
    orx->tm->abort(tid);
}

bool OR_manager::inform_fe(Tid const& tid, int result) 
{
    Address addr;
    bool retval = FALSE;

    if (orx->tstatus->get_fe(tid, addr)) {
	// See if FE is still connected
	orx->fe_map->read_lock();
	FE_manager *mgr = orx->fe_map->fetch(addr);

	if (mgr != 0) {
	    mgr->mutex_commit->grab(); {
		mgr->result = result;
		mgr->cond_commit->signal();
	    } mgr->mutex_commit->release(); 
	    retval = TRUE;
	}
	orx->fe_map->read_unlock();
    }
    return retval;
}

void OR_manager::commit_trans(or_or_message *msg) {
    OR_set empty_set;

    // Log committed record
    orx->tm->log_committed(msg->tid, &empty_set);

    // Start update phase thread
    start_update(orx->tstatus->get_log_index(msg->tid), msg);
 
    // Send ack to coordinator
    or_or_message ack_msg;
    ack_msg.msgtype = OR_ACK;
    ack_msg.tid = msg->tid;
    send_message(&ack_msg);

    orx->tm->truncate_vqueue_if_needed();
}
