#include <iostream.h>

#include "or.h"
#include "or_config.h"
#include "cachedir.h"
#include "fe_manager.h"
#include "or_send_data_msg.h"

#include "mm/mm.h"
#include "mm/segment.h"

extern Timer fetch_timer;

OR_send_data_msg::OR_send_data_msg(FE_manager *fe_manager_, Oref oref_, 
	   Seg_bmap prefetch_bmap_, bool alloc_request_, Ubits32 msg_id_):
    fe_manager(fe_manager_), oref(oref_), 
    prefetch_bmap(prefetch_bmap_), alloc_request(alloc_request_),
    msg_id(msg_id_) {}

bool OR_send_data_msg::encode(Network *net) {
    fetch_timer.start();
    Uint pageid = Oref_page(oref);
    Uint pagenum = Pagenum_from_pageid(pageid);
    Uint segid = Segid_from_pageid(pageid);

    Seg_bmap empty(FALSE);
    Seg_bmap sent(FALSE);
    Seg_bmap total_sent(FALSE);
    Seg_bmap allocated(FALSE);
    Seg_bmap remainder(FALSE); // The pages that will come in the next message
    // for eager prefetch

    if (pagenum < (Uint)prefetch_bmap.size()) prefetch_bmap.set(pagenum);

    Segment *seg = orx->mm->fetch_segment(segid, FALSE);
    iovec iov[Max_pages_in_seg]; // For the pages to be sent.
    th_assert(seg, "Segment not found while fetching object");

    int iov_count = 0;
    iovec iov_next[Max_pages_in_seg]; // For the pages to be sent in 2nd msg
    int count_next = 0;

    if (orx->config->eager_prefetch) {
	// If eager prefetching is on, send the number of pages specified by it
	// first and then the remaining pages.
	// Put "n" pages AFTER the requested page
	int num_pref = orx->config->eager_prefetch;
	for (Uint i = 0; i < Max_pages_in_seg; i++) {
	    Page *page = seg->lookup_page(i);
	    bool is_empty = page->empty();
	    if (prefetch_bmap.test(i) && !is_empty) {
		if (i >= pagenum && iov_count < num_pref) {
		    // Send this page in the first message
		    sent.set(i);
		    iov[iov_count].iov_base = (char *) page;
		    iov[iov_count].iov_len = Page_size;
		    iov_count++;
		} else {
		    remainder.set(i);
		    iov_next[count_next].iov_base = (char *) page;
		    iov_next[count_next].iov_len = Page_size;
		    count_next++;
		}
	    }
	    if (is_empty) empty.set(i);
	}
    } else {
	// No eager prefetching
	// Iterate over pages to be sent, in order of pagenum.
	// total_sent contains all the pages that were sent in this fetch
	for (Uint i = 0; i < Max_pages_in_seg; i++) {
	    Page *page = seg->lookup_page(i);
	    bool is_empty = page->empty();
	    if (prefetch_bmap.test(i) && !is_empty) {
		sent.set(i);
		iov[iov_count].iov_base = (char *) page;
		iov[iov_count].iov_len = Page_size;
		iov_count++;
	    }
	    if (is_empty) empty.set(i);
	}
	total_sent = sent;
    }

    // Record the pages sent in cachedir. Ask for allocation grant.
    allocated = orx->cachedir->enter(segid, total_sent, fe_manager,
				    Page_complete, alloc_request);

    fetch_timer.stop();
    // First message
    if (! (net->send_ubits32(orx->config->ornum, TRUE) &&
	   net->send_ubits32(segid, TRUE) &&
	   net->send_ubits32(0, TRUE) && // zero objects sent
	   sent.encode(net) &&
	   remainder.encode(net) &&
	   allocated.encode(net) &&
	   empty.encode(net) &&
	   net->send_vector(iov, iov_count, TRUE) &&
	   net->flush()))
	return FALSE;

    if (orx->config->eager_prefetch && !remainder.is_all_false()) {
	// Send the header for the next message also along with just the
	// pages
	if (! (net->send_ubits32 (msg_id, TRUE) &&
	       net->send_ubits32 (type(), TRUE) &&
	       net->send_vector(iov_next, count_next, TRUE) &&
	       net->flush()))
	    return FALSE;
    }

    orx->mm->mutex->grab(); {
	if (seg) seg->unpin();
    } orx->mm->mutex->release();
    return TRUE;
}
