// Copyright 1995 Barbara Liskov

/*
\section{OR Main Loop}

Loop waiting for connections from front-ends.  Each front-end connection
is allocated one thread to handle communication with that front-end.

\subsection{Todo}
\begin{itemize}
\item accept OR connections
      (need some way to distinguish them from FE connections)
\item recycle file descriptors if necessary
\item record "FE_manager" somewhere so that we can find the thread
      if necessary to handle a reconnection.
\end{itemize}
*/


#include <pthread.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <iostream.h>

#include "utils/other_unix.h"
#include "utils/compat.h"
#include "utils/fail.h"
#include "utils/network.h"
#include "common/locator.h"

#include "fe_map.h"
#include "fe_manager.h"
#include "or_manager.h"
#include "or_config.h"
#include "or.h"
#include "server.h"
#include "or_or_msg.h"
#include "mm/log.h"
#include "mm/mm.h"
#include "mm/mm_stats.h"
#include "config/vdefs/REPLICATION.h"
#include "gc/gc.h"
#include "gc/collector.h"

Uint const Default_net_bufsize = 50*1024; // For the network buffers


// Network error handler that prevents OR crashes.
static void print_warning(Device*, const char* msg) {
    warn(msg);
}

// Has a signal occurred?
// XXX Is there a race condition here?
static volatile int had_signal = 0;

//  XXX - used to only take (int) for __osf__
void signalled_shutdown(int) {
    had_signal = 1;
}

void do_gc(int) {
    if (gc)
	gc->collector->collect_partition();
}

void prepare_for_signals() {
    struct sigaction sa, sa_gc;

    // Shut down gracefully on SIGTERM
    sa.sa_handler = signalled_shutdown;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART;

    sigaction(SIGHUP,  &sa, 0);
    sigaction(SIGINT,  &sa, 0);
    sigaction(SIGTERM, &sa, 0);

    // Run garbage-collection (GC) on a SIGUSR1
    sa_gc.sa_handler = do_gc;
    sigemptyset(&sa_gc.sa_mask);
    sa_gc.sa_flags = SA_RESTART;

    sigaction(SIGUSR1, &sa_gc, 0);    
}

static Address next_fe_address () {
    static Ubits32 prev_stamp = 0; // The stamp for the previous FE stamp
    Tstamp stamp(TRUE);
    Ubits32 new_stamp = stamp.coarse_time();
    // Ensure that the new time is greater than the previuos stamp
    if (new_stamp <= prev_stamp)
	new_stamp = prev_stamp + 1;
    prev_stamp = new_stamp;
    Address result(orx->config->ornum, new_stamp);
    return result;
}


void serve_connections() {
    orx->fe_map = new FE_map; // Collection of per-FE info

    // Handle complications introduced by signals
    prepare_for_signals();

    // Create sockets
    int fe_sock, or_sock;
    if ((fe_sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	sysfail("socket");
    }
    if ((or_sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	sysfail("socket");
    }

    // Allow address reuse
    int opt = 1;
    if (setsockopt(fe_sock, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt)) < 0) {
	sysfail("setsockopt");
    }
    if (setsockopt(or_sock, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt)) < 0) {
	sysfail("setsockopt");
    }

    // Bind address
    struct sockaddr_in addr;
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = INADDR_ANY;
    addr.sin_port = htons(orx->config->fe_socket);
    if (bind(fe_sock, (struct sockaddr*) &addr, sizeof(addr)) < 0) {
	sysfail("bind");
    }
    
    addr.sin_port = htons(orx->config->or_socket);
    if (bind(or_sock, (struct sockaddr*) &addr, sizeof(addr)) < 0) {
	sysfail("bind");
    }

    // Allow connections
    if (listen(fe_sock, 5) < 0) {
	sysfail("listen");
    }
    if (listen(or_sock, 5) < 0) {
	sysfail("listen");
    }


    fprintf(stderr, "Server ready\n");
#if REPLICATION
    orx->log->set_allow_flushing();
#endif

    // Accept connections
    while (1) {
	fd_set readfds;
	struct sockaddr_in from;
#ifdef __linux__
	struct timeval tv;
#endif
	int from_size = sizeof(from);
	int fd, numfds;

	// Wait for FE or OR connection
	FD_ZERO(&readfds);
	FD_SET(fe_sock, &readfds);
	FD_SET(or_sock, &readfds);
	numfds = getdtablesize();
#ifdef __linux__
	tv.tv_sec = 0;
	tv.tv_usec = 500;
	if (select(numfds, &readfds, NULL, NULL, &tv) < 0)
#else
	if (select(numfds, &readfds, NULL, NULL, NULL) < 0)
#endif
	{
	    if ((errno == EINTR) && had_signal)
		break;
	    
	    syswarn("select");
//  #ifdef __linux__
//  	    pth_yield(NULL);
//  #endif
	    continue;	    
	}
//  #ifdef __linux__
//  	pth_yield(NULL);
//  #endif

	// Check for new FE connection 
	if (FD_ISSET(fe_sock, &readfds)) {
	    if ((fd = accept(fe_sock, (struct sockaddr*) &from, (socklen_t *)&from_size)) < 0) {
		if ((errno == EINTR) && had_signal)
		    break;
		
		syswarn("accept");
		continue;
	    }
	    
	    
	    // Receive the address from the FE. If the FE address is null,
	    // assign it a new one. In any case, send it back to the FE
	    Address fe_address = Address::Error_address;
	    Network* net = new Network(fe_address, fd);
	    net->set_handler(print_warning);
	    net->set_nodelay();
            net->set_buffsizes(Default_net_bufsize, Default_net_bufsize);

	    if (!fe_address.decode(net)) {
		syswarn("FE address was not received correctly");
		delete net; continue;
	    }
	    if (fe_address == Address::Error_address)
		fe_address = next_fe_address();
	    net->address = fe_address;
	    if (!fe_address.encode(net))
		syswarn("FE address was not sent correctly");
	    net->flush();
	    
	    // Establish a special channel with FE to handle iread requests
	    //	    bool new_conn;
	    // Network *ir_net = felocator->make_connection(id, new_conn);
	    // th_assert(ir_net, "Could not connect to FE in forward fetch");
	    // ir_net->set_handler(print_warning);
	    // ir_net->set_nodelay();
            // ir_net->set_buffsizes();

	    // Enter the FE into the FE_map.
	    orx->fe_map->write_lock();
	    orx->fe_map->store1(fe_address, NULL);
	    orx->fe_map->write_unlock();
	    FE_manager* mgr = new FE_manager(net, NULL);
	    
	    mgr->start();
	}

	// Check for new OR connection
	if (FD_ISSET(or_sock, &readfds)) {
	    if ((fd = accept(or_sock, (struct sockaddr*) &from, (socklen_t *)&from_size)) < 0) {
		if ((errno == EINTR) && had_signal)
		    break;
		
		syswarn("accept");
		continue;
	    }

	    Network* net = new Network(Address(), fd);
	    net->set_handler(print_warning);
	    net->set_nodelay();
            net->set_buffsizes(Default_net_bufsize, Default_net_bufsize);

	    // Read OR number
	    OR_num id;
	    if (! net->recv_buffer(&id, sizeof(id))) {
		// XXX - Log error
		fprintf(stderr, "could not read OR identifier\n");
		delete net;
		continue;
	    }

	    // Start new OR manager and add to manager set
	    fprintf(stderr, "Got new OR number %d\n", id);
	    net->address = OR_address(id);
	    OR_manager* mgr = new OR_manager(id, net);
	    orx->or_managers->add(id, mgr);
	    mgr->start();
	}
    }

    // Done processing connections because of a signal
    orx->log->shutdown();
    orx->mm->stats->print();
}
