#ifndef _SUPER_RW_LOCK_H
#define _SUPER_RW_LOCK_H

/* 
Super lock is useful in abstractions which use an RW_lock internally for
each method, but need to export the lock so that a sequence of methods may
be invoked atomically by the user. If none of these methods require a write
lock, an encompassing read lock can be used (as with most RW_locks).
Otherwise, the super lock should be used.

Super_RW_lock extend RW_locks to provide the following compatibility table:

   R  W  S
R  1  0  0
W  0  0  0
S  x  x  0

Here the lock in the left margin is grabbed before the one at the top;
1 means that the locks can be held simultaneously;
"x" is 1 if the same thread grabs the two locks and 0 otherwise.
This means that if a thread has a super lock, it can grab the read
or the write lock without deadlocking, but other threads cannot grab any
lock. Note that, as with RW_locks (except for RW_lock_mutex,  a thread can grab
a read lock multiple times, but not a write lock.

Besides providing the functionality of a super lock, the implementation
arranges for the read and write locks grabbed while holding the super lock
to be relatively fast. Therefore, super locks can be used to improve
performance even if atomicity of multiple methods is not an issue.
For example, if n entries are to be inserted in an atomic map, using a
super lock could reduce the underlying 2n mutex grab's and release's to 2.
The implementation also retains the use of non-recursive mutexes, 
which are faster than recursive mutexes.  

This is an original algorithm that has not been proven correct.
*/

#include "rwlock.h"
#include "pthread.h"

class Super_RW_lock: public RW_lock {
public:
    Super_RW_lock(RW_lock *lock);
    // Create a super lock using given RW_lock.
    // Any RW_lock that allows multiple read locks may be used.

    void super_lock();
    // Locks the super lock.

    void super_unlock();
    // Requires this thread called super_lock() with no subsequent unlock().

    // inherited
    void read_lock();
    void read_unlock();
    void write_lock();
    void write_unlock();

private:
    RW_lock *lock; // the underlying lock
    pthread_t super_locker; // the thread that holds the super lock

    unsigned int counter_a, counter_b;
    // Counters a and b are used to ensure that reading of super_locker 
    // happens atomically wrt modifications to it. Note that modifications
    // to pthread_t are not inherently atomic, since it is 128 bits.
    // The protocol is that counter a and b are equal except when 
    // super_locker is being modified. Counter a is incremented before
    // modifying super_locker and counter b is modified later.
    // Counter b is read before reading super_locker, and counter a is read
    // later. The read is accepted only if the counter a and b are equal.
    // Wrap arounds of counters are not expected to cause problems.

    bool have_super_lock();
    // Returns true if this thread holds a super lock.
};


inline bool Super_RW_lock::have_super_lock() {
    pthread_t locker;
    unsigned int b;
    do {
	b = counter_b;
	locker = super_locker;
    } while (counter_a != b);
    pthread_t self = pthread_self();
    return (locker.field1 == self.field1 && locker.field2 == self.field2);
}


#endif /* _SUPER_RW_LOCK_H */
